#include <msp430G2432.h>
#include "magnet.h"
#include "com.h"

//total_check_count will be reset to 0 after a swipe action
unsigned long total_check_count = 0; //total cycles of checking

unsigned int track2_baseline = 0;
unsigned char smclk_flag = 4;

//The reader will be locked after 60 seconds no activity
//@1MHZ MCLK
#define LOCK_TIME 250000

//We update the baseline of track 2 every 60 seconds
#define BASE_UPDATE_TIME 60

//@ 12K VLOCLK timerA clock
#define T_1000_MS 9600

//@ 1M SMCLK = 512Khz timerA clock
#define T_500_US 100//250
//10 bit adc result to represent the voltage of microphone port
// 200 -> about 0.6v
#define MIC_V_THRESHOLD 200

//buffers for store encoded data bits
unsigned int gui_MAG2_buf[MAG2_BIT_BUF_LEN] = { 0 };
//unsigned int gui_MAG3_buf[MAG3_BIT_BUF_LEN] = { 0 };

unsigned char buf1[MAG2_NIBBLE_BUF_LEN / 2];
//unsigned char buf2[MAG3_NIBBLE_BUF_LEN / 2];

extern unsigned char magnet_init(unsigned char flag);
extern int magnet_readTrack(unsigned int buf[], unsigned int buf_len,
		unsigned int mag3_buf[], unsigned int mag3_buf_len,
		unsigned long time_out);

extern int magnet_byteEncode(unsigned int input_buf[],
		unsigned char output_buf[], unsigned int buf_len);

extern void magnet_set_t2_baseline(unsigned int);
void power_off_opamp(void);
void power_on_opamp(void);

void gpioInit() {

//	P2DIR = 0xfe;
//
////P6.0,P6.1 ->AD0, AD1
//	P6SEL |= BIT0 + BIT1;
//	P6DIR &= ~(BIT0 + BIT1);

////P1.6 -> LED, Active low
//	P1DIR |= BIT6;
//	P1OUT |= BIT6;

////P5.0 -> KEY
//	P5DIR &= ~BIT0;
//
//	// Enter LPM0, enable interrupts
	__bis_SR_register( GIE);

}

void clear_buffer(void) {
	int i;
	//clear all buffers
	for (i = 0; i < (sizeof(buf1)); i++)
		buf1[i] = 0;

	//note: gui_magx_buf is int type, the lenth return by sizeof() is bytes
	for (i = 0; i < sizeof(gui_MAG2_buf) / 2; i++)
		gui_MAG2_buf[i] = 0;

}
//return:
//0:success;
//-1: time out;
//-2:track 2 OK but track 3 failed;
//-3:track 3 OK but track 3 failed;
//-4: all tracks failed
int read_card(void) {
	unsigned int i = 0;
	int len1, len2 = 0;
	int err = 0;

	//note:	buf lenth is by words
	err = magnet_readTrack(gui_MAG2_buf, sizeof(gui_MAG2_buf) / 2, 0,
//			sizeof(gui_MAG3_buf) / 2, 10000000);
			0, 0);

	//read time out
	if (-1 == err) {
		return -1;
	}

	len1 = magnet_byteEncodeRaw(gui_MAG2_buf, sizeof(gui_MAG2_buf) / 2, buf1,
			sizeof(buf1));

	if ((len1 <= T2_MIN_LEN)) {
		len1 = magnet_byteEncodeReverse(gui_MAG2_buf, sizeof(gui_MAG2_buf) / 2,
				buf1, sizeof(buf1));
	}
	com_init();

	if ((len1 > T2_MIN_LEN) && (len1 <= T2_MAX_LEN)) {
		com_sendAck2(ACK_CH_ALL, len1, buf1, 0, 0);
		return 0;
	}

	// track2 read error
	com_sendAck2(ACK_CH_ALL, 0, 0, 0, 0);
	return -2;

}

void set_mclk(unsigned char freq) {

	switch (freq) {

	case 1:
		DCOCTL = 0;
		BCSCTL1 = CALBC1_1MHZ + DIVA_1;  // Set DCO to 1MHz, ACLK/2 for WDT+
		DCOCTL = CALDCO_1MHZ;

		//MCLK=DCOCLK=1M
		//SMCLK=DCOCLK/8=128KHZ
		BCSCTL2 = DIVM_0 + DIVS_3;
		break;
	case 4:
		DCOCTL = 0;
		BCSCTL1 = CALBC1_8MHZ+ DIVA_1;  // Set DCO to 8MHz
		DCOCTL = CALDCO_8MHZ;

		//4.25 Mhz DCO
//		DCOCTL=0x60; //DCOX=3; MODX=0;
//		BCSCTL1=11|XT2OFF;//RSELX=11;

		//MCLK=DCOCLK/1=4.25M
		//SMCLK=DCOCLK/2=4.25MHZ
		BCSCTL2 = DIVM_1 + DIVS_1;
		break;
	case 16:
		DCOCTL = 0;
		BCSCTL1 = CALBC1_16MHZ+ DIVA_1;  // Set DCO to 16MHz
		DCOCTL = CALDCO_16MHZ;

		//MCLK=DCOCLK=16M
		//SMCLK=DCOCLK/4=4MHZ
		BCSCTL2 = DIVM_0 + DIVS_2;
		break;
	default:
		break;
	}
}

void stop_timer(void) {
	TA0R = 0;
	TA0CCTL0 = 0;

	//It is a bug for Timer A0? When wake up from timer interrupt,
	//SMCLK is switched to timer, but if the timer does not be started,
	//following s/w decoder will fails in first call.
	TA0CTL = TASSEL_2 + MC_1;
}
/**
 * @brief When sleep 1 sec, the timer use VLOCLK for it's clock
 * the VCLOCK feed to ACLK
 *
 * note: ACLK had already be divided by 2
 *  */
void start_timer_1s(void) {
	TA0CTL = 0;
	//ACLK, CLK/2,start timer with UP mode, enable interrupt, clear int flag
	TACCTL0 = CCIE; //enable compare interrupt
	TA0CCR0 = T_1000_MS;
	TA0CTL = TASSEL_1 + MC_1;

}

/**
 * @brief The clock for timer is 512khz, which is got by
 * SMCLK/1/2
 *
 */
void start_timer_500us(void) {

	TA0CTL = 0;
	//SMCLK, CLK/2,start timer with UP mode, enable interrupt, clear int flag
	TACCTL0 = CCIE; //enable compare interrupt
	TA0CCR0 = T_500_US;
	TA0CTL = TASSEL_2 + ID_1 + MC_1;

}

void sleep_1s(void) {

	power_off_opamp();
	start_timer_1s();

	__bis_SR_register(LPM3_bits | GIE);
	// Enter LPM0 w/ interrupt


	_no_operation();
	_no_operation();
	_no_operation();
	_no_operation();
	_no_operation();
	_no_operation();
	_no_operation();
	_no_operation();
	_no_operation();
	_no_operation();
	_no_operation();
	_no_operation();
	_no_operation();
}

void sleep_500us(void) {
	start_timer_500us();

	__bis_SR_register(LPM0_bits | GIE);
	// Enter LPM0 w/ interrupt


	stop_timer();

}

/**
 * @brief Configure and start the ADC conversion
 * @param channel: adc channel
 *                 0: track 2, 2: microphone
 * @return 10 bit ADC result
 */
unsigned int read_adc_channel(unsigned char channel) {
	unsigned int ret = 0;
	//==========  ADC configure ================//

	//ENC=0 to modify other bits
	ADC10CTL0 = 0;

	//10bit, VDD as ref, MODCLK @5M
	//70K~123 sps ADC @ (3.7M~6.3M MODCLOCK)/3
	ADC10CTL0 = ADC10SHT_0 + ADC10ON; // Sample time 4 ADCclks, ADC on

	// pulse mode, single channel, single converion,MODCLK(ADC10OSC)/3 -> 1.2M~2.1MHZ ADC CLOCK
	ADC10CTL1 = ADC10DIV_2;

	//==========  ADC channel selection ================//
	if (channel == 2) {
		ADC10CTL1 |= INCH_1;
		ADC10AE0 = 2;

	} else {
		ADC10CTL1 |= INCH_0;
		ADC10AE0 = 1;

	}

	//===========start conversion and wait for a result
	//start adc conversion without interrupt
	ADC10CTL0 &= ~ADC10IFG;
	ADC10CTL0 |= ENC + ADC10SC;

	while ((ADC10IFG & ADC10CTL0) == 0)
		;

	ret = ADC10MEM;
	//============stop adc module================
	//clear ENC to modify ADC10CTL0
	ADC10CTL0 &= ~ENC;
	//Disable ADC
	//stop ADC,  clear and disable ADC conv complete interrupt
	ADC10CTL0 &= ~(ADC10IE + ADC10SC); //
	ADC10CTL0 &= ~ADC10IFG; //must clear int flag in seperate step, or will fail
	//============================================
	ADC10CTL0 = 0;
	return ret;
}

void power_on_opamp(void) {
	//power the OPAMP
	//p1.6
	P1DIR |= 0x40;
	P1OUT |= 0x40;

	__delay_cycles(1000);

}
void power_off_opamp(void) {
	P1DIR |= 0x40;
	P1OUT &= ~0x40;

}
/**
 * @brief Get current attach state of the Microphone
 * @return 0: not attached
 *         1: attached
 */
unsigned char get_mic_state(void) {
	volatile unsigned int res = 0;
	res = read_adc_channel(2);

	if (res > MIC_V_THRESHOLD)
		return 1;
	else {
		_no_operation();
		return 0;
	}
}

/**
 * @breif Get track2 (OPAMP) output to detect swipe activity
 * track 2 will be checked every 500us, if 2 out of 3 times checking
 * are success, we believe a true swiping detected
 * To reduce power consumption, if too many times we check without valid
 * activity, the reader will rolls into locked state, under locked state,
 * the reader can not be operated unless a un-pluging and re-pluging cycle
 * performed.
 *
 * @return 1: A swiping activity detected
 *         0: No swiping activity
 *         0xff: time out
 */
unsigned char get_t2_state(void) {

	static unsigned char check_count = 0;
	static unsigned char valid_count = 0;

	unsigned int res, res1 = 0;
	total_check_count++;

	if (total_check_count >= LOCK_TIME) {
		total_check_count = 0;
		return 0xff;
	}
	res = read_adc_channel(0);
	res1 = read_adc_channel(0);

	if (((res <= (track2_baseline + 20)) && (res >= (track2_baseline - 20)))
			|| ((res1 <= (track2_baseline + 20))
					&& (res1 >= (track2_baseline - 20)))) {
		__no_operation();

		return 0;
	} else {
		__no_operation();
		return 1;
	}

#if 0
	total_check_count++;
	if (total_check_count >= LOCK_TIME) { //
		check_count = 0;
		valid_count = 0;
		total_check_count = 0;
		return 255;
	}
	check_count++;

	if ((res > (track2_baseline + 20)) || (res < (track2_baseline - 20))) {

		valid_count++;
		if (valid_count >= 2) {
			check_count = 0;
			valid_count = 0;
			total_check_count = 0;
			return 1;

		} else
		return 0;
	} else {
		if (check_count >= 3) {

			if (valid_count >= 2) {
				check_count = 0;
				valid_count = 0;
				total_check_count = 0;
				return 1;
			} else {
				check_count = 0;
				valid_count = 0;
				return 0;
			}
		} else {
			return 0;
		}

	}
#endif
}

// ACLK/2
// Watch dog use ACLK /VLOCLK
//about 6 second time out
void feed_dog(void) {

	//start watchdog, reset, clear watch timer, ACLK, divided by 32768
	WDTCTL = WDTPW + WDTCNTCL + WDTSSEL;

}

void stop_dog(void){

	WDTCTL = WDTPW + WDTHOLD;                 // Stop watchdog timer

}
unsigned int get_t2_baseline(void) {
	unsigned int adc, adc1 = 0;
	unsigned char count = 0;

	while (1) {
		adc = read_adc_channel(0);
		adc1 = read_adc_channel(0);
		count++;
		if ((adc >= (adc1 - 2)) && (adc <= (adc1 + 2))) {
			return adc;
		}
		if (count >= 100) {
			__no_operation();
			return track2_baseline;
		}
	}

}
void main(void) {

	unsigned char system_clock_flag = 16; //8: 8Mhz, 16:16Mhz system clock
	unsigned int ret = 0;
	unsigned char locked_flag = 0;
	unsigned char opa_power_flag = 0; //0: opa power is shut off
	unsigned char update_baseline_interval = 0; //every 60 seconds update the baseline voltage on track 2
	unsigned char plug_on_flag = 0;

	//Stop WDT
	WDTCTL = WDTPW + WDTHOLD;                 // Stop watchdog timer

	//enable VLOCLK for ACLK
	BCSCTL3 = LFXT1S_2;

	_delay_cycles(100);
	power_on_opamp();

	feed_dog();

	track2_baseline = get_t2_baseline();
	magnet_set_t2_baseline(track2_baseline);
	power_off_opamp();
	clear_buffer();
	set_mclk(1);

	__enable_interrupt();

	while (1) {
		feed_dog();

		if (locked_flag) { //locked state, can not be operated, must plug off and re-plug to exit locked state
			if (get_mic_state()) { //microphone still attached
				power_off_opamp();
				opa_power_flag = 0;

//				ADC10CTL1 = INCH_0;
//
//				ADC10AE0 = 3;
//				P1DIR = BIT2 + BIT3 + BIT4 + BIT5;
//				P1OUT = 0;
				sleep_1s();

			} else {
				locked_flag = 0;
			}

		} else { // standby state
			if (get_mic_state()) { //microphone still attached
//			if (1) { //microphone still attached

				if (opa_power_flag == 0) {
					opa_power_flag = 1;
					power_on_opamp();

				}
				ret = get_t2_state();
				switch (ret) {
				case 0:
					//	sleep_500us();
					break;
				case 1:
					set_mclk(4);
					_no_operation();
					smclk_flag = magnet_init(4);
					stop_dog(); //the decoder lib will modify clock configuration
					read_card();

					total_check_count = 0;
					clear_buffer();
					set_mclk(1);
					_delay_cycles(100000); //wait Microphone port stable
					track2_baseline = get_t2_baseline();
					magnet_set_t2_baseline(track2_baseline);
					break;
				case 0xff:
					locked_flag = 1;
					break;
				default:
					break;

				}

			} else { // update track 2 baseline, and continue sleep until next wake-up
				update_baseline_interval++;
				if (update_baseline_interval >= BASE_UPDATE_TIME) {
					power_on_opamp();
					update_baseline_interval = 0;
					track2_baseline = get_t2_baseline();
					magnet_set_t2_baseline(track2_baseline);
					power_off_opamp();

				}
				opa_power_flag = 0;
				total_check_count = 0;
				sleep_1s();

			}

		}

	} //while

// test: MCLK= DCOCLK/10
//	P4DIR |=0x4;
//
//	while(1){
//
//		P4OUT|=0x04;
//		P4OUT&=~0x04;
//	}

#if 0
	P3DIR
	|= 0x1;
	P3OUT &= ~0x01;

	while (1) {

//		if ((P5IN & 1) == 0) { //when S1 pressed, starting read

		P3OUT &= ~BIT0;
		ret = read_card();

		if (ret == 0) {

			//read card successfully
			__no_operation();

			P3OUT |= BIT0;
			__delay_cycles(2000000);
			P3OUT &= ~BIT0;
			//__delay_cycles(2000000);

		} else {
			__no_operation();

		}

		P3OUT |= BIT0;

//		}
	}

#endif

}

// Timer A0 interrupt service routine
#pragma vector=TIMER0_A0_VECTOR
__interrupt void Timer_A(void) {
	stop_timer();
	_bic_SR_register_on_exit(LPM3_bits);       // exit to active mode

}
