//*******************************************************************************
//  MSP-FET430P120 - Charge Li-Ion battery using MSP430F1232
//
//  Description: This application generates and varies PWM outputs on P1.2
//  using Timer_A to charge a Li-Ion battery
//  ADC10 monitors the battery voltage(A0), Rsense (A2: monitor's charge current)
//  and battery temperature(A1)
//  The watchdog timer controls the delay after constant voltage charge phase
//  to end charging
//  ACLK = n/a, SMCLK = MCLK = TACLK = default DCO ~800kHz.
//
//               MSP430F1232
//            -----------------
//        /|\|              XIN|-
//         | |                 |
//         --|RST          XOUT|-
//           |                 |
//           |         P1.2/TA1|--> CCR1 - Charge circuit
//
//  H. Grewal
//  Texas Instruments Inc.
//  Oct 2005
//  Built with IAR Embedded Workbench Version: 3.21A
//******************************************************************************

#include  <msp430x12x2.h>

void Set_DCO (void);
void configADC(void);

unsigned int AdcData[3] = {0};              // Store results A2,A1,A0
unsigned int const_volt = 0;                // Determine ConstVoltage/ConstCurrent
                                            // stage
unsigned int timedelay = 0;                 // Variable to Start Watchdog timer delay
unsigned int timercounts = 0;               // Track time delay
unsigned int delayflag = 0;

void main(void)
{
  WDTCTL = WDTPW + WDTHOLD;                 // Stop WDT
  P1DIR |= 0x0F;                            // P1.2 and P1.3 output
  P1OUT = 0;
  P1SEL |= 0x0C;                            // P1.2 and P1.3 TA1/2 otions
  Set_DCO();
  CCR0 = 256-1;                             // PWM Period
  CCTL1 = OUTMOD_7;                         // CCR1 reset/set
  CCR1 = 80;                                // CCR1 PWM duty cycle
  CCTL2 = OUTMOD_7;                         // CCR2 reset/set
  CCR2 = 128;                               // CCR2 PWM duty cycle
  TACTL = TASSEL_2 + MC_1;                  // SMCLK, up mode
  configADC();
  _BIS_SR(LPM0_bits+GIE);

  if(timedelay == 1)
  {
    delayflag = 1;
    WDTCTL = WDT_ADLY_1000;                   // WDT 1s, ACLK, interval timer
    IE1 |= WDTIE;                             // Enable WDT interrupt
    _BIS_SR(LPM0_bits);                       // Enter LPM3 w/interrupt
  }
}

//------------------------------------------------------------------------------
void Set_DCO (void)                         // Set DCO to selected frequency
//------------------------------------------------------------------------------
{
#define DELTA 938                           // Target DCO = DELTA*(4096) = 3.84 MHz

  unsigned int Compare, Oldcapture = 0;

  BCSCTL1 |= DIVA_3;                        // ACLK= LFXT1CLK/8
  CCTL2 = CM_1 + CCIS_1 + CAP;              // CAP, ACLK
  TACTL = TASSEL_2 + MC_2 + TACLR;          // SMCLK, cont-mode, clear

  while (1)
  {
    while (!(CCIFG & CCTL2));               // Wait until capture occured
    CCTL2 &= ~CCIFG;                        // Capture occured, clear flag
    Compare = CCR2;                         // Get current captured SMCLK
    Compare = Compare - Oldcapture;         // SMCLK difference
    Oldcapture = CCR2;                      // Save current captured SMCLK

    if (DELTA == Compare) break;            // If equal, leave "while(1)"
    else if (DELTA < Compare)               // DCO is too fast, slow it down
    {
      DCOCTL--;
      if (DCOCTL == 0xFF)
      {
        if (!(BCSCTL1 == (XT2OFF + DIVA_3)))
        BCSCTL1--;                          // Did DCO roll under?, Sel lower RSEL
      }
    }
    else
    {
      DCOCTL++;
      if (DCOCTL == 0x00)
        {
          if (!(BCSCTL1 == (XT2OFF + DIVA_3 + 0x07)))
          BCSCTL1++;                        // Did DCO roll over? Sel higher RSEL
        }
    }
  }
  CCTL2 = 0;                                // Stop CCR2
  TACTL = 0;                                // Stop Timer_A
}

void configADC(void)
{
  ADC10CTL0 &= ~ENC;                        // Set ENC to 0

  ADC10AE |= 0x07;                          // P2.2,1,0 analog input enable

  ADC10CTL0 = SREF_1 + ADC10SHT_3 + ADC10ON + ADC10IE + REFON + REFOUT;
                                            // VREF+/Vss ref,
                                            // S&H time is 64 ADC10CLKS
                                            // Turn on ADC10
                                            // Enable interrupt

  ADC10CTL1 = INCH_2 + SHS_3 + CONSEQ_3;    // A2 input chan,
                                            // TA2 trigger
                                            // Repeat seq of chans

  while( ADC10CTL1 & BUSY ){}               // Wait for BUSY bit to clear

  ADC10DTC0 |= ADC10CT;
  ADC10DTC1 = 3;                            // ADC DTC block size
  ADC10SA = (unsigned int)AdcData;          // ADC DTC block start address

  ADC10CTL0 |= ENC;                         // Set ENC to 1
}

//
// ADC10 Interrupt Service Routine
// Monitor Battery voltage and Rsense current and switches between
// Constant current and constant voltage charging stages
// Need to implement external thermistor battery temperature monitor
//
#pragma vector=ADC10_VECTOR
__interrupt void adc10_ISR(void)
{
  if (AdcData[0] < 355)                     // Batt_Temperature too high?
  {
    TACTL = 0;
    P1SEL = 0;
    P1OUT = 0x01;
    ADC10CTL0 = 0;
    timercounts = 0;
    timedelay = 0;
    delayflag = 0;
    WDTCTL = WDTPW + WDTHOLD;              // Stop WDT
  }
  if (const_volt == 0)
  {
    if (AdcData[2] < 100 | AdcData[2] > 1021)// Vout < 1V or >4.3V, Alarm
      P1OUT |= 0x02;
  }
  if ((AdcData[2] < 890) && const_volt == 0)// Vout < 3.6V, slow Charge
  {
   if (AdcData[1] > 300)                    // AdcData[1] measures Rsense voltage
    CCR1--;                                 // Adjust PWM
   else
   {
    CCR1++;
    CCR1 &= 0xFF;                           // 8-bit resolution
   }
  }
  else if((AdcData[2] > 1022)| const_volt == 1)// Vout >4.1V, Const Voltage
  {
     const_volt=1;
     CCR1--;
   }
   if (const_volt == 1)
   {
    if ((AdcData[1] < 1) && (delayflag == 0) ) // Current < 0.1C, Stop Charging after
    {                                       // 30 minute delay
     timedelay  = 1;
     LPM0_EXIT;
    }
    if (AdcData[2] < 1021)                  // 4.1 < Vout < 4.2
      CCR1++;
   }
   else if ((AdcData[2] >= 890) && const_volt == 0) // Vout > 3.6 V, Fast Charge
   {
    if (AdcData[1] > 500)
     CCR1--;
    else
    {
     CCR1++;
     CCR1 &= 0xFF;
    }
   }
  _NOP();
}

// Watchdog Timer interrupt service routine
// Enter every 8 secs to create 15 min delay before Stop
// charging
#pragma vector=WDT_VECTOR
__interrupt void watchdog_timer(void)
{
  P1OUT ^= 0x01;
  if(timercounts == 110)
  {
     TACTL = 0;                             // before stop charging
     P1SEL = 0;
     P1OUT = 0;
     ADC10CTL0 = 0;
     timercounts = 0;
     timedelay = 0;
     delayflag = 0;
     WDTCTL = WDTPW + WDTHOLD;              // Stop WDT
  }
  timercounts++;

}


