/*******************************************************************************
* Name: init.c 
* 
* Description: Initialization Functions for the microcontroller and peripherals
* on the board. These functions must be called first before the charging sequence
* or the SMBus communication with the battery is initiated.
* 
* Texas Instruments, Inc
* 
* Version: 1.0
*******************************************************************************/

/*
 * Copyright (C) {2011} Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

// Include Header files 
#include "init.h"
#include "led.h"
#include "pwm.h"
#include "device.h"
#include "F5XX_6XX_Core_Lib/hal_pmm.h"

/* ****************************************************************************
 * Function Name: UCS_Init
 * 
 * Description: Initialize the Universal Clock Source System. 
 * o Select internally generated REFO (~ 32 kHz) as the FLL reference clock 
 * o Select internally generated REFO (~ 32 kHz) as ACLK
 * o Program the DCO to approx 20 MHz (for SMCLK and MCLK)
 * ***************************************************************************/
void UCS_Init(void)
{
	// Set PMM VCore to Level 2 for 20 MHz DCO MCLK/SMCLK operation
	SetVCore(PMMCOREV_2);                   
	
	// Select internally generated REFO (32 kHz) as the ACLK source
	UCSCTL4 |= SELA__REFOCLK;	
	UCSCTL3 |= SELREF__REFOCLK;				// Select FLL Reference = REFO 
	
  	__bis_SR_register(SCG0);                  // Disable the FLL control loop
  UCSCTL0 = 0x0000;                         // Set lowest possible DCOx, MODx
  UCSCTL1 = DCORSEL_6;                      // Select DCO range of operation
  UCSCTL2 = FLLD_1 + 624;                   // Set DCO Multiplier for 20 MHz
							                // (N + 1) * FLLRef = Fdco
							                // (624 + 1) * 32768 = 20 MHz
							                // Set FLL Div = fDCOCLK/2
  __bic_SR_register(SCG0);                  // Enable the FLL control loop

  // Worst-case settling time for the DCO when the DCO range bits have been
  // changed is n x 32 x 32 x f_MCLK / f_FLL_reference. 
  // 32 x 32 x 20 MHz / 32,768 Hz = 640,000 MCLK cycles for DCO to settle
  __delay_cycles(640000);

  // Loop until XT1, XT2 & DCO fault flag is cleared
  do
  {
    UCSCTL7 &= ~(XT2OFFG + XT1LFOFFG + DCOFFG);
                                            // Clear XT2,XT1,DCO fault flags
	SFRIFG1 &= ~OFIFG;                      // Clear fault flags
  } while(SFRIFG1 & OFIFG);                 // Test oscillator fault flag
						
	
}

/* ****************************************************************************
 * Function Name: Timer_Init
 * 
 * Description: Initialize Timer A1 to establish a TIMER_TICK timebase. The
 * timeout duration for the SMBus clock low extension and the blinking period
 * of the LEDs will be based on this Timer A1. 
 * ***************************************************************************/
void Timer_Init(void)
{
	// Setup Timer A1 to count regular timer ticks and also to measure timeout
	// during SMBus communication
	TA1CCR0 = TIMER_TICK;							// Program Timer A1	CCR register
	TA1CCTL0 |= CCIE;								// Enable interrupts on timeout
	__enable_interrupt();							// Enable GIE bit	
	TA1CTL = TASSEL__ACLK + TACLR + MC__UP;			// Begin Timer A1 to measure timer tick			

}

/* ****************************************************************************
 * Function Name: PWM_Init
 * 
 * Description: Setup Timer A0 to output PWM on the voltage and current 
 * channels for the battery charger circuits. A 10-bit resolution implies 
 * 1023 steps. For an input voltage range of 58V, step resolution ~ 56 mV. 
 * PWM Frequency = Timer Source Clock/Number of PWM step-resolution
 * 				 = 20 MHz/1023 ~ 20 kHz
 * ***************************************************************************/
void PWM_Init(void)
{
	
	// Setup output timer pwm channels: V_PWM_1 = P1.2 = TA0.1, I_PWM_1 = P1.3 
	// = TA0.2, V_PWM_2 = P1.4 = TA0.3, I_PWM_2 = P1.5 = TA0.4
	P1SEL |= BIT5 + BIT4 + BIT3 + BIT2;			
	P1DIR |= BIT5 + BIT4 + BIT3 + BIT2;	
	
	// Setup period registers
	TA0CCR0 = PWM_DUTY_100;							// PWM Clock = SMCLK/PWM_DUTY_100
	TA0CCR1 = PWM_DUTY_0;							// 0% Duty Cycle
	TA0CCR2 = PWM_DUTY_0;							// 0% Duty Cycle
	TA0CCR3 = PWM_DUTY_0;							// 0% Duty Cycle
	TA0CCR4 = PWM_DUTY_0;							// 0% Duty Cycle
	
	// Setup output modes to initialize PWM outputs in LOW state
	
	TA0CCTL1 &= ~(OUT);
	TA0CCTL2 &= ~(OUT);
	TA0CCTL3 &= ~(OUT);
	TA0CCTL4 &= ~(OUT);
				
	TA0CCTL1 = OUTMOD_0;
	TA0CCTL2 = OUTMOD_0;
	TA0CCTL3 = OUTMOD_0;
	TA0CCTL4 = OUTMOD_0;
	
	// TA0 CLK = SMCLK ~ 4MHz, Up Mode, Initialize TA0 to Zero 
	TA0CTL = TASSEL__SMCLK + TACLR + MC__UP;	
	
}		


/* ****************************************************************************
 * Function Name: ADC_Init
 * 
 * Description: Set up the ADC for single-sample conversions. 
 * ***************************************************************************/
void ADC_Init(void)
{

	ADC10CTL0 = ADC10SHT_8  +			// Sampling Time: 256 x ADC10CLKs
			    ADC10ON;				// Turn on the ADC10
			     
    ADC10IE = ADC10IE0;	 				// Enable ADC10 Interrupts
			     
	ADC10CTL1 = ADC10SHS_0  +			// S/H source = ADC10SC
				ADC10SHP +				// Sampling Timer generates S/H signal
				ADC10DIV_0  +			// ADC10CLK divider = 1
				ADC10SSEL_0 +			// ADC Clock = ADC10OSC			     
				ADC10CONSEQ_0;			// Single-channel, single-conversion
}
