/* --COPYRIGHT--,BSD
 * Copyright (c) 2015, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
/*******************************************************************************
 *
 * JTAG_lock_FR5xx_with_password.c
 *
 * Demonstrates enabling JTAG Lock With Password protection on the
 * MSP430FR5xx/6xx family devices. The JTAG signatures are set to lock with
 * password and a password length of 2 words. The password is also set to
 * 12h 34h 56h 78h, or 0x34127856. The password protection only takes effect
 * after a BOR reset.
 *
 * NOTE: The first time loading a new project after having this code present
 * on the device with the JTAG Lock With Password enabled, in the new project
 * .ccxml file, under Advanced Setup > Target Configuration, provide the correct
 * password "0x34127856", or else a Security Fuse Blown message will appear in
 * CCS and you will be unable to load the part.
 *
 * K. Pier
 * Texas Instruments Inc.
 * November 2015
 * Built with CCS v6.1.1 and IAR v6.40.1
 *
 ******************************************************************************/
#include <msp430.h> 
#include <stdint.h>

/*
 * Lock the JTAG with password.
 *
 * Setting 0xAAAA for JTAG signature 1 on FR5xx/6xx devices locks the device
 * with a password. JTAG signature 2 is the password length in words.
 * The only way back into device is to provide the password via the toolchain.
 */
#pragma RETAIN(JTAG_signatures)
#pragma DATA_SECTION(JTAG_signatures, ".jtagsignature")
const uint16_t JTAG_signatures[] = {0xAAAA, 0x0002};

/*
 * Set the password.
 *
 * To access the device again after JTAG password is enabled, provide the
 * password in the MSP430FR5969.ccxml file under Advanced Setup > Target
 * Configuration.
 * Select MSP430.
 * Under Cpu Properties, in the box for Password: (Hex format) write the
 * password in words, with the least significant word first. In this example,
 * provide the password "0x34127856"
 *
 * To disable JTAG password again, load any project that does not have a JTAG
 * password defined - remember to provide the correct password to be able to
 * load this new project the first time.
 */
#pragma RETAIN(JTAG_password)
#pragma DATA_SECTION(JTAG_password, ".jtagpassword")
const uint8_t JTAG_password[] = {0x12, 0x34, 0x56, 0x78};

/*
 * main.c
 */
void main(void) {
    WDTCTL = WDTPW | WDTHOLD;	// Stop watchdog timer
	
    P1DIR |= BIT0;
    PM5CTL0 &= ~LOCKLPM5;

    while(1)
    {
    	P1OUT ^= BIT0;
    	__delay_cycles(50000);
    }
}
