#include <bq2562x_regmap.h>
#include <i2c_comm.h>
#include "bq2562x_charger.h"

//********************************************Low level Setting***************************************//
static bool bq2562x_get_charge_enable(void)
{
	uint8_t chrg_ctrl_0;

	// I2C Read
	I2C_Read(BQ2562X_CHRG_CTRL_0,&chrg_ctrl_0, 1);

	if (chrg_ctrl_0 & BQ2562X_CHRG_EN)
	{
			return true;
	}
	return false;
}

static void bq2562x_set_charge_enable(bool val)
{
    uint8_t chrg_ctrl_0;
    uint8_t chrg_ctrl_0_enable;

    // I2C Read
    I2C_Read(BQ2562X_CHRG_CTRL_0,&chrg_ctrl_0, 1);

    if(val == true)
    {
        chrg_ctrl_0_enable = chrg_ctrl_0 | BQ2562X_CHRG_EN;
    }
    else
    {
        chrg_ctrl_0_enable = chrg_ctrl_0 & (~BQ2562X_CHRG_EN);
    }

    I2C_Write(BQ2562X_CHRG_CTRL_0, &chrg_ctrl_0_enable, 1);
}

static int32_t bq2562x_get_vbat_adc(void)
{
    uint8_t i2c_data[2];
    int32_t vbat_adc;

	 I2C_Read(BQ2562X_ADC_VBAT_LSB,i2c_data, 2);

	vbat_adc = ((i2c_data[1] << 8) | i2c_data[0]) >> BQ2562X_ADC_VBAT_MOVE_STEP;

	return vbat_adc * BQ2562X_ADC_VBAT_STEP_uV;
}

static int32_t bq2562x_get_vbus_adc(void)
{
    uint8_t i2c_data[2];
	int32_t vbus_adc;

	I2C_Read(BQ2562X_ADC_VBUS_LSB,i2c_data, 2);

	vbus_adc = ((i2c_data[1] << 8) | i2c_data[0]) >> BQ2562X_ADC_VBUS_MOVE_STEP;
	return vbus_adc * BQ2562X_ADC_VBUS_STEP_uV;
}

static int32_t bq2562x_get_ibat_adc(void)
{
	int32_t ibat_adc;
    uint8_t i2c_data[2];

	I2C_Read(BQ2562X_ADC_IBAT_LSB,i2c_data, 2);
    ibat_adc = ((i2c_data[1] << 8) | i2c_data[0]) >> BQ2562X_ADC_IBAT_MOVE_STEP;

	return ibat_adc * BQ2562X_ADC_IBAT_STEP_uA;
}

static int32_t bq2562x_get_ibus_adc(void)
{
	int32_t ibus_adc;
    uint8_t i2c_data[2];

    I2C_Read(BQ2562X_ADC_IBUS_LSB,i2c_data, 2);
	ibus_adc =  ((i2c_data[1] << 8) | i2c_data[0]) >> BQ2562X_ADC_IBUS_MOVE_STEP;

	return ibus_adc * BQ2562X_ADC_CURR_STEP_uA;
}

static int32_t bq2562x_get_term_curr(void)
{
    int32_t iterm;
    uint8_t i2c_data[2];

    I2C_Read(BQ2562X_TERM_CTRL_LSB,i2c_data, 2);
	iterm = ((i2c_data[1] << 8) | i2c_data[0]) >> BQ2562X_ITERM_MOVE_STEP;

	return iterm * BQ2562X_TERMCHRG_CURRENT_STEP_uA;
}

static int32_t bq2562x_get_prechrg_curr(void)
{
	int32_t prechrg_curr;
    uint8_t i2c_data[2];

    I2C_Read(BQ2562X_PRECHRG_CTRL_LSB,i2c_data, 2);
	prechrg_curr = ((i2c_data[1] << 8) | i2c_data[0]) >> BQ2562X_PRECHRG_MOVE_STEP;

	return prechrg_curr * BQ2562X_PRECHRG_CURRENT_STEP_uA;
}

static int32_t bq2562x_get_ichg_curr(void)
{
    int32_t ichg;
    uint8_t i2c_data[2];

    I2C_Read(BQ2562X_CHRG_I_LIM_LSB,i2c_data, 2);
	ichg = ((i2c_data[1] << 8) | i2c_data[0]) >> BQ2562X_ICHG_MOVE_STEP;

	return ichg * BQ2562X_ICHRG_CURRENT_STEP_uA;
}

static int32_t bq2562x_get_chrg_volt(void)
{
    int32_t chrg_volt;
    uint8_t i2c_data[2];

    I2C_Read(BQ2562X_CHRG_V_LIM_LSB,i2c_data, 2);
    chrg_volt = ((i2c_data[1] << 8) | i2c_data[0]) >> BQ2562X_CHRG_V_LIM_MOVE_STEP;

    return chrg_volt * BQ2562X_VREG_V_STEP_uV;
}

static int32_t bq2562x_get_input_volt_lim(void)
{
    int32_t vlim;
    uint8_t i2c_data[2];

    I2C_Read(BQ2562X_INPUT_V_LIM_LSB,i2c_data, 2);
    vlim = ((i2c_data[1] << 8) | i2c_data[0]) >> BQ2562X_INPUT_V_LIM_MOVE_STEP;

    return vlim * BQ2562X_VINDPM_STEP_uV;
}

static int32_t bq2562x_get_input_curr_lim(void)
{
    int32_t ilim;
    uint8_t i2c_data[2];

    I2C_Read(BQ2562X_INPUT_I_LIM_LSB,i2c_data, 2);
    ilim = ((i2c_data[1] << 8) | i2c_data[0]) >> BQ2562X_INPUT_I_LIM__MOVE_STEP;

    return ilim * BQ2562X_IINDPM_STEP_uA;
}

static bc1p2_mode bq2562x_get_online_status(void)
{
    uint8_t chrg_stat_1;
    int32_t online_status;

    I2C_UpdateBits(BQ2562X_CHRG_STAT_1, BQ2562X_UNKNOWN_500MA);
    delay_cycles(1000);
    I2C_Read(BQ2562X_CHRG_STAT_1,&chrg_stat_1, 1);

    online_status = chrg_stat_1 & BQ2562X_VBUS_STAT_MSK;

    return (bc1p2_mode)online_status;
}

static void bq2562x_set_term_curr(int32_t term_current)
{
    int32_t reg_val;
    uint8_t i2c_data[2];

	term_current = clamp(term_current, BQ2562X_TERMCHRG_I_MIN_uA,
					BQ2562X_TERMCHRG_I_MAX_uA);

	reg_val = (term_current / BQ2562X_TERMCHRG_CURRENT_STEP_uA) << BQ2562X_ITERM_MOVE_STEP;

	i2c_data[1] = (reg_val >> 8) & 0xff;
	i2c_data[0] = reg_val & 0xff;
	I2C_Write(BQ2562X_TERM_CTRL_LSB, i2c_data, 2);
}

static void bq2562x_set_prechrg_curr(int32_t pre_current)
{
	int32_t reg_val;
    uint8_t i2c_data[2];

	pre_current = clamp(pre_current, BQ2562X_PRECHRG_I_MIN_uA,
					BQ2562X_PRECHRG_I_MAX_uA);

	bq2562x_set_charge_enable(false);

	reg_val = (pre_current / BQ2562X_PRECHRG_CURRENT_STEP_uA) << BQ2562X_PRECHRG_MOVE_STEP;

    i2c_data[1] = (reg_val >> 8) & 0xff;
    i2c_data[0] = reg_val & 0xff;
    I2C_Write(BQ2562X_PRECHRG_CTRL_LSB, i2c_data, 2);
	bq2562x_set_charge_enable(true);
}


static void bq2562x_set_ichrg_curr(int32_t chrg_curr)
{
	int32_t reg_val;
    uint8_t i2c_data[2];

	chrg_curr = clamp(chrg_curr, BQ2562X_ICHRG_I_MIN_uA,BQ2562X_ICHRG_I_MAX_uA);

    bq2562x_set_charge_enable(false);

    reg_val = ((chrg_curr / BQ2562X_ICHRG_CURRENT_STEP_uA) << BQ2562X_ICHG_MOVE_STEP);
    i2c_data[1] = (reg_val >> 8) & 0xff;
    i2c_data[0] = reg_val & 0xff;
    I2C_Write(BQ2562X_CHRG_I_LIM_LSB, i2c_data, 2);

    bq2562x_set_charge_enable(true);
}

static void bq2562x_set_chrg_volt(int32_t chrg_volt)
{
	int32_t reg_val;
    uint8_t i2c_data[2];

	chrg_volt = clamp(chrg_volt, BQ2562X_VREG_V_MIN_uV, BQ2562X_VREG_V_MAX_uV);

	reg_val = (chrg_volt / BQ2562X_VREG_V_STEP_uV) << BQ2562X_CHRG_V_LIM_MOVE_STEP;
	i2c_data[1] = (reg_val >> 8) & 0xff;
    i2c_data[0] = reg_val & 0xff;
    I2C_Write(BQ2562X_CHRG_V_LIM_LSB, i2c_data, 2);
}


static void bq2562x_set_input_volt_lim(int32_t vindpm)
{
    int32_t reg_val;
    uint8_t i2c_data[2];

	vindpm = clamp(vindpm, BQ2562X_VINDPM_V_MIN_uV,
						BQ2562X_VINDPM_V_MAX_uV);

	reg_val = (vindpm / BQ2562X_VINDPM_STEP_uV) << BQ2562X_INPUT_V_LIM_MOVE_STEP;
    i2c_data[1] = (reg_val >> 8) & 0xff;
    i2c_data[0] = reg_val & 0xff;
    I2C_Write(BQ2562X_INPUT_V_LIM_LSB, i2c_data, 2);
}



static void bq2562x_set_input_curr_lim(int32_t iindpm)
{
	int32_t reg_val;
    uint8_t i2c_data[2];

	iindpm = clamp(iindpm, BQ2562X_IINDPM_I_MIN_uA,
						BQ2562X_IINDPM_I_MAX_uA);

	reg_val = (iindpm / BQ2562X_IINDPM_STEP_uA) << BQ2562X_INPUT_I_LIM__MOVE_STEP;
    i2c_data[1] = (reg_val >> 8) & 0xff;
    i2c_data[0] = reg_val & 0xff;
    I2C_Write(BQ2562X_INPUT_I_LIM_LSB, i2c_data, 2);
}

//********************************************High level Setting***************************************//
void bq2562x_get_state(bq2562x_device *bq)
{
	uint8_t chrg_stat_0, chrg_stat_1;
	uint8_t fault_0, fault_1;

	I2C_UpdateBits(BQ2562X_ADC_CTRL, BQ2562X_ADC_EN);
	I2C_Read(BQ2562X_CHRG_STAT_1,&chrg_stat_1, 1);

	if (bq2562x_get_charge_enable())
	    bq->state.chrg_status = (chg_status)((chrg_stat_1 & BQ2562X_CHG_STAT_MSK)>>3);
	else
	    bq->state.chrg_status = (chg_status)BQ2562X_NOT_CHRGING;

	bq->state.chrg_type = (bc1p2_mode)(chrg_stat_1 & BQ2562X_VBUS_STAT_MSK);

	I2C_Read(BQ2562X_FAULT_STAT_0,&fault_0, 1);
	bq->state.health = fault_0 & BQ2562X_TEMP_MASK;
	bq->state.fault_0 = fault_0;

    bq->state.vbat_adc_uV = bq2562x_get_vbat_adc();
    bq->state.vbus_adc_uV = bq2562x_get_vbus_adc();
    bq->state.ibat_adc_uA = bq2562x_get_ibat_adc();
    bq->state.ibus_adc_uA  = bq2562x_get_ibus_adc();

    //Reset watchdog
    I2C_UpdateBits(BQ2562X_CHRG_CTRL_0, BQ2562X_WD_RST);
}


void bq2562x_hw_init(bq2562x_device *bq)
{
	uint8_t wd_reg_val = BQ2562X_WATCHDOG_DIS;
	uint8_t wd_max_val = BQ2562X_NUM_WD_VAL - 1;
	uint8_t i;
	int32_t test;

	if (bq->init_data->watchdog_timer_s) {
		if (bq->init_data->watchdog_timer_s >= bq2562x_watchdog_time[wd_max_val])
			wd_reg_val = wd_max_val;
		else {
			for (i = 0; i < wd_max_val; i++) {
				if (bq->init_data->watchdog_timer_s > bq2562x_watchdog_time[i] &&
				        bq->init_data->watchdog_timer_s < bq2562x_watchdog_time[i + 1]) {
					wd_reg_val = i;
					break;
				}
			}
		}
	}
	I2C_UpdateBits(BQ2562X_CHRG_CTRL_0,wd_reg_val);

	bq2562x_set_ichrg_curr(bq->bat_info->constant_charge_current_max_ua);
	test = bq2562x_get_ichg_curr();
	bq2562x_set_prechrg_curr(bq->bat_info->precharge_current_ua);
	test = bq2562x_get_prechrg_curr();
	bq2562x_set_chrg_volt(bq->bat_info->constant_charge_voltage_max_uv);
    test = bq2562x_get_chrg_volt();
	bq2562x_set_term_curr(bq->bat_info->charge_term_current_ua);
	test = bq2562x_get_term_curr();
	bq2562x_set_input_volt_lim(bq->init_data->vInputLowlim_uV);
    test = bq2562x_get_input_volt_lim();
	bq2562x_set_input_curr_lim(bq->init_data->iInputHighLim_uA);
	test = bq2562x_get_input_curr_lim();
}
