//*****************************************************************************
//
// shared_key_functions.c - Contains functions that are called to handle the
//                          various encryption-based boot loading processes
//                          which are part of the TivaWare boot loader.
//
// Copyright 2021 Texas Instruments Incorporated.
// Software License Agreement
// 
//   Redistribution and use in source and binary forms, with or without
//   modification, are permitted provided that the following conditions
//   are met: 
//
//   1. Redistributions of source code must retain the above copyright
//   notice, this list of conditions and the following disclaimer.
//   
//   2. Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in the
//   documentation and/or other materials provided with the distribution.
// 
//   3. Neither the name of Texas Instruments Incorporated nor the names
//   of its contributors may be used to endorse or promote products
//   derived from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//*****************************************************************************

#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include "inc/hw_flash.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "driverlib/aes.h"
#include "driverlib/eeprom.h"
#include "driverlib/flash.h"
#include "driverlib/gpio.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "driverlib/sysctl.h"
#include "linker_defines.h"
#include "shared_functions.h"

//*****************************************************************************
//
// Reserves a memory section to store the reboot key in RAM.
//
//*****************************************************************************
#pragma DATA_SECTION(vui64RebootKey,".RebootKey")
static volatile uint64_t vui64RebootKey;

//*****************************************************************************
//
// Define a randomly generated reboot key that is used to verify that the boot
// loader has been called by the application.
//
//*****************************************************************************
#define REBOOT_KEY 0x4200616C65676E41

//*****************************************************************************
//
// Define the GPIO peripheral, port, and pin for the external I/O check within
// the Shared Key Boot Loader.
//
//*****************************************************************************
#define GPIO_UPDATE_PERIPH              SYSCTL_PERIPH_GPIOJ
#define GPIO_UPDATE_BASE                GPIO_PORTJ_BASE
#define GPIO_UPDATE_PIN                 GPIO_PIN_1

//*****************************************************************************
//
// Global boolean variable to track whether or not valid keys are loaded.
//
//*****************************************************************************
bool bValidKeys;

//*****************************************************************************
//
// Global buffer to store AES encryption keys in RAM memory.
//
//*****************************************************************************
uint32_t g_pui32Keys[32];

//*****************************************************************************
//
// This function checks if the GPIO that is defined for receiving an external
// signal to trigger the boot loader has been pulled low. To change the check
// to look for a high signal, replace the '0' with 'GPIO_UPDATE_PIN'.
//
// \return True or False.
//
//*****************************************************************************
static bool CheckGPIOPin(void)
{
    //
    // Enable the GPIO peripheral.
    //
    MAP_SysCtlPeripheralEnable(GPIO_UPDATE_PERIPH);

    //
    // Configure the GPIO port and pin as an input with the internal pull-up
    // resistor enabled.
    //
    MAP_GPIODirModeSet(GPIO_UPDATE_BASE, GPIO_UPDATE_PIN, GPIO_DIR_MODE_IN);
    GPIOPadConfigSet(GPIO_UPDATE_BASE, GPIO_UPDATE_PIN,
                         GPIO_STRENGTH_2MA, GPIO_PIN_TYPE_STD_WPU);

    //
    // Return true if the GPIO has been pulled to ground.
    //
    if(MAP_GPIOPinRead(GPIO_UPDATE_BASE, GPIO_UPDATE_PIN) == 0)
    {
        return true;
    }
    else
    {
        return false;
    }
}

//*****************************************************************************
//
// This function checks for if there valid AES keys loaded into the final two
// blocks of the EEPROM.
//
// A valid key is defined as a key that is not 0xFFFF.FFFF.
//
// \return None.
//
//*****************************************************************************
void CheckValidKeys(void)
{
    uint32_t ui32LoopCount, ui32EEPROMInit;

    //
    // Enable the EEPROM module.
    //
    MAP_SysCtlPeripheralEnable(SYSCTL_PERIPH_EEPROM0);

    //
    // Wait for the EEPROM module to be ready.
    //
    while(!MAP_SysCtlPeripheralReady(SYSCTL_PERIPH_EEPROM0))
    {
    }

    //
    // This function must be called after SysCtlPeripheralEnable and before the
    // EEPROM is accessed in order to check for errors in the EEPROM state.
    //
    ui32EEPROMInit = MAP_EEPROMInit();

    //
    // Check if the EEPROM Initialization returned an error.  If so, mark that
    // the keys are not valid and return immediately.
    //
    if(ui32EEPROMInit != EEPROM_INIT_OK)
    {
        bValidKeys = false;
        return;
    }

    //
    // Read the keys from the last two blocks of EEPROM which is reserved
    // for the storage of four 32-byte keys.
    //
    MAP_EEPROMRead(g_pui32Keys, 6016, 128);

    //
    // Search the EEPROM data for any values which are not 0xFFFF.FFFF.
    //
    for(ui32LoopCount = 0; ui32LoopCount < 32; ui32LoopCount++)
    {
        if(g_pui32Keys[ui32LoopCount] != 0xFFFFFFFF)
        {
            //
            // If a valid key is found, break out of the loop.
            //
            break;
        }
    }

    //
    // If the data is all 0xFFFF.FFFF, then there are no valid keys present.
    //
    if(ui32LoopCount == 32)
    {
        bValidKeys = false;
    }
    //
    // If the first 16 fields were all 0xFFFF.FFFF, then the first key pair
    // is invalidated.
    //
    else if(ui32LoopCount >= 16)
    {
        //
        // Use memcpy to overwrite the first key pair with the second key pair.
        //
        memcpy(g_pui32Keys, g_pui32Keys + 16, 64);
        bValidKeys = true;
    }
    else
    {
        bValidKeys = true;
    }
}

//*****************************************************************************
//
// This function is called only when there a no keys in EEPROM.  It checks for
// valid key data in the first 128 bytes of flash at APP_BASE (not all 0xFF)
// and that the rest of flash is blank (is 0xFF).  If found, it will program
// the keys to EEPROM, then erase the flash at APP_BASE.
//
// \return None.
//
//*****************************************************************************
static void ValidateKeysInFlash(void)
{
    uint32_t *pui32Address;

    //
    // Checks for non blank-data in first 32 words of Flash.
    //
    for(pui32Address = (uint32_t *)APP_BASE;
                 pui32Address < (uint32_t *)(APP_BASE + 0x80); pui32Address++)
    {
        if(*pui32Address != 0xFFFFFFFFu)
        {
            bValidKeys = true;
        }
    }

    //
    // Checks if the rest of flash is blank.
    //
    if(bValidKeys == true)
    {
        //
        // Loop through all Flash addresses.
        //
        for(pui32Address = (uint32_t *)(APP_BASE + 0x80);
                 pui32Address < (uint32_t *)APP_END; pui32Address++)
        {
            //
            // If the flash is not blank, then clear valid keys flag.
            //
            if(*pui32Address != 0xFFFFFFFFu)
            {
                bValidKeys = false;
            }
        }
    }

    //
    // If the first two checks pass, then a valid key image has been provided.
    //
    if(bValidKeys == true)
    {
        //
        // Copy keys to EEPROM.
        //
        MAP_EEPROMProgram((uint32_t *)APP_BASE, 6016, 128);

        //
        // Copy the keys to RAM.
        //
        memcpy(g_pui32Keys, (uint32_t *)APP_BASE, 128);

        //
        // Erase the keys from Flash memory.
        //
        MAP_FlashErase(APP_BASE);

        //
        // Now hide the last two EEPROM blocks.
        //
        MAP_EEPROMBlockHide(94);
        MAP_EEPROMBlockHide(95);

#ifdef RELEASE
        //
        // Prevent bootloader from being modified.
        //
        MAP_FlashProtectSet(0,FlashReadOnly);
        MAP_FlashProtectSave();
        HWREG(FLASH_FMA) = 0x75100000;
        HWREG(FLASH_FMD) = HWREG(FLASH_BOOTCFG) & 0x7FFFFFFC;
        HWREG(FLASH_FMC) = FLASH_FMC_WRKEY | FLASH_FMC_COMT;
#endif
    }
}

//*****************************************************************************
//
// This function checks if the device is coming from a software reset caused by
// MyReinitFunc() by checking for the reboot key in RAM as well as the reset
// cause before deciding whether to remain in the boot loader or return to the
// application.
//
// \return True or False.
//
//*****************************************************************************
static bool CalledByApp(void)
{
    //
    // Compare the current RAM stored key with the defined key and verify that
    // the reset was caused by a software reset.
    //
    if((vui64RebootKey == REBOOT_KEY) &&
            ((MAP_SysCtlResetCauseGet() & SYSCTL_CAUSE_SW) == SYSCTL_CAUSE_SW))
    {
        //
        // Reset the RAM stored reboot key.
        //
        vui64RebootKey = 0;

        //
        // Clear the reset cause register.
        //
        MAP_SysCtlResetCauseClear(SYSCTL_CAUSE_SW);
        return true;
    }
    else
    {
        return false;
    }
}

//*****************************************************************************
//
// This function checks if the AES hash loaded at the end of Flash memory
// matches the AES hash calculated of the entire application image.
//
// \return ulReturnValue is 0 if the signature is valid and 1 is the signature
// fails validation.
//
//*****************************************************************************
static unsigned long CheckSignature(void)
{
    unsigned long ulReturnValue;

    //
    // If signatures match, execute the application.
    //
    if(VerifyAESHash() == true)
    {
        ulReturnValue = 0;
    }
    else
    {
        //
        // Stay in boot loader.
        //
        ulReturnValue = 1;
    }
    return ulReturnValue;
}

//*****************************************************************************
//
// This function is called when the application calls the boot loader.  Since
// the boot loader must run from reset to be able to read the keys, this
// function sets a variable in RAM and does a software reset.  Later,
// MyCheckUpdateFunc() will check for a software reset and the RAM variable.
// If both are true, it will stay in the boot loader instead of running the
// application.
//
// \return None.
//
//*****************************************************************************
void MyReinitFunc(void)
{
    //
    // Store the reboot key in the predefined RAM variable.
    //
    vui64RebootKey = REBOOT_KEY;

    //
    // Trigger a soft reset.
    //
    MAP_SysCtlReset();
}

//*****************************************************************************
//
// AES256Decryption(uint8_t *pucBuffer, uint32_t ulSize);
//
// This function will be called as each packet is received.  The packet will 
// then be decrypted using the key AES256 with the first valid key stored in
// EEPROM.
//
// \return None.
//
//*****************************************************************************
void MyDecryptionFunc(unsigned char *pucBuffer, unsigned int ui32Length)
{
    uint32_t ui32Count;

    //
    // Only execute if there are valid AES keys loaded.
    //
    if(bValidKeys == true)
    {
        //
        // The packet size must be multiple of 16 bytes.
        //
        if((ui32Length & 0xF) == 0)
        {
            //
            // Write the length register first, which triggers the engine to
            // start using this context.
            //
            MAP_AESLengthSet(AES_BASE, (uint64_t)ui32Length);

            //
            // Now loop until the blocks are written.
            //
            for(ui32Count = 0; ui32Count < ui32Length; ui32Count += 16)
            {
                //
                // Write the data registers.
                //
                MAP_AESDataWrite(AES_BASE, (uint32_t *)pucBuffer + (ui32Count / 4));

                //
                // Read the data registers.
                //
                MAP_AESDataRead(AES_BASE, (uint32_t *)pucBuffer + (ui32Count / 4));
            }
        }
    }
}

//*****************************************************************************
//
// This function will be called after all of the data has been programmed but
// before execution is passed to the new program.
//
// If there are no valid keys loaded, it with check the loaded image to see if
// it is a valid key image.  If yes, it will program the keys into EEPROM, hide
// the EEPROM, erase the flash at APP_BASE then reset the part.
//
// \return None.
//
//*****************************************************************************
void MyEndFunc(void)
{
    //
    // First ensure there are not valid keys and the device has a hardware AES
    // encryption module.
    //
    if((bValidKeys == false) && (CheckAESHardware() == true))
    {
        //
        // Then check if valid key image in flash to be written to EEPROM.
        //
        ValidateKeysInFlash();
    }
}

//*****************************************************************************
//
// This function will check the Hash each time from reset and will stay in the
// boot loader if it fails.  You can add additional checks to keep the device
// in the boot loader, such as checking an input pin.
//
// Flow of operation:
//  1) Check for valid keys in the EEPROM.
//  2) If there are valid keys, disable JTAG in case it is not already disabled.
//  3) Initialize the AES module using the latest keys
//  4) Hide the key EEPROM blocks
//
// \return 1 to stay in boot loader or 0 to proceed to executing the
// application.
//
//*****************************************************************************
unsigned long MyCheckUpdateFunc(void)
{
    unsigned long ulReturnValue;
    uint32_t ui32LoopCount;

    //
    // Verify that the device this function is executing on contains a hardware
    // AES encryption module as the Shared Key Boot Loader is dependent on
    // calls to the TM4C129x hardware AES module.  This function will keep the
    // code executing in the shared key boot loader forever if the part number
    // does not match.
    // 
    if(CheckAESHardware() == false)
    {
        bValidKeys = false;
        return 1;
    }
    
    //
    // Enable the CCM module.
    //
    MAP_SysCtlPeripheralEnable(SYSCTL_PERIPH_CCM0);
    
    //
    // Wait for the CCM module to be ready.
    //
    while(!MAP_SysCtlPeripheralReady(SYSCTL_PERIPH_CCM0))
    {
    }
    
    //
    // Check if there are valid keys in the EEPROM.
    //
    CheckValidKeys();
    
    //
    // If there are valid keys, check if the function was called by the
    // application or triggered by the external GPIO signal.  If so, move to
    // the next step.  If not, check for whether or not the image currently 
    // programmed in Flash memory matches the hash stored at the end of Flash.
    // The application will only execute if the the hashes match.
    //
    if(bValidKeys == true)
    {
        if((CheckGPIOPin() == true) || (CalledByApp() == true))
        {
            //
            // Stay in the boot loader.
            //
            ulReturnValue = 1;
        }
        else
        {
            //
            // If the return value is 0, execute the application.
            // Otherwise, execution will stay in the boot loader.
            //
            ulReturnValue = CheckSignature();
        }
    }
    else
    {
        //
        // If there are not valid keys in EEPROM, check if the Flash image
        // contains valid keys.
        //
        ValidateKeysInFlash();
        
        //
        // Stay in the boot loader.
        //
        ulReturnValue = 1;
    }
    
    //
    // If ulReturnValue has been set to 1, then stay in the boot loader to
    // decrypt incoming image.
    //
    // If ulReturnValue has been set to 0, then the hash signature of the Flash
    // image has been validated and the application can execute after the the 
    // EEPROM block containing the Encryption keys are hidden and the keys
    // stored in RAM are cleared to 0xFFFF.FFFF.
    //
    if(ulReturnValue == 1)
    {
        //
        // Prepare to decrypt the incoming data.
        //
        EnableAESCBC(AES_CFG_DIR_DECRYPT);
    }
    else
    {
        MAP_AESReset(AES_BASE);
        
        //
        // Keys are stored in the lasts two blocks of EEPROM.
        // You may choose to hide the last six blocks which make up one sector.
        // This prevents writes to blocks 90-93 causing sector erase and
        // potentially disturbing the keys.
        //
        //MAP_EEPROMBlockHide(90);
        //MAP_EEPROMBlockHide(91);
        //MAP_EEPROMBlockHide(92);
        //MAP_EEPROMBlockHide(93);

        //
        // Now hide the last two EEPROM blocks.
        //
        MAP_EEPROMBlockHide(94);
        MAP_EEPROMBlockHide(95);
        
        //
        // Erase RAM image of keys.
        //
        for(ui32LoopCount = 0; ui32LoopCount < 32; ui32LoopCount++)
        {
            g_pui32Keys[ui32LoopCount] = 0xFFFFFFFFu;
        }
    }

    return ulReturnValue;
}
