//*****************************************************************************
//
// eeprom_erase.c - Simple program to erase the contents of the EEPROM.
//
// Copyright 2021 Texas Instruments Incorporated.
// Software License Agreement
// 
//   Redistribution and use in source and binary forms, with or without
//   modification, are permitted provided that the following conditions
//   are met: 
//
//   1. Redistributions of source code must retain the above copyright
//   notice, this list of conditions and the following disclaimer.
//   
//   2. Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in the
//   documentation and/or other materials provided with the distribution.
// 
//   3. Neither the name of Texas Instruments Incorporated nor the names
//   of its contributors may be used to endorse or promote products
//   derived from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//*****************************************************************************

#include <stdint.h>
#include <stdbool.h>
#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "driverlib/sysctl.h"
#include "driverlib/gpio.h"
#include "driverlib/eeprom.h"

//*****************************************************************************
//
//! This example project demonstrates how to erase the EEPROM entirely.  After
//! executing the EEPROM erase, the program will validate that the EEPROM is
//! has been properly erased before entering a loop to blink an LED which  
//! serves as the external indicator that the EEPROM has been successfully
//! cleared.
//
//*****************************************************************************

//*****************************************************************************
//
// Define pin to LED mapping.
//
//*****************************************************************************
#define USER_LED1  GPIO_PIN_0
#define USER_LED2  GPIO_PIN_1

//*****************************************************************************
//
// Create a buffer with space equal to the EEPROM size.
//
//*****************************************************************************
#define EEPROM_WORDS 1536
uint32_t pui32Data[EEPROM_WORDS]; /* 6KB */

//*****************************************************************************
//
// The error routine that is called if the driver library encounters an error.
//
//*****************************************************************************
#ifdef DEBUG
void
__error__(char *pcFilename, uint32_t ui32Line)
{
}
#endif

//*****************************************************************************
//
// Main application to erase the entire contents of the EEPROM.
//
//*****************************************************************************

int
main(void)
{
    uint32_t ui32SysClock;
    uint32_t ui32EEPROMInit;
    uint32_t ui32LoopCount;
    //
    // Run from the PLL at 120 MHz.
    // Note: SYSCTL_CFG_VCO_240 is a new setting provided in TivaWare 2.2.x and
    // later to better reflect the actual VCO speed due to SYSCTL#22.
    //
    ui32SysClock = SysCtlClockFreqSet((SYSCTL_XTAL_25MHZ |
                                       SYSCTL_OSC_MAIN |
                                       SYSCTL_USE_PLL |
                                       SYSCTL_CFG_VCO_240), 120000000);

    //
    // Enable and wait for the port to be ready for access.
    //
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPION);
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPION))
    {
    }
    
    //
    // Configure the GPIO port for the LED operation.
    //
    GPIOPinTypeGPIOOutput(GPIO_PORTN_BASE, (USER_LED1 | USER_LED2));

    //
    // Enable the EEPROM module.
    //
    SysCtlPeripheralEnable(SYSCTL_PERIPH_EEPROM0);
    
    //
    // Wait for the EEPROM module to be ready.
    //
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_EEPROM0))
    {
    }
    
    //
    // Wait for the EEPROM Initialization to complete.
    //
    ui32EEPROMInit = EEPROMInit();
    
    //
    // Check if the EEPROM Initialization returned an error and do not
    // proceed if so.
    //
    if(ui32EEPROMInit != EEPROM_INIT_OK)
    {
        while(1)
        {
        }
    }
    
    //
    // Erase the EEPROM.
    //
    EEPROMMassErase();
    
    //
    // Read the entire EEPROM.
    //
    for(ui32LoopCount = 0; ui32LoopCount < EEPROM_WORDS; ui32LoopCount++)
    {
        EEPROMRead(pui32Data, 0x000, EEPROM_WORDS << 2);
    }
    
    //
    // Loop forever.
    //
    while(1)
    {
        //
        // Turn on the LED.
        //
        GPIOPinWrite(GPIO_PORTN_BASE, (USER_LED1 | USER_LED2), USER_LED1);

        //
        // Delay for a bit.
        //
        SysCtlDelay(ui32SysClock / 6);

        //
        // Turn on the LED.
        //
        GPIOPinWrite(GPIO_PORTN_BASE, (USER_LED1 | USER_LED2), USER_LED2);

        //
        // Delay for a bit.
        //
        SysCtlDelay(ui32SysClock / 6);
    }
}
