//*****************************************************************************
//
// xmodem.c - Driver for the UART Xmodem simple file transfer protocol.
//
// Copyright 2021 Texas Instruments Incorporated.
// Software License Agreement
// 
//   Redistribution and use in source and binary forms, with or without
//   modification, are permitted provided that the following conditions
//   are met: 
//
//   1. Redistributions of source code must retain the above copyright
//   notice, this list of conditions and the following disclaimer.
//   
//   2. Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in the
//   documentation and/or other materials provided with the distribution.
// 
//   3. Neither the name of Texas Instruments Incorporated nor the names
//   of its contributors may be used to endorse or promote products
//   derived from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//*****************************************************************************
 
#include <ctype.h>
#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "inc/hw_uart.h"
#include "driverlib/aes.h"
#include "driverlib/gpio.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "driverlib/sysctl.h"
#include "driverlib/uart.h"
#include "linker_defines.h"
#include "shared_functions.h"
#include "shared_key_image_encrypt.h"
#include "xmodem.h"

#define NUMBER_PACKETS ((APP_END - APP_BASE)/128)

//*****************************************************************************
//
// \param pucBuffer is a pointer to an unsigned char buffer large enough to
// store all of the data.
//
// This function reads data from the UART using the Xmodem protocol. Each
// packet is 128 bytes long.  The function will return true if the first packet
// did not have any checksum errors and the proper number of packets were
// received.
//
// \return True or False.
//
//*****************************************************************************
bool ReadXmodemKey(unsigned char *pucBuffer)
{
    unsigned char ucChar;
    unsigned int uiLoopCount1, uiLoopCount2;

    //
    // Loop for ten attempts to read a packet.  If no valid packet is received
    // then the function will return false.  The reason for ten attempts is
    // because Xmodem protocol typically includes a maximum of ten retries
    // before aborting a packet transfer.
    //
    for(uiLoopCount1 = 0; uiLoopCount1 < 10; uiLoopCount1++)
    {
        //
        // Read the first block.
        //
        if(ReadXmodemPacket(1, pucBuffer) == 1)
        {
            //
            // If received correctly, send an ACK.
            //
            UARTCharPut(UART0_BASE, ACK);
            break;
        }
        else
        {
            //
            // Packet was not received correctly, send a NACK.
            //
            UARTCharPut(UART0_BASE, NACK);
        }
    }
    
    //
    // If a packet was received correctly the remaining packets do not contain
    // key data as it is only part of the first 128-bytes received.  Therefore,
    // discard all other packages.
    //
    if(uiLoopCount1 < 10)
    {
        //
        // Read and discard up to 132 empty packets
        //
        for(uiLoopCount1 = 0; uiLoopCount1 < 132; uiLoopCount1++)
        {
            ucChar = UARTCharGet(UART0_BASE);
            
            if(ucChar == EOT)
            {
                UARTCharPut(UART0_BASE, ACK);
                return true;
            }
            else if(ucChar == SOH)
            {
                //
                // Throw away the packet.
                //
                for(uiLoopCount2 = 0; uiLoopCount2 < 131; uiLoopCount2++)
                {
                    UARTCharGet(UART0_BASE);
                }
                UARTCharPut(UART0_BASE, ACK);
            }
        }
        
        return false;
    }
    
    return false;
}

//*****************************************************************************
//
// \param pucBuffer is a pointer to an unsigned char buffer large enough to
// store all of the data.
// \param ucBlock is the number of blocks to receive.
//
// This function reads a 128 byte packet from the UART using Xmodem protocol.
// The function returns true if the packet checksum matches.  It does not
// acknowledge the packet.  This allows other processing to be done before
// another packet is sent.
//
// \return The block number, -2 if an EOT was received, and -1 if the operation
// failed.
//
//*****************************************************************************
int ReadXmodemPacket(unsigned char ucBlock, unsigned char *pucBuffer)
{
    unsigned char ucCheckSum1;
    unsigned char ucCheckSum2;
    unsigned char ucBlockNumber;
    unsigned char ucInverseBlockNumber;
    unsigned char ucChar;
    unsigned int uiLoopCount;

    //
    // Loop until a break is triggered.
    //
    while(true)
    {
        //
        // Check for 500mS before sending again.
        //
        for(uiLoopCount = 0; uiLoopCount < 2000000; uiLoopCount++)
        {
            //
            // Poll for a received character.  The first character will
            // indicate the packet header.
            //
            ucChar = UARTCharGetNonBlocking(UART0_BASE);
            
            //
            // If the packet header is either SOH or EOT, break out of the
            // for loop.
            //
            if((ucChar == SOH) || (ucChar == EOT))
            {
                break;
            }
        }
        if(ucChar == EOT)
        {
            //
            // If an EOT is received, return with a -2.
            //
            return -2;
        }
        else if(ucChar == SOH)
        {
            //
            // SOH received, break out of the while(1).
            //
            break;
        }
        else
        {
            //
            // If a SOH was not received, issue a NACK and resume waiting for a
            // received character.
            //
            UARTCharPut(UART0_BASE, NACK);
        }
    }
    
    //
    // Initialize the checksum as zero.
    //
    ucCheckSum1 = 0;
    
    //
    // The next character received after the SOH is the block number.
    //
    ucBlockNumber = UARTCharGet(UART0_BASE);
    ucBlock = ucBlock & 0xFF;
    
    //
    // Verify that the block number passed into the function matches with the
    // received block number.
    //
    if((ucBlockNumber != ucBlock) && (ucBlockNumber != (ucBlock -1)))
    {   
        //
        // If not block number, we may be in the middle of a repeat packet.
        //
        return -1;
    }
    
    //
    // The next character received is the inverse block number.
    //
    ucInverseBlockNumber = UARTCharGet(UART0_BASE);
    
    //
    // Verify that the inverse block number received also matches the received
    // block number after the inversion is undone.
    //
    if(ucBlockNumber != (255 - ucInverseBlockNumber))
    {
        //
        // If not block number, we may be in the middle of a repeat packet.
        //
        return -1;
    }
    
    //
    // Read the 128-byte packet and store it in the provided buffer.
    //
    for(uiLoopCount = 0; uiLoopCount < 128; uiLoopCount++)
    {
        //
        // Use a local variable to pass the value through while avoiding
        // syntax errors.
        //
        ucChar = UARTCharGet(UART0_BASE);
        *pucBuffer++ = ucChar;
        
        //
        // Update the checksum value.
        //
        ucCheckSum1 += ucChar;
    }
    
    //
    // After the 128-byte packet is a checksum value for the packet.
    //
    ucCheckSum2 = UARTCharGet(UART0_BASE);
    
    //
    // Verify the checksums match.  If so, return the block number. If not
    // return -1.
    //
    if(ucCheckSum1 == ucCheckSum2)
    {
        return ucBlockNumber;
    }
    else
    {
        return -1;
    }
}

//*****************************************************************************
//
// This function encrypts and outputs the current application image over UART
// using the Xmodem protocol.
//
// This function could hang if the UART connection is severed so either the
// application needs to have a breakout via timer or this function needs to be
// modified.  Modifications could include adding UART error handling to the
// function which could include adding a retry mechanism.
//
// \return None.
//
//*****************************************************************************
void OutputEncryptedImage(void)
{
    unsigned int uiBlockNumber;
    unsigned int uiLoopCount;
    unsigned int puiOutputBuffer[32];
    unsigned int *pInput;
    unsigned int uiCheckSum;
    unsigned char *pucOutputBuffer;

    //
    // Set the pointer for the initial flash memory location to the start of
    // application flash.
    //
    pInput = (unsigned int *)APP_BASE;

    //
    // Enable the AES module for Encryption.
    //
    EnableAESCBC(AES_CFG_DIR_ENCRYPT);

    while(UARTCharGet(UART0_BASE) != NACK)
    {
        // Wait until the terminal is ready to receive.
    }
   
    //
    // Loop through for the number of 128 byte blocks to output.
    //
    for(uiBlockNumber = 1; uiBlockNumber <= NUMBER_PACKETS; uiBlockNumber++)
    {
        //
        // Encrypt 128 bytes of data.
        //
        AESDataProcess(AES_BASE, pInput, puiOutputBuffer, 128);

        //
        // Repeat the transmission if the terminal responded with a NACK.
        //
        do
        {
            //
            // Output the Start of Header packet, the block number, and
            // the inverse block number.
            //
            UARTCharPut(UART0_BASE, SOH);
            UARTCharPut(UART0_BASE, (uiBlockNumber & 0xFF));
            UARTCharPut(UART0_BASE, (255 - (uiBlockNumber & 0xFF)));
            
            //
            // Type cast the output of the AES encryption from an unsigned int
            // to an unsigned character.
            //
            pucOutputBuffer = (unsigned char *)puiOutputBuffer;
            
            //
            // Reset the check sum to zero.
            //
            uiCheckSum = 0;
            
            //
            // Send out the 128 byte packet of encrypted data while calculating
            // the checksum with every byte sent.
            //
            for(uiLoopCount = 0; uiLoopCount < 128; uiLoopCount++)
            {
                UARTCharPut(UART0_BASE, *pucOutputBuffer);
                uiCheckSum += *pucOutputBuffer++;
            }
            
            //
            // Transmit the calculated checksum.
            //
            UARTCharPut(UART0_BASE, (uiCheckSum & 0xFF));
        }while(UARTCharGet(UART0_BASE) == NACK);
        
        //
        // Point to next 128 bytes.
        //
        pInput += 32;
    }
    
    //
    // Send the End of Transmission packet.
    //
    UARTCharPut(UART0_BASE, EOT);
    
    //
    // Get and discard final ACK.
    //
    UARTCharGet(UART0_BASE);
}

//*****************************************************************************
//
// This function is used to verify if a loaded application image matches an
// encrypted image sent over UART with the Xmodem protocol.  The UART data is
// received and decrpyted and then compared to the corresponding block of flash
// memory.  This function does not return a true or false and instead outputs
// over UART whether or not the verification was completed successfully.
//
// \return None.
//
//*****************************************************************************
void VerifyEncryptedImage(void)
{
    unsigned int uiBlockNumber;
    int iReturnValue;
    unsigned char pucInputBuffer[128];
    unsigned char pucOutputBuffer[128];
    unsigned char *pucAddress;
    bool bVerificationFailed;

    bVerificationFailed = false;
    
    //
    // Enable the AES module for Decryption.
    //
    EnableAESCBC(AES_CFG_DIR_DECRYPT);
    
    //
    // Loop through for the number of 128 byte blocks to verify.
    //
    for(uiBlockNumber = 1; uiBlockNumber <= NUMBER_PACKETS + 1; uiBlockNumber++)
    {
        //
        // Read the next packet and make sure it is a valid packet before
        // continuing on.
        //
        iReturnValue = ReadXmodemPacket(uiBlockNumber, pucInputBuffer);
        
        if(iReturnValue == -1)
        {
            //
            // Checksum failed, request re-transmission of packet.
            //
            UARTCharPut(UART0_BASE, NACK);
            
            //
            // Retry the same packet.
            //
            uiBlockNumber--;
        }
        else if(iReturnValue == (uiBlockNumber & 0xFF))
        {
            if (bVerificationFailed == false)
            {
                //
                // Return value and block number match, good packet.
                // Decrypt the packet contents and store them in the local buffer.
                //
               
                AESDataProcess(AES_BASE, (uint32_t *)pucInputBuffer,
                               (uint32_t *)pucOutputBuffer, 128);
                
                //
                // Get the corresponding address where this block of data should be
                // located in the application image.
                //
                pucAddress = (unsigned char *)(APP_BASE - 128 +
                                               (uiBlockNumber * 128));
            
                //
                // Compare the contents of the decrpyted data and what is in
                // the flash memory and set the failed flag if they do not match.
                //
                if(memcmp(pucOutputBuffer, pucAddress, 128) != 0)
                {
                    //
                    // Some terminal software programs do not handle aborting a
                    // transfer well, so just keep acknowledging packets without
                    // breaking out of the loop.
                    //
                    bVerificationFailed = true;
                }
            }

            //
            // Send an ACK for the packet that was received to trigger the next
            // transmission.
            //
            UARTCharPut(UART0_BASE, ACK);
        }
        else if((iReturnValue + 1) == (uiBlockNumber & 0xFF))
        {
            //
            // Xmodem sent duplicate packet, we can ignore it.
            //
            UARTCharPut(UART0_BASE, ACK);

            //
            // Retry the same packet.
            //
            uiBlockNumber--;
        }
        else if(iReturnValue == -2)
        {
            //
            // End of Transmission was received.  Transmit the ACK.
            //
            UARTCharPut(UART0_BASE, ACK);
            
            //
            // Check if the EOT was received before the entire firmware image
            // has been received.  If so, flag that the verification failed and
            // report the cause over UART.  If not, we are done and can break
            // from the loop to output the result of the verification.
            //
            if(uiBlockNumber <= NUMBER_PACKETS)
            {
                PutStr(" File verification terminated early\n\r");
                bVerificationFailed = true;
            }
            break;
        }
    }
    
    //
    // Output the result of the verification process over UART.
    //
    if(bVerificationFailed == false)
    {
        PutStr(" File verification complete\n\r");
    }
    else
    {
        PutStr(" File verification failed\n\r");
    }
}

//*****************************************************************************
//
// This function handles receiving the firmware image over UART with the Xmodem
// protocol and loading it into flash memory.
// 
// To ensure the image is programmed correctly, checksum checks are made on the
// data being received, the block number is verified to be accurate, and the
// flash programming operation is verified to be successful.
//
// \return None.
//
//*****************************************************************************
void XmodemLoadProgram(void)
{
    unsigned int uiBlockNumber;
    unsigned int uiPacketLength;
    int iReturnValue;
    char ucChar;
    unsigned char pucInputBuffer[128];
    unsigned char *pucAddress;
    bool bProgrammingFailed;

    bProgrammingFailed = false;

    //
    // Turn off prefetch buffers.
    //
    HWREG(0x400FDFC8) = 0x00010000;
    
    //
    // Loop through for the number of 128 byte blocks to program.
    //
    for(uiBlockNumber = 1; uiBlockNumber <= NUMBER_PACKETS; uiBlockNumber++)
    {
        //
        // Read the next packet and make sure it is a valid packet before
        // continuing on.
        //
        iReturnValue = ReadXmodemPacket(uiBlockNumber, pucInputBuffer);
        
        if(iReturnValue == -1)
        {
            //
            // Checksum failed, request re-transmission of packet.
            //
            UARTCharPut(UART0_BASE, NACK);
            
            //
            // Retry the same packet.
            //
            uiBlockNumber--;
        }
        else if(iReturnValue == -2)
        {
            //
            // End of Transmission was received before the full image was
            // transmitted.
            //
            break;
        }
        else if(iReturnValue == (uiBlockNumber & 0xFF))
        {
            //
            // Return value and block number match, good packet.
            //
            if(bProgrammingFailed == false)
            {
                //
                // Calculate the next programming address based on APP_BASE
                // and the current block number.
                //
                pucAddress = (unsigned char *)(APP_BASE - 128 +
                                               (uiBlockNumber * 128));
                
                if(uiBlockNumber == NUMBER_PACKETS)
                {
                    //
                    // This is the final block of the image, so do not program
                    // the last 16 bytes which contains the verification hash.
                    //
                    uiPacketLength = 112;
                }
                else
                {
                    uiPacketLength = 128;
                }
                
                //
                // Program the packet and verify that the operation succeeded.
                //
                if(MAP_FlashProgram((uint32_t *)pucInputBuffer,
                                    (uint32_t)pucAddress,uiPacketLength) != 0)
                {
                    bProgrammingFailed = true;
                }
            }
            
            //
            // Send an ACK for the packet that was received to trigger the next
            // transmission.
            //
            UARTCharPut(UART0_BASE, ACK);
        }
        else if(iReturnValue == ((uiBlockNumber - 1) & 0xFF))
        {
            //
            // Xmodem sent duplicate packet, we can ignore it.
            //
            UARTCharPut(UART0_BASE, ACK);
            
            //
            // Retry the same packet.
            //
            uiBlockNumber--;
        }
        else
        {
            //
            // Case where we missed a packet, need to abort.
            //
            bProgrammingFailed = true;
            UARTCharPut(UART0_BASE, ACK);
        }
    }
    
    if(iReturnValue == -2)
    {
        //
        // End of Transmission was received before the entire firmware image
        // was received, acknowledge it before exiting the function.
        //
        UARTCharPut(UART0_BASE, ACK);
    }
    else
    {
        //
        // Check if the End of Transmission packet has been received after the
        // whole firmware image was programmed and acknowledge it.
        //
        ucChar = UARTCharGet(UART0_BASE);
        if(ucChar == EOT)
        {
            UARTCharPut(UART0_BASE, ACK);
        }
    }
    
    if(bProgrammingFailed == false)
    {
        //
        // Calculate the AES Hash for the firmware image that has been loaded 
        // and have it placed in pucInputBuffer.
        //
        CalculateAESHash((uint32_t *)pucInputBuffer);
        
        //
        // Program the AES Hash into the final 16 bytes of the firmware image.
        //
        MAP_FlashProgram((uint32_t *)pucInputBuffer, (APP_END - 16), 16);
        PutStr(" File programming complete\n\r");
    }
    else
    {
        PutStr(" File programming failed\n\r");
    }
    
    //
    // Enable prefetch buffers by clearing valid tags in the prefetch buffer
    // and forcing prefetch buffers to be enabled.
    //
    HWREG(0x400FDFC8) = 0x00120000;
}

