//*****************************************************************************
//
// shared_functions.c - Contains functions used across multiple projects for
//                      the shared key boot loader.
//
// Copyright 2021 Texas Instruments Incorporated.
// Software License Agreement
// 
//   Redistribution and use in source and binary forms, with or without
//   modification, are permitted provided that the following conditions
//   are met: 
//
//   1. Redistributions of source code must retain the above copyright
//   notice, this list of conditions and the following disclaimer.
//   
//   2. Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in the
//   documentation and/or other materials provided with the distribution.
// 
//   3. Neither the name of Texas Instruments Incorporated nor the names
//   of its contributors may be used to endorse or promote products
//   derived from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//*****************************************************************************

#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "driverlib/aes.h"
#include "driverlib/eeprom.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "driverlib/sysctl.h"
#include "linker_defines.h"
#include "shared_functions.h"

//*****************************************************************************
//
// This function checks to ensure the current device contains a hardware AES
// encryption module which is used by the Shared Key Boot Loader.
//
// \return True or False.
//
//*****************************************************************************
bool CheckAESHardware(void)
{
    uint32_t ui32partNumber;

    //
    // Read the DID1 register and store the upper 16 bytes.
    //
    ui32partNumber = HWREG((uint32_t *)0x400FE004) & 0xFFFF0000;
    
    //
    // Check for a match with each device that comes with a hardware AES
    // module and return true or false accordingly.
    //
    switch(ui32partNumber)
    {
    case TM4C129CNCPDT:
    case TM4C129CNCZAD:
    case TM4C129DNCPDT:
    case TM4C129DNCZAD:
    case TM4C129EKCPDT:
    case TM4C129ENCPDT:
    case TM4C129ENCZAD:
    case TM4C129LNCZAD:
    case TM4C129XKCZAD:
    case TM4C129XNCZAD:
        return true;
    default:
        return false;
    }
}

//*****************************************************************************
//
// \param pui32HashBuffer is a buffer to store the hash tag after processing.
// The provided buffer must at least 4 words in size.
//
// This function will create a hash tag using the AES key and the entire 
// defined space of Flash memory except for the final 16 memory locations.
//
// \return None.
//
//*****************************************************************************
void CalculateAESHash(uint32_t *pui32HashBuffer)
{
    //
    // Enable the AES module in CBC-MODE for encryption.
    //
    EnableAESCBCMAC(AES_CFG_DIR_ENCRYPT);
    
    //
    // Use the AES module to generate the hash of the Flash memory image based
    // on the starting and ending address for the application.
    //
    MAP_AESDataAuth(AES_BASE, (uint32_t *)APP_BASE, APP_END - APP_BASE - 16,
                pui32HashBuffer);
}

//*****************************************************************************
//
// This function will create a hash tag using the AES key and the entire
// defined space of Flash memory except for the final 16 memory locations.
//
// \return True or False.
//
//*****************************************************************************
bool VerifyAESHash(void)
{
    uint32_t pui32HashBuffer[4];

    //
    // Generate the AES hash from the current application image loaded in Flash
    // memory.
    //
    CalculateAESHash(pui32HashBuffer);
    
    //
    // Compare that the generated AES hash matches with the hash that is stored
    // in the final 16 bytes of the application image and return the result.
    //
    if(memcmp(pui32HashBuffer, (uint8_t *)(APP_END - 16), 16) == 0)
    {
        return true;
    }
    else
    {
        return false;
    }
}

//*****************************************************************************
//
// \param ui32encryptDecrypt is the AES module setting to set the direction of
// the AES module for either encryption or decryption mode. The valid values
// that may be passed are AES_CFG_DIR_ENCRYPT and AES_CFG_DIR_DECRYPT.
//
// This function is used to enable the the AES module in Cipher-Block Chaining
// mode. 
//
// \return None.
//
//*****************************************************************************
void EnableAESCBC(uint32_t ui32encryptDecrypt)
{
    uint32_t pui32IV[4], ui32LoopCount;

    //
    // Reset the AES module.
    //
    MAP_AESReset(AES_BASE);
    
    //
    // Configure the AES module for AES CBC.
    //
    MAP_AESConfigSet(AES_BASE, ui32encryptDecrypt | AES_CFG_MODE_CBC |
                 AES_CFG_KEY_SIZE_256BIT);
                 
    //
    // Transfer the const array for the Initialization Vector to a standard
    // array that can be used with TivaWare AES functions.  The Initialization
    // Vector values must be set in InitializationVector.c.
    //
    for (ui32LoopCount = 0; ui32LoopCount < 4; ui32LoopCount++)
    {
        pui32IV[ui32LoopCount] = g_pui32IV[ui32LoopCount];
    }

    //
    // Load the Initialization Vector.
    //
    MAP_AESIVSet(AES_BASE, pui32IV);
    
    //
    // Load the first key of the AES key pair which is used for standard
    // AES CBC operations.
    //
    MAP_AESKey1Set(AES_BASE, g_pui32Keys, AES_CFG_KEY_SIZE_256BIT);
}

//*****************************************************************************
//
// \param ui32encryptDecrypt is the AES module setting to set the direction of
// the AES module for either encryption or decryption mode. The valid values
// that may be passed are AES_CFG_DIR_ENCRYPT and AES_CFG_DIR_DECRYPT.
//
// This function is used to enable the the AES module in Cipher-Block Chaining
// Message Authorization Code mode.
//
// \return None.
//
//*****************************************************************************
void EnableAESCBCMAC(uint32_t ui32encryptDecrypt)
{
    uint32_t pui32IV[4], ui32LoopCount;

    //
    // Reset the AES module.
    //
    MAP_AESReset(AES_BASE);
    
    //
    // Configure the AES module for AES CBC-MAC.
    //
    MAP_AESConfigSet(AES_BASE, ui32encryptDecrypt | AES_CFG_MODE_CBCMAC |
                 AES_CFG_KEY_SIZE_256BIT);

    //
    // Transfer the const variable for the Initialization Vector to a standard
    // array.  The Initialization Vector itself can be set i
    // InitializationVector.c.
    //
    for (ui32LoopCount = 0; ui32LoopCount < 4; ui32LoopCount++)
    {
        pui32IV[ui32LoopCount] = g_pui32IV[ui32LoopCount];
    }

    //
    // Load the Initialization Vector.
    //
    MAP_AESIVSet(AES_BASE, pui32IV);
    
    //
    // Load the second key of the AES key pair which is used for AES CBC-MAC
    // operations.
    //
    MAP_AESKey1Set(AES_BASE, g_pui32Keys + 8, AES_CFG_KEY_SIZE_256BIT);
}

