//*****************************************************************************
//
// shared_key_image_encrypt.c - Application to receive key / firmware images
//                              and generate an encrypted firmware image to be
//                              used with the shared key boot loader.
//
// Copyright 2021 Texas Instruments Incorporated.
// Software License Agreement
// 
//   Redistribution and use in source and binary forms, with or without
//   modification, are permitted provided that the following conditions
//   are met: 
//
//   1. Redistributions of source code must retain the above copyright
//   notice, this list of conditions and the following disclaimer.
//   
//   2. Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in the
//   documentation and/or other materials provided with the distribution.
// 
//   3. Neither the name of Texas Instruments Incorporated nor the names
//   of its contributors may be used to endorse or promote products
//   derived from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//*****************************************************************************

#include <ctype.h>
#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include "inc/hw_flash.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "inc/hw_uart.h"
#include "driverlib/aes.h"
#include "driverlib/eeprom.h"
#include "driverlib/flash.h"
#include "driverlib/gpio.h"
#include "driverlib/pin_map.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "driverlib/sysctl.h"
#include "driverlib/uart.h"
#include "linker_defines.h"
#include "shared_functions.h"
#include "shared_key_image_encrypt.h"
#include "xmodem.h"

//*****************************************************************************
//
// The system clock frequency in Hz.
//
//*****************************************************************************
uint32_t g_ui32SysClock;

//*****************************************************************************
//
// Each possible state of the storage location for the encryption keys within
// device memory.  This value is updated when encryption keys are loaded into
// memory.  A locked state means that the JTAG is locked and key cannot be
// updated.
//
//*****************************************************************************
enum keyState
{
    empty,
    ramBased,
    eepromBased,
    locked
};

//*****************************************************************************
//
// Global variable for the current storage location of the encryption keys.
//
//*****************************************************************************
static enum keyState eValidKeys;

//*****************************************************************************
//
// Buffer to store AES encryption keys in RAM memory.
//
//*****************************************************************************
uint32_t g_pui32Keys[32];

//*****************************************************************************
//
// Send a string to the UART until reaching a null character.
//
//*****************************************************************************
void PutStr(char *pucString)
{
    while(*pucString != '\0')
    {
        MAP_UARTCharPut(UART0_BASE, *pucString++);
    }
}

//*****************************************************************************
//
// Flush the FIFO for the RX UART to ensure there is no garbage data present.
//
//*****************************************************************************
static void FlushUARTRXFIFO(void)
{
    while(UARTCharGetNonBlocking(UART0_BASE) != -1)
    {
    }
}

//*****************************************************************************
//
// Print a simple help menu over UART.
//
//*****************************************************************************
static void PrintMenu(void)
{
    PutStr("K - Load two 256-bit key pairs to EEPROM\n\r");
    PutStr("D - Disable JTAG\n\r");
    PutStr("L - Load a program using XMODEM transfer\n\r");
    PutStr("X - Execute the program loaded at address APP_BASE\n\r");
    PutStr("O - Output a binary image that is signed and encrypted\n\r");
    PutStr("V - Verify binary image against program in flash\n\r");
    PutStr("H - Print this help menu\n\r");
}

//*****************************************************************************
//
// Receive a single character command over UART and echo the character back to
// so the user sees their input if local echo is not enabled.
//
//*****************************************************************************
static unsigned char GetCommand(void)
{
    unsigned char ucCmd;

    MAP_UARTCharPut(UART0_BASE, '>');
    ucCmd = MAP_UARTCharGet(UART0_BASE);
    ucCmd = toupper(ucCmd);
    MAP_UARTCharPut(UART0_BASE, ucCmd);
    MAP_UARTCharPut(UART0_BASE, '\n');
    MAP_UARTCharPut(UART0_BASE, '\r');
    return ucCmd;
}

//*****************************************************************************
//
// Disables JTAG for Release ready firmware only.  This process can only be
// undone by using the device unlock process to restore the entire device to
// factory settings which includes erasing of all data stored.
//
//*****************************************************************************
static void DisableJTAG(void)
{
#ifdef RELEASE
    //
    // Prevent bootloader from being modified.
    //
    MAP_FlashProtectSet(0,FlashReadOnly);
    MAP_FlashProtectSave();
    HWREG(FLASH_FMA) = 0x75100000;
    HWREG(FLASH_FMD) = HWREG(FLASH_BOOTCFG) & 0x7FFFFFFC;
    HWREG(FLASH_FMC) = FLASH_FMC_WRKEY | FLASH_FMC_COMT;
    PutStr(" Must power cycle to complete JTAG lockout\n\r");
#else
    PutStr(" Disable JTAG not implemented in the debug version\n\r");
#endif
}

//*****************************************************************************
//
// This function is used to check for if valid encryption keys are present in
// in device EEPROM memory.  If the EEPROM does not have a key, it will return
// as empty.  Otherwise it will either return as eepromBased or locked
// depending on if the JTAG interface is locked or not.
// 
// \return Returns the current location where the encrpytion keys are stored.
//
//*****************************************************************************
static enum keyState CheckForValidKeys(void)
{
    uint32_t ui32LoopCount, ui32BootCfg, ui32EEPROMInit;

    //
    // Enable the EEPROM module.
    //
    SysCtlPeripheralEnable(SYSCTL_PERIPH_EEPROM0);
    
    //
    // Wait for the EEPROM module to be ready.
    //
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_EEPROM0))
    {
    }
    
    //
    // Wait for the EEPROM Initialization to complete.
    //
    ui32EEPROMInit = EEPROMInit();
    
    //
    // Check if the EEPROM Initialization returned an error and inform
    // the application.
    //
    if(ui32EEPROMInit != EEPROM_INIT_OK)
    {
        PutStr("** Error: EEPROM initialization did not complete\n\r");
        
        return empty;
    }

    //
    // Read the two keys from the second to last block of EEPROM.
    //
    EEPROMRead(g_pui32Keys, 6016, 64);
    
    //
    // Check that the keys are valid by ensuring they are not equal
    // to 0xFFFF.FFFF.
    //
    for(ui32LoopCount = 0; ui32LoopCount < 16; ui32LoopCount++)
    {
        if(g_pui32Keys[ui32LoopCount] != 0xFFFFFFFFu)
        {
            break;
        }
    }
    
    //
    // If the prior loop executed to completion, then there are no valid key
    // pairs in EEPROM, return that keys are still empty.
    //
    if(ui32LoopCount == 16)
    {
        return empty;
    }
    
    //
    // Read BOOTCFG to see if JTAG is locked.
    //
    ui32BootCfg = HWREG(FLASH_BOOTCFG);
    
    if((ui32BootCfg & 3) == 2)
    {  
        //
        // JTAG is still enabled, return that keys are in EEPROM.
        //
        return eepromBased;
    }
    else
    {
        //
        // JTAG is disabled, now hide the last two EEPROM blocks and return
        // that the JTAG is locked.
        //
        MAP_EEPROMBlockHide(94);
        MAP_EEPROMBlockHide(95);
        return locked;
    }
}

//*****************************************************************************
//
// This function is used to retrieve the initial encryption key package by
// receiving it from UART.  The keys are then loaded into the AES peripheral
// and the key status reflects that the keys would be currently stored in RAM.
//
//*****************************************************************************
static int LoadKeys(void)
{
    uint32_t ui32LoopCount;

    //
    // Ensure that JTAG is not locked.
    //
    if(eValidKeys == locked)
    {
        PutStr("** Failed: JTAG is locked and keys cannot be changed\n\r");
    }
    else
    {
        PutStr("** Start transmission of 128 byte key file by Xmodem\n\r<");
        
        //
        // Prepare UART for image reception.
        //
        FlushUARTRXFIFO();
        
        //
        // Receive key image over UART.
        //
        if(ReadXmodemKey((unsigned char *)g_pui32Keys) == true)
        {
            //
            // Check that the keys are valid by ensuring they are not equal
            // to 0xFFFF.FFFF.
            //
            for(ui32LoopCount = 0; ui32LoopCount < 32; ui32LoopCount++)
            {
                //
                // If there are non-0xFFFF.FFFF keys, then break from the
                // check after marking where the first valid key is located.
                //
                if(g_pui32Keys[ui32LoopCount] != 0xFFFFFFFFu)
                {
                    //
                    // Valid keys are only provided in the second half of the
                    // key image.  Keys are used based on the first half of the
                    // key image so overwrite the empty first half with the
                    // valid second half keys.
                    //
                    if (ui32LoopCount >= 16)
                    {
                        memcpy(g_pui32Keys, g_pui32Keys + 16, 64);
                    }

                    break;
                }

                //
                // If we reach this statement on the last byte, then an image
                // of all 0xFFFF.FFFF keys has been provided.
                //
                if (ui32LoopCount == 31)
                {
                    PutStr(" Invalid key image provided.\n\r");

                    return 0;
                }
            }

            //
            // Load the AES keys received as the AES Key 1 set.
            //
            MAP_AESKey1Set(AES_BASE, g_pui32Keys, AES_CFG_KEY_SIZE_256BIT);

            eValidKeys = ramBased;
            
            PutStr(" Keys loaded successfully\n\r");
        }
        else
        {
            PutStr(" Key load failed\n\r");

            return 0;
        }
    }

    return 1;
}

//*****************************************************************************
//
// This function takes encryption keys that are stored in RAM via the
// g_pui32Keys buffer and stores them into EEPROM.
//
//*****************************************************************************
static void SaveKeys(void)
{
    switch(eValidKeys)
    {
    case empty:
        PutStr("** Failed: No valid keys loaded\n\r");
        break;
    case ramBased:
        //
        // Program the 4 keys in the last two blocks of EEPROM.
        //
        if(MAP_EEPROMProgram(g_pui32Keys, 6016, 128) == 0)
        {
            PutStr("** Keys successfully programmed into EEPROM\n\r");
        }
        else
        {
            PutStr("** Failed: EEPROM programming failed\n\r");
        }
        break;
    case eepromBased:
        PutStr("** Failed: Keys are already in EEPROM\n\r");
        break;
    case locked:
        PutStr("** Failed: JTAG is locked and keys cannot be changed\n\r");
        break;
    }
}

//*****************************************************************************
//
// This function erases the flash memory before loading in a new firmware image
// over UART via the Xmodem protocol.  The firmware image must be validated
// after this function executes.
//
//*****************************************************************************
static void LoadProgram(void)
{
    uint32_t ui32Address;

    PutStr("** Erasing Flash\n\r");

    //
    // Start by erasing all of flash memory.
    //
    for(ui32Address = APP_BASE; ui32Address < APP_END; ui32Address += 0x4000)
    {
        if(MAP_FlashErase(ui32Address) != 0)
        {
            PutStr("** Failed: Flash could not be erased\n\r");
            break;
        }
    }

    if(ui32Address == APP_END)
    {
        PutStr("** Start transmission of binary program file\n\r<");

        //
        // Prepare UART for image reception.
        //
        FlushUARTRXFIFO();

        //
        // Load the new firmware binary into the device over UART with the
        // Xmodem protocol.  This image is only verified as being sent over
        // UART correctly via a checksum and has not been validated as a
        // legitimate firmware image yet.
        //
        XmodemLoadProgram();
    }
}

//*****************************************************************************
//
// This function verifies that the current firmware image is a valid one and
// then deletes the encryption keys from RAM before jumping to the application.
//
//*****************************************************************************
static void ExecuteProgram(void)
{
    uint32_t ui32LoopCount;

    //
    // Verify that the current firmware image in valid.
    //
    if(VerifyAESHash() == true)
    {
        PutStr("< Hash value authenticated\n\r");
        
        //
        // Delete the RAM image of the keys.
        //
        for(ui32LoopCount = 0; ui32LoopCount < 32; ui32LoopCount++)
        {
            g_pui32Keys[ui32LoopCount] = 0xFFFFFFFFu;
        }
        
        while(MAP_UARTBusy(UART0_BASE))
        {
	    //
            // Wait for message to finish transmitting.
	    //
        }
        JumpToApplication();
    }
    else
    {
        PutStr("** Error: Hash value did not authenticate\n\r");
    }
}

//*****************************************************************************
//
// Configure the UART and its pins.  This must be called before UARTprintf().
//
//*****************************************************************************
static void ConfigureUART(void)
{
    //
    // Enable the GPIO Peripheral used by the UART.
    //
    MAP_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);

    //
    // Enable UART0.
    //
    MAP_SysCtlPeripheralEnable(SYSCTL_PERIPH_UART0);

    //
    // Configure GPIO Pins for UART mode.
    //
    MAP_GPIOPinConfigure(GPIO_PA0_U0RX);
    MAP_GPIOPinConfigure(GPIO_PA1_U0TX);
    MAP_GPIOPinTypeUART(GPIO_PORTA_BASE, GPIO_PIN_0 | GPIO_PIN_1);

    //
    // Initialize the UART for console I/O.
    //
    MAP_UARTConfigSetExpClk(UART0_BASE, g_ui32SysClock, 460800,
           (UART_CONFIG_WLEN_8 | UART_CONFIG_STOP_ONE | UART_CONFIG_PAR_NONE));
    MAP_UARTFIFOEnable(UART0_BASE);
}

//*****************************************************************************
//
// Main function which handles initializations and receiving UART commands from
// the user.
//
//*****************************************************************************
int 
main(void)
{
    unsigned char ucCmd;

    //
    // Initialize the eValidKeys enum as empty.
    //
    eValidKeys = empty;

    //
    // Run from the PLL at 120 MHz.
    // Note: SYSCTL_CFG_VCO_240 is a new setting provided in TivaWare 2.2.x and
    // later to better reflect the actual VCO speed due to SYSCTL#22.
    //
    g_ui32SysClock = SysCtlClockFreqSet((SYSCTL_XTAL_25MHZ |
                                       SYSCTL_OSC_MAIN |
                                       SYSCTL_USE_PLL |
                                       SYSCTL_CFG_VCO_240), 120000000);
                                       
    //
    // Initialize the UART and write status.
    //
    ConfigureUART();
    PutStr("Create Shared Key Image Tool, Version 1.00\n\r");
    
    //
    // Ensure that the program is executing on a TM4C device which comes with
    // the hardware AES encryption module.
    //
    if(CheckAESHardware() == false)
    {
        PutStr("This device does not support AES hardware\n\r");
        return 0;
    }
    
    //
    // Enable the CCM module.
    //
    SysCtlPeripheralEnable(SYSCTL_PERIPH_CCM0);
    
    //
    // Protect the boot block from erase or program
    //
    //FlashProtectSet(0x00000000, FlashReadOnly);
    
    //
    // Check for valid AES keys being present before receiving commands.
    //
    eValidKeys = CheckForValidKeys();
    
    while(1)
    {
        //
        // Print the help menu.
        //
        PrintMenu();
        
        //
        // Wait for a command to be sent.
        //
        ucCmd = GetCommand();
        
        //
        // Process command.
        //
        switch(ucCmd)
        {
        case 'K':
            //
            // Receive the two key pairs.
            //
            if(LoadKeys())
            {
                //
                // Valid keys received.  Load the key pairs into EEPROM.
                //
                SaveKeys();
            }
            break;
        case 'D':
            //
            // Disable the JTAG.  The process to do so will only be executed
            // if the firmware is set to be in Release state.  This can only
            // be reserved with the device unlock procedure.
            //
            DisableJTAG();
            break;
        case 'L':
            if(eValidKeys == empty)
            {
                PutStr("** Error: No valid keys loaded yet\n\r");
            }
            else
            {
                //
                // Erase the flash memory and then receive a new firmware
                // image over UART.  This image will not be validated against
                // the AES hash yet.
                //
                LoadProgram();
            }
            break;
        case 'X':
            if(eValidKeys == empty)
            {
                PutStr("** Error: No valid keys loaded yet\n\r");
            }
            else
            {
                //
                // Validates the AES hash of the firmware image and then
                // executes the program.
                //
                ExecuteProgram();
            }
            break;
        case 'O':
            if(eValidKeys == empty)
            {
                PutStr("** Error: No valid keys loaded yet\n\r");
            }
            else
            {
                PutStr("** Start xmodem file receive\n\r<");

                //
                // Outputs the current loaded firmware image with the image
                // being both signed and encrypted before the transfer.
                //
                OutputEncryptedImage();

                PutStr(" Encrypted firmware image sent\n\r");
            }
            break;
        case 'V':
            if(eValidKeys == empty)
            {
                PutStr("** Error: No valid keys loaded yet\n\r");
            }
            else
            {
                PutStr("** Start transmission of encrypted data file by"
                        " Xmodem\n\r<");

                //
                // Prepare UART for image reception.
                //
                FlushUARTRXFIFO();

                //
                // Receive an encrypted firmware image and compare it's
                // contents against the current firmware image.
                //
                VerifyEncryptedImage();
            }
            break;
        case 'H':
            break;
        default:
            PutStr("< Invalid command ");
            MAP_UARTCharPut(UART0_BASE, ucCmd);
            MAP_UARTCharPut(UART0_BASE, '\n');
            MAP_UARTCharPut(UART0_BASE, '\r');
        }
    }
}


