/*
 * display.c
 *
 * ============================================================================
 * Copyright (c) Texas Instruments Inc 2005
 *
 * Use of this software is controlled by the terms and conditions found in the
 * license agreement under which this software has been supplied or provided.
 * ============================================================================
 */

/* Standard Linux headers */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include <sched.h>
#include <pthread.h>
#include <asm/types.h>
#include <sys/resource.h>
#include <linux/videodev2.h>
#include <linux/fb.h>

/* Demo headers */
#include <timerutil.h>
#include <rendezvous.h>
#include <fifoutil.h>
#include "decode.h"
#include "display.h"
#include <video/davincifb.h>

/* Custom Davinci FBDEV defines (should be in device driver header) */
#define VID0_INDEX 0
#define VID1_INDEX 1
#define ZOOM_1X    0
#define ZOOM_2X    1
#define ZOOM_4X    2

struct Zoom_Params
{
    u_int32_t WindowID;
    u_int32_t Zoom_H;
    u_int32_t Zoom_V;
};
#define FBIO_SETZOOM        _IOW('F', 0x24, struct Zoom_Params)
#define FBIO_WAITFORVSYNC   _IOW('F', 0x20, u_int32_t)

/* Video display is triple buffered */
#define NUM_BUFS                 3

/* Black color in UYVY format */
#define UYVY_BLACK               0x10801080

/* The levels of initialization */
#define DISPLAYDEVICEINITIALIZED 0x1

static int initDisplayDevice(char *displays[], Resolution resolution);
static void cleanupDisplayDevice(int fd, char *displays[],
				 Resolution resolution);
static int flipDisplayBuffers(int fd, int displayIdx);

/******************************************************************************
 * flipDisplayBuffers
 ******************************************************************************/
static int flipDisplayBuffers(int fd, int displayIdx)
{
    struct fb_var_screeninfo vInfo;
    int                      dummy;

    if (ioctl(fd, FBIOGET_VSCREENINFO, &vInfo) == -1) {
        ERR("Failed FBIOGET_VSCREENINFO (%s)\n", strerror(errno));
        return FAILURE;
    }

    vInfo.yoffset = vInfo.yres * displayIdx;

    /* Swap the working buffer for the displayed buffer */
    if (ioctl(fd, FBIOPAN_DISPLAY, &vInfo) == -1) {
        ERR("Failed FBIOPAN_DISPLAY (%s)\n", strerror(errno));
        return FAILURE;
    }

    /* Wait for vertical sync */
    if (ioctl(fd, FBIO_WAITFORVSYNC, &dummy) == -1) {
        ERR("Failed FBIO_WAITFORVSYNC (%s)\n", strerror(errno));
        return FAILURE;
    }
    
    return SUCCESS;
}

/******************************************************************************
 * initDisplayDevice
 ******************************************************************************/
static int initDisplayDevice(char *displays[], Resolution resolution)
{
    struct fb_var_screeninfo varInfo;
    struct lcd_timing_params lcd_timing;
    struct Zoom_Params       zoom;
    unsigned int            *buf;
    int                      fd;
    int                      i;


    /* Open video display device */
    if(resolution == RES_720P || resolution == RES_1080I )
      //|| resolution == RES_VGA)
      fd = open(FBVID_HD_DEVICE, O_RDWR);
    else
      fd = open(FBVID_DEVICE, O_RDWR);

    if (fd == -1) {
        ERR("Failed to open fb device %s (%s)\n", FBVID_DEVICE,
                                                  strerror(errno));
        return FAILURE;
    }
    
    if (ioctl(fd, FBIOGET_VSCREENINFO, &varInfo) == -1) {
        ERR("Failed FBIOGET_VSCREENINFO on %s (%s)\n", FBVID_DEVICE,
                                                       strerror(errno));
        return FAILURE;
    }

    varInfo.bits_per_pixel = SCREEN_BPP;
    varInfo.xoffset = 0;
   
    if(resolution == RES_NTSC) {
      varInfo.xres = WIDTH_NTSC;
      varInfo.yres = 480;
    } else if(resolution == RES_PAL) {
      varInfo.xres = WIDTH_PAL;
      varInfo.yres = 576;
    } else if(resolution == RES_VGA) {
      varInfo.xres = 640;
      varInfo.yres = 480;
      lcd_timing.h_interval = 800;
      lcd_timing.v_interval = 524;
      lcd_timing.h_valid = 640;
      lcd_timing.v_valid = 480;
      lcd_timing.h_pulse = 96;
      lcd_timing.v_pulse = 32;
      lcd_timing.h_start = 23;
      lcd_timing.v_start = 32;
      lcd_timing.h_delay = 0;
      lcd_timing.v_delay = 0;
      lcd_timing.hsync_pol = 0;
      lcd_timing.vsync_pol = 0;
      lcd_timing.oe_pol = 1;
    } else if(resolution == RES_720P) {
      varInfo.xres = WIDTH_720P;
      varInfo.yres = HEIGHT_720P;
    } else if(resolution == RES_1080I) {
      varInfo.xres = WIDTH_1080I;
      varInfo.yres = HEIGHT_1080I;
    } else {
      printf("Resolution not set.  Using NTSC Default.\n");
      varInfo.xres = WIDTH_NTSC;
      varInfo.yres = 480;
    }

    /* Work around to force FBIOPUT_VSCREENINFO to call davincifb_set_par */
    if(varInfo.yoffset == 0) {
      varInfo.yoffset = varInfo.yres;

      /* Set video display format */
      /*if(resolution == RES_VGA) {
	if (ioctl(fd, FBIO_SETLCDTIMEPARAMS, &lcd_timing) == -1) {
	  ERR("Failed FBIO_SETLCDTIMEPARAMS on %s (%s)\n", FBVID_DEVICE,
	      strerror(errno));
	  return FAILURE;
	} 
	} else*/ if (ioctl(fd, FBIOPUT_VSCREENINFO, &varInfo) == -1) {
	ERR("Failed FBIOPUT_VSCREENINFO on %s (%s)\n", FBVID_DEVICE,
	    strerror(errno));
	return FAILURE;
	}
    }

    varInfo.yoffset = 0;

    /* Set video display format */
    /*if(resolution == RES_VGA) {
      if (ioctl(fd, FBIO_SETLCDTIMEPARAMS, &lcd_timing) == -1) {
	ERR("Failed FBIO_SETLCDTIMEPARAMS on %s (%s)\n", FBVID_DEVICE,
	    strerror(errno));
	return FAILURE;
      } 
      } else*/ if (ioctl(fd, FBIOPUT_VSCREENINFO, &varInfo) == -1) {
      ERR("Failed FBIOPUT_VSCREENINFO on %s (%s)\n", FBVID_DEVICE,
	  strerror(errno));
      return FAILURE;
      }

    switch(resolution) {
    case RES_NTSC:
      if (varInfo.xres != WIDTH_NTSC ||
	  varInfo.yres != 480 ||
	  varInfo.bits_per_pixel != SCREEN_BPP) {
        ERR("Failed to get the requested screen size: %dx%d at %d bpp\n",
            WIDTH_NTSC, HEIGHT_NTSC, SCREEN_BPP);
        return FAILURE;
      }
      break;
    case RES_PAL:
      if (varInfo.xres != WIDTH_PAL ||
	  varInfo.yres != 576 ||
	  varInfo.bits_per_pixel != SCREEN_BPP) {
        ERR("Failed to get the requested screen size: %dx%d at %d bpp\n",
            WIDTH_PAL, HEIGHT_PAL, SCREEN_BPP);
        return FAILURE;
      }
      break;
    case RES_VGA:
      if ((varInfo.xres != WIDTH_VGA && varInfo.xres != WIDTH_NTSC) ||
	  varInfo.yres != 480 ||
	  varInfo.bits_per_pixel != SCREEN_BPP) {
        ERR("Failed to get the requested screen size: %dx%d at %d bpp\n",
            WIDTH_PAL, HEIGHT_PAL, SCREEN_BPP);
        return FAILURE;
      }
      break;
    case RES_720P:
      if (varInfo.xres != WIDTH_720P ||
	  varInfo.yres != HEIGHT_720P ||
	  varInfo.bits_per_pixel != SCREEN_BPP) {
        ERR("Failed to get the requested screen size: %dx%d at %d bpp\n",
            WIDTH_720P, HEIGHT_720P, SCREEN_BPP);
        return FAILURE;
      }
      break;
    case RES_1080I:
      if (varInfo.xres != WIDTH_1080I ||
	  varInfo.yres != HEIGHT_1080I ||
	  varInfo.bits_per_pixel != SCREEN_BPP) {
        ERR("Failed to get the requested screen size: %dx%d at %d bpp\n",
            WIDTH_1080I, HEIGHT_1080I, SCREEN_BPP);
        return FAILURE;
      }
      break;
    default:
      if (varInfo.xres != WIDTH_NTSC ||
	  varInfo.yres != 480 ||
	  varInfo.bits_per_pixel != SCREEN_BPP) {
        ERR("Failed to get the requested screen size: %dx%d at %d bpp\n",
            WIDTH_NTSC, HEIGHT_NTSC, SCREEN_BPP);
        return FAILURE;
      }
      break;
    }      

    /* Map the video buffers to user space */
    switch(resolution) {
    case RES_NTSC:
      displays[0] = (char *) mmap (NULL,
				   FRAME_SIZE_NTSC * NUM_BUFS,
				   PROT_READ | PROT_WRITE,
				   MAP_SHARED,
				   fd, 0);
      
      if (displays[0] == MAP_FAILED) {
        ERR("Failed mmap on %s (%s)\n", FBVID_DEVICE, strerror(errno));
        return FAILURE;
      }
      break;
    case RES_PAL:
      displays[0] = (char *) mmap (NULL,
				   FRAME_SIZE_PAL * NUM_BUFS,
				   PROT_READ | PROT_WRITE,
				   MAP_SHARED,
				   fd, 0);
      
      if (displays[0] == MAP_FAILED) {
        ERR("Failed mmap on %s (%s)\n", FBVID_DEVICE, strerror(errno));
        return FAILURE;
      }
      break;
    case RES_VGA:
      displays[0] = (char *) mmap (NULL,
				   FRAME_SIZE_NTSC * NUM_BUFS,
				   PROT_READ | PROT_WRITE,
				   MAP_SHARED,
				   fd, 0);
      
      if (displays[0] == MAP_FAILED) {
        ERR("Failed mmap on %s (%s)\n", FBVID_DEVICE, strerror(errno));
        return FAILURE;
      }
      break;
    case RES_720P:
      displays[0] = (char *) mmap (NULL,
				   FRAME_SIZE_720P * NUM_BUFS,
				   PROT_READ | PROT_WRITE,
				   MAP_SHARED,
				   fd, 0);

      if (displays[0] == MAP_FAILED) {
        ERR("Failed mmap on %s (%s)\n", FBVID_DEVICE, strerror(errno));
        return FAILURE;
      }
      break;
    case RES_1080I:
      displays[0] = (char *) mmap (NULL,
				   FRAME_SIZE_1080I * NUM_BUFS,
				   PROT_READ | PROT_WRITE,
				   MAP_SHARED,
				   fd, 0);
      
      if (displays[0] == MAP_FAILED) {
        ERR("Failed mmap on %s (%s)\n", FBVID_DEVICE, strerror(errno));
        return FAILURE;
      }
      break;
    default:
      displays[0] = (char *) mmap (NULL,
				   FRAME_SIZE_NTSC * NUM_BUFS,
				   PROT_READ | PROT_WRITE,
				   MAP_SHARED,
				   fd, 0);
      
      if (displays[0] == MAP_FAILED) {
        ERR("Failed mmap on %s (%s)\n", FBVID_DEVICE, strerror(errno));
        return FAILURE;
      }
      break;
    }

    /* Clear the video buffers */
    buf = (unsigned int *) displays[0];
 
   switch(resolution) {
    case RES_NTSC:
      for (i=0; i<FRAME_SIZE_NTSC * NUM_BUFS / sizeof(unsigned int); i++) {
        buf[i] = UYVY_BLACK;
      }

      DBG("Display buffer %d mapped to address %#lx\n", 0,
	  (unsigned long) displays[0]);

      for (i=0; i<NUM_BUFS-1; i++) {
        displays[i+1] = displays[i] + FRAME_SIZE_NTSC;
        DBG("Display buffer %d mapped to address %#lx\n", i+1,
            (unsigned long) displays[i+1]);
      }
      break;
    case RES_PAL:
      for (i=0; i<FRAME_SIZE_PAL * NUM_BUFS / sizeof(unsigned int); i++) {
        buf[i] = UYVY_BLACK;
      }

      DBG("Display buffer %d mapped to address %#lx\n", 0,
	  (unsigned long) displays[0]);

      for (i=0; i<NUM_BUFS-1; i++) {
        displays[i+1] = displays[i] + FRAME_SIZE_PAL;
        DBG("Display buffer %d mapped to address %#lx\n", i+1,
            (unsigned long) displays[i+1]);
      }
      break;
    case RES_VGA:
      for (i=0; i<FRAME_SIZE_NTSC * NUM_BUFS / sizeof(unsigned int); i++) {
        buf[i] = UYVY_BLACK;
      }

      DBG("Display buffer %d mapped to address %#lx\n", 0,
	  (unsigned long) displays[0]);

      for (i=0; i<NUM_BUFS-1; i++) {
        displays[i+1] = displays[i] + FRAME_SIZE_NTSC;
        DBG("Display buffer %d mapped to address %#lx\n", i+1,
            (unsigned long) displays[i+1]);
      }
      break;
    case RES_720P:
      for (i=0; i<FRAME_SIZE_720P * NUM_BUFS / sizeof(unsigned int); i++) {
        buf[i] = UYVY_BLACK;
      }

      DBG("Display buffer %d mapped to address %#lx\n", 0,
	  (unsigned long) displays[0]);

      for (i=0; i<NUM_BUFS-1; i++) {
        displays[i+1] = displays[i] + FRAME_SIZE_720P;
        DBG("Display buffer %d mapped to address %#lx\n", i+1,
            (unsigned long) displays[i+1]);
      }
      break;
    case RES_1080I:
      for (i=0; i<FRAME_SIZE_1080I * NUM_BUFS / sizeof(unsigned int); i++) {
        buf[i] = UYVY_BLACK;
      }

      DBG("Display buffer %d mapped to address %#lx\n", 0,
	  (unsigned long) displays[0]);

      for (i=0; i<NUM_BUFS-1; i++) {
        displays[i+1] = displays[i] + FRAME_SIZE_1080I;
        DBG("Display buffer %d mapped to address %#lx\n", i+1,
            (unsigned long) displays[i+1]);
      }
      break;
    default:
      for (i=0; i<FRAME_SIZE_NTSC * NUM_BUFS / sizeof(unsigned int); i++) {
        buf[i] = UYVY_BLACK;
      }

      DBG("Display buffer %d mapped to address %#lx\n", 0,
	  (unsigned long) displays[0]);

      for (i=0; i<NUM_BUFS-1; i++) {
        displays[i+1] = displays[i] + FRAME_SIZE_NTSC;
        DBG("Display buffer %d mapped to address %#lx\n", i+1,
            (unsigned long) displays[i+1]);
      }
      break;
    }

    /* Make sure there is no zoom of this video window */
    zoom.WindowID = VID1_INDEX;
    zoom.Zoom_H = ZOOM_1X;
    zoom.Zoom_V = ZOOM_1X;
   
    if (ioctl(fd, FBIO_SETZOOM, &zoom)) {
      ERR("Failed setting zoom parameters\n");
      return FAILURE;
    }
    
    return fd;
}

/******************************************************************************
 * cleanupDisplayDevice
 ******************************************************************************/
static void cleanupDisplayDevice(int fd, char *displays[], 
				 Resolution resolution)
{
  switch(resolution) {
  case RES_NTSC:
    munmap(displays[0], FRAME_SIZE_NTSC * NUM_BUFS);
    break;
  case RES_PAL:
    munmap(displays[0], FRAME_SIZE_PAL * NUM_BUFS);
    break;
  case RES_VGA:
    munmap(displays[0], FRAME_SIZE_NTSC * NUM_BUFS);
    break;
  case RES_720P:
    munmap(displays[0], FRAME_SIZE_720P * NUM_BUFS);
    break;
  case RES_1080I:
    munmap(displays[0], FRAME_SIZE_1080I * NUM_BUFS);
    break;
  default:
    munmap(displays[0], FRAME_SIZE_NTSC * NUM_BUFS);
    break;
  }

    close(fd);
}


/******************************************************************************
 * displayThrFxn
 ******************************************************************************/
void *displayThrFxn(void *arg)
{
    DisplayEnv     *envp          = (DisplayEnv *) arg;
    BufferElement   flush         = { DISPLAY_FLUSH };
    void           *status        = THREAD_SUCCESS;
    unsigned int    initMask      = 0;
    int             displayIdx    = 0;
    int             fbFd          = 0;
    int             framesDropped = 0;
    int             totalFrames   = 0;
    char           *displays[NUM_BUFS];
    unsigned long   frameTime;
    TimerUtilObj    timer;
    BufferElement   e;
    int i;

    /* Initialize the video display device */
    fbFd = initDisplayDevice(displays, envp->resolution);

    if (fbFd == FAILURE) {
        cleanup(THREAD_FAILURE);
    }

    DBG("Video display device initialized.\n");

    /* Load Display Buffer Pointers into Video Env */
    for(i=0; i<NUM_BUFS; i++) {
      envp->videoEnv->bufferElements[i].frameBuffer = displays[i];
      envp->videoEnv->bufferElements[i].id = i;
    }

    initMask |= DISPLAYDEVICEINITIALIZED;

    /* Signal that initialization is done and wait for other threads */
    Rendezvous_meet(envp->hRendezvous);

    DBG("Entering display main loop.\n");
    while (TRUE) {
        if (!gblGetPlay() && !gblGetQuit()) {
            usleep(PAUSE);
            continue;
        }

	//        TimerUtil_reset(&timer);

        /* Receive a buffer with a decoded frame from the video thread */
        if (FifoUtil_get(&envp->outFifo, &e) == FIFOUTIL_FAILURE) {        
            breakLoop(THREAD_FAILURE);
        }

        /* Is the video thread flushing the pipe? */
        if (e.id == DISPLAY_FLUSH) {
            breakLoop(THREAD_SUCCESS);
        } else if (e.id == DISPLAY_PRIME) {
            pthread_mutex_lock(&envp->prime);
            pthread_mutex_unlock(&envp->prime);
            continue;
        }

        /* Set next display index */
	displayIdx = e.id;

        /* Give back the buffer to the video thread */
        if (FifoUtil_put(&envp->inFifo, &e) == FIFOUTIL_FAILURE) {
            breakLoop(THREAD_FAILURE);
        }

        /* Flip display buffer and working buffer */
	flipDisplayBuffers(fbFd, displayIdx); 

        gblIncFrames();

	//        TimerUtil_total(&timer, &frameTime);

        /* See if displaying the buffer exceeds our frame budget */
	/*if((envp->resolution == RES_NTSC) || (envp->resolution == RES_PAL)) {
	  if (frameTime > 35000) {
	    DBG("Warning: Frame dropped [%lu us]\n", frameTime);
            framesDropped++;
	  }
	} else {
	  if (frameTime > 17500) {
	    DBG("Warning: Frame dropped [%lu us]\n", frameTime);
	    framesDropped++;
	  }
	  }*/
    }

    totalFrames = gblGetAndResetFrames();

    printf("\nTotal number of video frames dropped: %d\n", framesDropped);
    printf("\nTotal video frames: %d\n",totalFrames);

cleanup:
    /* Make sure the other threads aren't waiting for init to complete */
    Rendezvous_force(envp->hRendezvous);

    /* Make sure the video thread isn't stuck in FifoUtil_get() */
    FifoUtil_put(&envp->inFifo, &flush);

    /* Clean up the display thread */
    if (initMask & DISPLAYDEVICEINITIALIZED) {
        cleanupDisplayDevice(fbFd, displays, envp->resolution);
    }

    return status;
}


