//############################################################################
//
// FILE:   F2833x_EEPROM.c
//
// TITLE:  EEPROM Read, Write, and Erase Functions
//
//############################################################################
// Authors: Tim Love / Pradeep Shinde
// Release Date: Sep 2009
//############################################################################


#include "F2833x_EEPROM.h"					// EEPROM Include File				

// Global Variables
Uint16 Read_Buffer[64];
Uint16 Write_Buffer[64];	
Uint16 Bank_Counter = 0;
Uint16 Page_Counter = 0;
Uint16 Bank_Status[1] = {0};
Uint16 Page_Status[1] = {0};
Uint16 *Bank_Pointer;		
Uint16 *Page_Pointer;
Uint16 *Sector_End;

FLASH_ST FlashStatus;
FLASH_ST ProgStatus;

// All Flash API functions need to be ran from internal RAM. Place all functions
// that contain Flash API calls in ramfuncs to allow copy from Flash to RAM.
#pragma CODE_SECTION(EEPROM_Erase,"ramfuncs");
#pragma CODE_SECTION(EEPROM_Write,"ramfuncs");
#pragma CODE_SECTION(EEPROM_UpdateBankStatus,"ramfuncs");
#pragma CODE_SECTION(EEPROM_UpdatePageStatus,"ramfuncs");
#pragma CODE_SECTION(EEPROM_ProgramSingleByte,"ramfuncs");


//######################### EEPROM_GET_VALID_BANK ############################
void EEPROM_GetValidBank()
{
//Each page holds 64 data words
//Page size = Page_status word + 64W = 65W
//Bank Size = Bank_status word + 8 * Page size = 1 + 8*65 = 521W

	Uint16 i;

   	RESET_BANK_POINTER;		// Reset Bank Pointer to enable search for current Bank
   	RESET_PAGE_POINTER;		// Reset Page Pointer to enable search for current Page

	// Find Current Bank 
	for(i=0;i<8;i++)
	{
		Bank_Status[0] = *(Bank_Pointer);		// Read contents of Bank Pointer			
		
		if(Bank_Status[0] == EMPTY_BANK)		// Check for Unused Bank  
		{
			Bank_Counter = i;					// Set Bank Counter to number of current page								 	
			return;								// If Bank is Unused, return as EEPROM is empty
		}
	
		if(Bank_Status[0] == CURRENT_BANK)		// Check for Current Bank 
		{
			Bank_Counter = i;					// Set Bank Counter to number of current bank
			Page_Pointer = Bank_Pointer + 1;	// Set Page Pointer to first page in current bank
			break;								// Break from loop as current bank has been found
		}

		if(Bank_Status[0] == USED_BANK)   		// Check for Used Bank											
			Bank_Pointer += 521; 				// If Bank has been used, set pointer to next bank
		
	}

	// Find Current Page
	for(i=0;i<8;i++)
	{
		Page_Status[0] = *(Page_Pointer);		// Read contents of Page Pointer
		
		// Check for Blank Page or Current Page	
		if(Page_Status[0] == BLANK_PAGE || Page_Status[0] == CURRENT_PAGE)
		{
			Page_Counter = i;					// Set Page Counter to number of current page		
			break;								// Break from loop as current page has been found
		}

		if(Page_Status[0] == USED_PAGE)			// Check for Used Page
			Page_Pointer += 65;					// If page has been used, set pointer to next page
	}

	if (Bank_Counter==7 && Page_Counter==7)		// Check for full EEPROM
	{
		EEPROM_Erase();							// Erase flash sector being used as EEPROM
  		RESET_BANK_POINTER; 					// Reset Bank Pointer as EEPROM is empty
   		RESET_PAGE_POINTER;						// Reset Pank Pointer as EEPROM is empty
		asm(" ESTOP0");
	}
}
//######################### EEPROM_GET_VALID_BANK ############################


//############################# EEPROM_ERASE #################################
void EEPROM_Erase()
{
	Uint16  Status;

	#ifdef F28332					// If F28332/F28232 is being used erase Sector D
	Status = Flash_Erase((SECTORD),&FlashStatus);
	#else							// Otherwise erase Sector H for other devices
	Status = Flash_Erase((SECTORH),&FlashStatus);
	#endif

   	if(Status != STATUS_SUCCESS)	// If Erase fails halt program or handle error 
   	{
      	asm(" ESTOP0");
   	} 
}
//############################# EEPROM_ERASE #################################


//############################# EEPROM_READ ##################################
void EEPROM_Read()
{
	Uint16 i;

	EEPROM_GetValidBank();					// Find Current Bank and Current Page

	// Transfer contents of Current Page to Read Buffer
	for(i=0;i<64;i++)
		Read_Buffer[i] = *(++Page_Pointer); 
}
//############################# EEPROM_READ ##################################


//############################ EEPROM_WRITE ##################################
void EEPROM_Write()
{
	// Variables needed for Flash API Functions
	Uint16 Status;
	Uint32 Length;
	FLASH_ST ProgStatus;

	EEPROM_GetValidBank();					// Find Current Bank and Current Page

    EEPROM_UpdatePageStatus();				// Update Page Status of previous page 
	EEPROM_UpdateBankStatus();				// Update Bank Status of current and previous bank

	// Program data located in Write_Buffer to current page
	Length = 64;							// Set Length to size of page length for programming
	Status = Flash_Program(Page_Pointer+1,Write_Buffer,Length,&ProgStatus);
	
	// Modify Page Status from Blank Page to Current Page if flash programming was successful
	if (Status == STATUS_SUCCESS)   				 
	{
		Page_Status[0] = CURRENT_PAGE;  	// Set Page Status to Current Page
		Length = 1;							// Set Length for programming
		Status = Flash_Program(Page_Pointer,Page_Status,Length,&ProgStatus);
	}
}
//############################ EEPROM_WRITE ##################################


//###################### EEPROM_UPDATE_BANK_STATUS ###########################
void EEPROM_UpdateBankStatus()
{
	// Variables needed for Flash API Functions
	Uint16 Status;
	Uint32 Length;

	Length = 1;								// Set Length for programming

	Bank_Status[0] = *(Bank_Pointer);		// Read Bank Status from Bank Pointer
	Page_Status[0] = *(Page_Pointer);		// Read Page Status from Page Pointer

	// Program Bank Status for Empty EEPROM
	if (Bank_Status[0] == EMPTY_BANK)		
	{	
		Bank_Status[0] = CURRENT_BANK;  	// Set Bank Status to Current Bank
		
		// Program Bank Status to current bank
		Status = Flash_Program(Bank_Pointer,Bank_Status,Length,&ProgStatus);
		Page_Counter =0;
		Page_Pointer = Bank_Pointer + 1;	// Set Page Pointer to first page of current bank
	}	

	// Program Bank Status of full bank and following bank
	if (Bank_Status[0] == CURRENT_BANK && Page_Counter == 7)	
	{	
		Bank_Status[0] = USED_BANK;			// Set Bank Status to Used Bank
		
		// Program Bank Status to full bank			
		Status = Flash_Program(Bank_Pointer,Bank_Status,Length,&ProgStatus);
		
		Bank_Pointer +=521;					// Increment Bank Pointer to next bank
	
		Bank_Status[0] = CURRENT_BANK;		// Set Bank Status to Current Bank
				
		// Program Bank Status to current bank		
		Status = Flash_Program(Bank_Pointer,Bank_Status,Length,&ProgStatus);

		Page_Counter =0;
		Page_Pointer = Bank_Pointer +1;		// Set Page Pointer to first page of current bank
	}
}
//###################### EEPROM_UPDATE_BANK_STATUS ###########################


//###################### EEPROM_UPDATE_PAGE_STATUS ###########################
void EEPROM_UpdatePageStatus()
{
	
	Uint16 Status;
	Uint32 Length;

	Bank_Status[0] = *(Bank_Pointer);		// Read Bank Status from Bank Pointer			
	Page_Status[0] = *(Page_Pointer);		// Read Page Status from Page Pointer

	// Check if Page Status is blank. If so return to EEPROM_WRITE.
	if(Page_Status[0] == BLANK_PAGE)
		return;
	
	// Program previous page's status to Used Page
	else
	{			
		Page_Status[0] = USED_PAGE; 		// Set Page Status to Used Page 	
		Length = 1;							// Set Length for programming
		Status = Flash_Program(Page_Pointer,Page_Status,Length,&ProgStatus);		 
		Page_Pointer +=65;					// Increment Page Pointer to next page
	}
}
//###################### EEPROM_UPDATE_PAGE_STATUS ###########################


/*//###################### EEPROM_GET_SINGLE_POINTER ###########################
void EEPROM_GetSinglePointer()
{
	Uint16 *End_Address;

	RESET_BANK_POINTER;						// Reset Bank Pointer to beginning of sector
	End_Address = (Uint16 *)END_OF_SECTOR;	// Set End_Address for sector

	while(*(Bank_Pointer) != 0xFFFF)		// Test each location for data
	{
		Bank_Pointer++;						// Increment to next location
		if(Bank_Pointer == End_Address)		// Test if sector is full 
		{    
			EEPROM_Erase();					// Erase flash sector being used as EEPROM
  			RESET_BANK_POINTER; 			// Reset Bank Pointer as EEPROM is empty
   			//RESET_PAGE_POINTER;				// Reset Pank Pointer as EEPROM is empty
			asm(" ESTOP0");
		}
	}
}
//###################### EEPROM_GET_SINGLE_POINTER ###########################*/

//###################### EEPROM_GET_SINGLE_POINTER ###########################
void EEPROM_GetSinglePointer(Uint16 First_Call)
{
	Uint16 *End_Address;

	End_Address = (Uint16 *)END_OF_SECTOR;	// Set End_Address for sector

	if(First_Call == 1)						// If this is first call to function, find valid pointer
	{	
		RESET_BANK_POINTER;					// Reset Bank Pointer to beginning of sector
		while(*(Bank_Pointer) != 0xFFFF)	// Test each location for data
			Bank_Pointer++;					// Increment to next location
	}
		
	if(Bank_Pointer >= End_Address)			// Test if sector is full 
	{    
		EEPROM_Erase();						// Erase flash sector being used as EEPROM
  		RESET_BANK_POINTER; 				// Reset Bank Pointer as EEPROM is empty
   		asm(" ESTOP0");
	}
	
}
//###################### EEPROM_GET_SINGLE_POINTER ###########################


//##################### EEPROM_PROGRAM_SINGLE_BYTE ###########################
void EEPROM_ProgramSingleByte(Uint16 data)
{
	// Variables needed for Flash API Functions
	Uint16 Status;
	Uint32 Length;

	Write_Buffer[0] = data;					// Prepare data to be programmed
		
	Length = 1;								// Set Length for programming
	Status = Flash_Program(Bank_Pointer++,Write_Buffer,Length,&ProgStatus);
	EEPROM_GetSinglePointer(0);				// Test for full sector	
}
//##################### EEPROM_PROGRAM_SINGLE_BYTE ###########################
