/*
 * Copyright (C) 2018 Texas Instruments Incorporated - http://www.ti.com/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the
 * distribution.
 *
 * Neither the name of Texas Instruments Incorporated nor the names of
 * its contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>
#include <ti/csl/cslr.h>
#include <ti/csl/tistdtypes.h>
#include <ti/csl/soc.h>
#include <ti/csl/arch/csl_arch.h>
#include <ti/csl/hw_types.h>
#include <ti/drv/uart/UART_stdio.h>
#include <ti/drv/uart/soc/UART_soc.h>
#include <ti/board/board.h>
#include <sbl_soc.h>
#include <sbl_soc_cfg.h>
#include <sbl_err_trap.h>
#include <sbl_sci_client.h>
#if defined(SOC_J721E)
#include <ti/drv/sciclient/src/rm_pm_hal/include/soc/j721e/hosts.h>
#include <ti/csl/soc/j721e/src/csl_soc_firewalls.h>
#endif

#if defined(SOC_J721E)

struct ti_sci_msg_fwl_region {
    uint16_t            fwl_id;
    uint16_t            region;
    uint32_t            n_permission_regs;
    uint32_t            control;
    uint32_t            permissions[FWL_MAX_PRIVID_SLOTS];
    uint64_t        start_address;
    uint64_t        end_address;
} __attribute__((__packed__));

void J721E_dump_owner_req(struct tisci_msg_fwl_change_owner_info_req *req)
{
	uint32_t i = 0;

	SBL_log(SBL_LOG_ERR,"\n");
	SBL_log(SBL_LOG_ERR,"Ownership Request:\n");
	SBL_log(SBL_LOG_ERR,"req.fwl_id                = 0x%x\n",req->fwl_id);
	SBL_log(SBL_LOG_ERR,"req.owner_index           = 0x%x\n",req->owner_index);
	SBL_log(SBL_LOG_ERR,"req.region                = 0x%x\n",req->region);

    uint8_t *bPtr = (uint8_t *) req;
	for (i = 0; (i < sizeof(struct tisci_msg_fwl_change_owner_info_req)); i++)
	{
		SBL_log(SBL_LOG_ERR,"%02x ", bPtr[i]);
	}
	SBL_log(SBL_LOG_ERR,"\n");
	return;
}


void J721E_dump_region_req(struct tisci_msg_fwl_set_firewall_region_req *req)
{
	uint32_t i = 0;
    uint32_t *startAddr;
    uint32_t *endAddr;

    startAddr = (uint32_t *) &req->start_address;
    endAddr = (uint32_t *) &req->end_address;

	SBL_log(SBL_LOG_ERR,"\n");
	SBL_log(SBL_LOG_ERR,"Region Set Request:\n");
	SBL_log(SBL_LOG_ERR,"req.fwl_id            = %d\n",   req->fwl_id);
	SBL_log(SBL_LOG_ERR,"req.region            = %d\n",   req->region);
	SBL_log(SBL_LOG_ERR,"req.control           = 0x%x\n", req->control);
	SBL_log(SBL_LOG_ERR,"req.start_address     = 0x%x%x\n", startAddr[1], startAddr[0]);
	SBL_log(SBL_LOG_ERR,"req.end_address       = 0x%x%x\n", endAddr[1], endAddr[0]);
	SBL_log(SBL_LOG_ERR,"req.n_permission_regs = 0x%x\n", req->n_permission_regs);
	for(i = 0; i < req->n_permission_regs; i++)
	{
		SBL_log(SBL_LOG_ERR, "req.permissions[%d]    = 0x%x\n", i, req->permissions[0]);
	}
	SBL_log(SBL_LOG_ERR, "\n");
	return;
}

void J721E_Set_Firewall(uint32_t isBuildHs)
{
    int32_t status = CSL_EFAIL;

	struct ti_sci_msg_fwl_region j721e_fwl_data[] = {

	    /* compute_cluster Master firewall  - background region 0 */
	    {
	        .fwl_id = CSL_MSTR_FW_A72SS0_CORE0_CPU_0_CPU_0_MSMC_ID,
	        .region = 0,
	        .n_permission_regs = 1,
	        .control = 0x30A,
	        .start_address = 0x00000000,
	        .end_address = 0xFFFFFFfff,
	        .permissions = { 0x1FFFF }, // PrivId 1U
	    },
	    /* compute_cluster Master firewall  - region 1 */
	    {
	        .fwl_id = CSL_MSTR_FW_A72SS0_CORE0_CPU_0_CPU_0_MSMC_ID,
	        .region = 1,
	        .n_permission_regs = 1,
	        .control = 0x20A,
	        .start_address = 0xa0000000,
	        .end_address = 0xa8ffffff,
	        .permissions = { 0x10000 }, // PrivId 1U
	    },
	    /* compute_cluster Master firewall  - region 2 */
	    {
	        .fwl_id = CSL_MSTR_FW_A72SS0_CORE0_CPU_0_CPU_0_MSMC_ID,
	        .region = 2,
	        .n_permission_regs = 1,
	        .control = 0x20A,
	        .start_address = 0xce000000,
	        .end_address = 0xfbffffff,
	        .permissions = { 0x10000 }, // PrivId 1U
	    },


	};

	uint32_t j721e_fwl_count = 3; // Number of entries
	uint32_t i = 0;
	uint32_t j = 0;

    struct tisci_msg_fwl_set_firewall_region_resp respFwCtrl = {0};
    struct tisci_msg_fwl_set_firewall_region_req reqFwCtrl;

	for (i = 0; i < j721e_fwl_count; i++)
	{
	    /* Setting Owner */
	    struct tisci_msg_fwl_change_owner_info_req req;
	        req.fwl_id = (uint16_t)j721e_fwl_data[i].fwl_id;
	        req.region = (uint16_t) j721e_fwl_data[i].region;
	        req.owner_index = (uint8_t) HOST_ID_MCU_0_R5_1; // Cortex R5 context 1 on MCU island(Boot)

	    struct tisci_msg_fwl_change_owner_info_resp resp = {0};
	    // Uncomment for debug J721E_dump_owner_req(&req);
	    status = Sciclient_firewallChangeOwnerInfo(&req, &resp, SCICLIENT_SERVICE_WAIT_FOREVER);
		if (status != CSL_PASS)
		{
			SBL_log(SBL_LOG_ERR,"Firewall Unable to change Owner, %d\n", i);
		    J721E_dump_owner_req(&req);
		}

		/* Setting Region */
		reqFwCtrl.fwl_id = (uint16_t) j721e_fwl_data[i].fwl_id;
		reqFwCtrl.region = (uint16_t) j721e_fwl_data[i].region;
		reqFwCtrl.n_permission_regs = (uint32_t) j721e_fwl_data[i].n_permission_regs;
		reqFwCtrl.control = (uint32_t) j721e_fwl_data[i].control;
		for(j = 0; j < reqFwCtrl.n_permission_regs; j++)
		{
			reqFwCtrl.permissions[j] = (uint32_t) j721e_fwl_data[i].permissions[j];
		}
		reqFwCtrl.start_address = j721e_fwl_data[i].start_address;
		reqFwCtrl.end_address = j721e_fwl_data[i].end_address;

		// Uncomment for debug J721E_dump_region_req(&reqFwCtrl);
		status = Sciclient_firewallSetRegion(&reqFwCtrl, &respFwCtrl, SCICLIENT_SERVICE_WAIT_FOREVER);
		if (status != CSL_PASS)
		{
			SBL_log(SBL_LOG_ERR,"Firewall entry/%d, region set failed.\n", i);
			J721E_dump_region_req(&reqFwCtrl);
		}
	}
}
#endif

/* This function is to be called from other apps (e.g., mcusw boot app) to set QoS settings */
void SBL_SetFirewall(uint32_t isBuildHs)
{
#if defined(SOC_J721E)
    J721E_Set_Firewall(isBuildHs);
#endif
}


