/*
 * Copyright (C) 2018-2019 Texas Instruments Incorporated - http://www.ti.com/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the
 * distribution.
 *
 * Neither the name of Texas Instruments Incorporated nor the names of
 * its contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 *  \file       uart_main.c
 *
 *  \brief      This file receives image with header from uart uniflash
 *              and flash on device mentioned in the header.
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include <xmodem.h>
#include <ti/csl/arch/csl_arch.h>
#include <ti/board/utils/uniflash/target/soc/soc.h>

#include <ti/drv/uart/UART_stdio.h>
#include <ti/drv/uart/soc/UART_soc.h>

#include <ti/drv/sciclient/soc/V1/sciclient_firmware_V1.h>

#include <ti/drv/sciclient/sciclient.h>

#ifdef SPI_FLASH
#include <spi.h>
#endif
#ifdef QSPI_FLASH
#include <qspi.h>
#endif
#ifdef OSPI_FLASH
#include <ospi.h>
#endif
#ifdef EMMC_FLASH
#include <emmc.h>
#endif
#ifdef HPF_FLASH
#include <hyperflash.h>
#endif

#if defined(am65xx_evm) || defined(am65xx_idk) || defined(j721e_evm)
#include <ti/osal/CacheP.h>
#endif

/* ========================================================================== */
/*                            Defines                                         */
/* ========================================================================== */

#define MAX_CHAR                (60)
#define FNAME_SZ                (512U)
#define OFFSET_ADDR             (0x80000000U)
#define SIZE_ADDR               (0x80000004U)
#define LOAD_ADDR               (0x90000000U)
#define CHECK_ADDR              (0xA0000000U)
#define FLASH_SIZE_ALIGN        (4096)

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

extern uint32_t DELAY;
extern uint8_t uart_inst;

UFP_flashConfig UPF_flashFxnPtr[FLASH_DEVICE_MAX] = {
    { NULL },                   // NAND
    { NULL },                   // SPI
    { NULL },                   // QSPI
    { &UFP_ospiFxnTable },      // OSPI
};

#if defined(am65xx_evm) || defined(am65xx_idk) || defined(j721e_evm)
#pragma DATA_SECTION(gSysFirmware, ".firmware")
uint32_t gSysFirmware[] = SCICLIENT_FIRMWARE;
#endif

static Board_flashHandle gOspiHandle;

const CSL_ArmR5MpuRegionCfg gCslR5MpuCfg[CSL_ARM_R5F_MPU_REGIONS_MAX] =
{
    {
        /* Region 0 configuration: complete 32 bit address space = 4Gbits */
        .regionId         = 0U,
        .enable           = 1U,
        .baseAddr         = 0x0U,
        .size             = CSL_ARM_R5_MPU_REGION_SIZE_4GB,
        .subRegionEnable  = CSL_ARM_R5_MPU_SUB_REGION_ENABLE_ALL,
        .exeNeverControl  = 1U,
        .accessPermission = CSL_ARM_R5_ACC_PERM_PRIV_USR_RD_WR,
        .shareable        = 0U,
        .cacheable        = (uint32_t)FALSE,
        .cachePolicy      = 0U,
        .memAttr          = 0U,
    },
    {
        /* Region 1 configuration: 128 bytes memory for exception vector execution */
        .regionId         = 1U,
        .enable           = 1U,
        .baseAddr         = 0x0U,
        .size             = CSL_ARM_R5_MPU_REGION_SIZE_32KB,
        .subRegionEnable  = CSL_ARM_R5_MPU_SUB_REGION_ENABLE_ALL,
        .exeNeverControl  = 0U,
        .accessPermission = CSL_ARM_R5_ACC_PERM_PRIV_USR_RD_WR,
        .shareable        = 0U,
        .cacheable        = (uint32_t)TRUE,
        .cachePolicy      = CSL_ARM_R5_CACHE_POLICY_NON_CACHEABLE,
        .memAttr          = 0U,
    },
    {
        /* Region 2 configuration: 1 MB OCMS RAM - Covers RAM sizes for multiple SoCs */
        .regionId         = 2U,
        .enable           = 1U,
        .baseAddr         = 0x41C00000,
        .size             = CSL_ARM_R5_MPU_REGION_SIZE_1MB,
        .subRegionEnable  = CSL_ARM_R5_MPU_SUB_REGION_ENABLE_ALL,
        .exeNeverControl  = 0U,
        .accessPermission = CSL_ARM_R5_ACC_PERM_PRIV_USR_RD_WR,
        .shareable        = 0U,
        .cacheable        = (uint32_t)TRUE,
        .cachePolicy      = CSL_ARM_R5_MEM_ATTR_CACHED_WT_NO_WA,
        .memAttr          = 0U,
    },
    {
        /* Region 3 configuration: 2 MB MCMS3 RAM */
        .regionId         = 3U,
        .enable           = 1U,
        .baseAddr         = 0x70000000,
        .size             = CSL_ARM_R5_MPU_REGION_SIZE_8MB,
        .subRegionEnable  = CSL_ARM_R5_MPU_SUB_REGION_ENABLE_ALL,
        .exeNeverControl  = 0U,
        .accessPermission = CSL_ARM_R5_ACC_PERM_PRIV_USR_RD_WR,
        .shareable        = 0U,
        .cacheable        = (uint32_t)TRUE,
        .cachePolicy      = CSL_ARM_R5_MEM_ATTR_CACHED_WT_NO_WA,
        .memAttr          = 0U,
    },
    {
        /* Region 4 configuration: 2 GB DDR RAM */
        .regionId         = 4U,
        .enable           = 1U,
        .baseAddr         = 0x80000000,
        .size             = CSL_ARM_R5_MPU_REGION_SIZE_2GB,
        .subRegionEnable  = CSL_ARM_R5_MPU_SUB_REGION_ENABLE_ALL,
        .exeNeverControl  = 0U,
        .accessPermission = CSL_ARM_R5_ACC_PERM_PRIV_USR_RD_WR,
        .shareable        = 0U,
        .cacheable        = (uint32_t)TRUE,
        .cachePolicy      = CSL_ARM_R5_MEM_ATTR_CACHED_WT_NO_WA,
        .memAttr          = 0U,
    },
    {
        /* Region 5 configuration: 64 KB BTCM */
        .regionId         = 5U,
        .enable           = 1U,
        .baseAddr         = 0x41010000,
        .size             = CSL_ARM_R5_MPU_REGION_SIZE_32KB,
        .subRegionEnable  = CSL_ARM_R5_MPU_SUB_REGION_ENABLE_ALL,
        .exeNeverControl  = 0U,
        .accessPermission = CSL_ARM_R5_ACC_PERM_PRIV_USR_RD_WR,
        .shareable        = 0U,
        .cacheable        = (uint32_t)TRUE,
        .cachePolicy      = CSL_ARM_R5_CACHE_POLICY_NON_CACHEABLE,
        .memAttr          = 0U,
    },
    {
        /* Region 6 configuration: Covers first 32MB of EVM Flash (FSS DAT0) */
        .regionId         = 6U,
        .enable           = 1U,
        .baseAddr         = 0x50000000,
        .size             = CSL_ARM_R5_MPU_REGION_SIZE_32MB,
        .subRegionEnable  = CSL_ARM_R5_MPU_SUB_REGION_ENABLE_ALL,
        .exeNeverControl  = 0U,
        .accessPermission = CSL_ARM_R5_ACC_PERM_PRIV_USR_RD_WR,
        .shareable        = 0U,
        .cacheable        = (uint32_t)TRUE,
        .cachePolicy      = CSL_ARM_R5_MEM_ATTR_CACHED_WT_NO_WA,
        .memAttr          = 0U,
    },
    {
        /* Region 7 configuration: Covers next 16MB of EVM Flash (FSS DAT0) */
        .regionId         = 7U,
        .enable           = 1U,
        .baseAddr         = 0x52000000,
        .size             = CSL_ARM_R5_MPU_REGION_SIZE_16MB,
        .subRegionEnable  = CSL_ARM_R5_MPU_SUB_REGION_ENABLE_ALL,
        .exeNeverControl  = 0U,
        .accessPermission = CSL_ARM_R5_ACC_PERM_PRIV_USR_RD_WR,
        .shareable        = 0U,
        .cacheable        = (uint32_t)TRUE,
        .cachePolicy      = CSL_ARM_R5_MEM_ATTR_CACHED_WT_NO_WA,
        .memAttr          = 0U,
    },
    {
        /* Region 8 configuration: Covers next 8MB of EVM Flash (FSS DAT0) */
        .regionId         = 8U,
        .enable           = 1U,
        .baseAddr         = 0x53000000,
        .size             = CSL_ARM_R5_MPU_REGION_SIZE_8MB,
        .subRegionEnable  = CSL_ARM_R5_MPU_SUB_REGION_ENABLE_ALL,
        .exeNeverControl  = 0U,
        .accessPermission = CSL_ARM_R5_ACC_PERM_PRIV_USR_RD_WR,
        .shareable        = 0U,
        .cacheable        = (uint32_t)TRUE,
        .cachePolicy      = CSL_ARM_R5_MEM_ATTR_CACHED_WT_NO_WA,
        .memAttr          = 0U,
    },
    {
        /* Region 9 configuration: Covers next 4MB of EVM Flash (FSS DAT0) */
        .regionId         = 9U,
        .enable           = 1U,
        .baseAddr         = 0x53800000,
        .size             = CSL_ARM_R5_MPU_REGION_SIZE_4MB,
        .subRegionEnable  = CSL_ARM_R5_MPU_SUB_REGION_ENABLE_ALL,
        .exeNeverControl  = 0U,
        .accessPermission = CSL_ARM_R5_ACC_PERM_PRIV_USR_RD_WR,
        .shareable        = 0U,
        .cacheable        = (uint32_t)TRUE,
        .cachePolicy      = CSL_ARM_R5_MEM_ATTR_CACHED_WT_NO_WA,
        .memAttr          = 0U,
    },
    {
        /* Region 10 configuration: Covers next 2MB of EVM Flash (FSS DAT0) */
        .regionId         = 10U,
        .enable           = 1U,
        .baseAddr         = 0x53C00000,
        .size             = CSL_ARM_R5_MPU_REGION_SIZE_2MB,
        .subRegionEnable  = CSL_ARM_R5_MPU_SUB_REGION_ENABLE_ALL,
        .exeNeverControl  = 0U,
        .accessPermission = CSL_ARM_R5_ACC_PERM_PRIV_USR_RD_WR,
        .shareable        = 0U,
        .cacheable        = (uint32_t)TRUE,
        .cachePolicy      = CSL_ARM_R5_MEM_ATTR_CACHED_WT_NO_WA,
        .memAttr          = 0U,
    },
    {
        /* Region 11 configuration: Covers next 1MB of EVM Flash (FSS DAT0) */
        .regionId         = 11U,
        .enable           = 1U,
        .baseAddr         = 0x53E00000,
        .size             = CSL_ARM_R5_MPU_REGION_SIZE_1MB,
        .subRegionEnable  = CSL_ARM_R5_MPU_SUB_REGION_ENABLE_ALL,
        .exeNeverControl  = 0U,
        .accessPermission = CSL_ARM_R5_ACC_PERM_PRIV_USR_RD_WR,
        .shareable        = 0U,
        .cacheable        = (uint32_t)TRUE,
        .cachePolicy      = CSL_ARM_R5_MEM_ATTR_CACHED_WT_NO_WA,
        .memAttr          = 0U,
    },
    {
        /* Region 12 configuration: Covers next 512KB of EVM Flash (FSS DAT0) */
        .regionId         = 12U,
        .enable           = 1U,
        .baseAddr         = 0x53F00000,
        .size             = CSL_ARM_R5_MPU_REGION_SIZE_512KB,
        .subRegionEnable  = CSL_ARM_R5_MPU_SUB_REGION_ENABLE_ALL,
        .exeNeverControl  = 0U,
        .accessPermission = CSL_ARM_R5_ACC_PERM_PRIV_USR_RD_WR,
        .shareable        = 0U,
        .cacheable        = (uint32_t)TRUE,
        .cachePolicy      = CSL_ARM_R5_MEM_ATTR_CACHED_WT_NO_WA,
        .memAttr          = 0U,
    },
    {
        /* Region 13 configuration: Covers next 256KB of EVM Flash (FSS DAT0) */
        .regionId         = 13U,
        .enable           = 1U,
        .baseAddr         = 0x53F80000,
        .size             = CSL_ARM_R5_MPU_REGION_SIZE_256KB,
        .subRegionEnable  = CSL_ARM_R5_MPU_SUB_REGION_ENABLE_ALL,
        .exeNeverControl  = 0U,
        .accessPermission = CSL_ARM_R5_ACC_PERM_PRIV_USR_RD_WR,
        .shareable        = 0U,
        .cacheable        = (uint32_t)TRUE,
        .cachePolicy      = CSL_ARM_R5_MEM_ATTR_CACHED_WT_NO_WA,
        .memAttr          = 0U,
    },
    {
        /* Region 14 configuration (Non-cached for PHY tuning data): Covers last 256KB of EVM Flash (FSS DAT0) */
        .regionId         = 14U,
        .enable           = 1U,
        .baseAddr         = 0x53FC0000,
        .size             = CSL_ARM_R5_MPU_REGION_SIZE_256KB,
        .subRegionEnable  = CSL_ARM_R5_MPU_SUB_REGION_ENABLE_ALL,
        .exeNeverControl  = 0U,
        .accessPermission = CSL_ARM_R5_ACC_PERM_PRIV_USR_RD_WR,
        .shareable        = 0U,
        /* OSPI PHY tuning algorithm which runs in DAC mode needs
         * cache to be disabled for this section of FSS data region.
         */
        .cacheable        = (uint32_t)FALSE,
        .cachePolicy      = 0U,
        .memAttr          = 0U,
    },
    {
        /* Region 15 configuration: 128 MB FSS DAT1 */
        .regionId         = 15U,
        .enable           = 1U,
        .baseAddr         = 0x58000000,
        .size             = CSL_ARM_R5_MPU_REGION_SIZE_128MB,
        .subRegionEnable  = CSL_ARM_R5_MPU_SUB_REGION_ENABLE_ALL,
        .exeNeverControl  = 0U,
        .accessPermission = CSL_ARM_R5_ACC_PERM_PRIV_USR_RD_WR,
        .shareable        = 0U,
        .cacheable        = (uint32_t)TRUE,
        .cachePolicy      = CSL_ARM_R5_MEM_ATTR_CACHED_WT_NO_WA,
        .memAttr          = 0U,
    },
};

static int8_t ospiFlashWrite(uint8_t *src, uint32_t offset, uint32_t length)
{
    int32_t status;
    uint32_t startBlockNum, endBlockNum, pageNum, i;
    uint32_t writeMode;

    writeMode = OSPI_FLASH_OCTAL_PAGE_PROG;

    if (!(offset % NOR_SECTOR_SIZE))
    {
        /* Get starting block number */
        if (Board_flashOffsetToSectorPage(gOspiHandle, offset, &startBlockNum, &pageNum, false))
        {
            return -1;
        }

        /* Get ending block number */
        if (Board_flashOffsetToSectorPage(gOspiHandle, offset+length, &endBlockNum, &pageNum, false))
        {
            return -1;
        }

        /* Erase blocks, to which data has to be written */
        for(i = startBlockNum; i <= endBlockNum; i++)
        {
            if (Board_flashEraseBlk(gOspiHandle, i))
            {
                return -1;
            }
        }
    }

    /* Write buffer to flash */
    #if 0
    for(i = 0; i < length; i+=OSPI_WR_LEN)
    {
        if (Board_flashWrite(gOspiHandle, offset, src, OSPI_WR_LEN, &writeMode))
        {
            return -1;
        }
        offset = offset + OSPI_WR_LEN;
        src = src + OSPI_WR_LEN;
    }
    #else
    status = Board_flashWrite(gOspiHandle, offset, src, length, &writeMode);
    #endif

    return status;
}

static int8_t ospiFlashRead(uint8_t *dst, uint32_t offset, uint32_t length)
{
    uint32_t readMode;
    readMode = OSPI_FLASH_OCTAL_READ;

    if (Board_flashRead(gOspiHandle, offset, dst, length, &readMode))
    {
        return -1;
    }

    return 0;
}

#define ALIGN(x, a) (((x) + ((a) - 1)) & (~((a) - 1)))

int main(void)
{
    uint32_t                fileSize, orig_fileSize;
    uint32_t                offset;
    uint8_t                *srcAddr = (uint8_t *)LOAD_ADDR;
    uint8_t                *checkAddr = (uint8_t *)CHECK_ADDR;
    OSPI_v0_HwAttrs         ospi_cfg;
    Board_STATUS            status;
    Board_initCfg           boardCfg;
    Board_FlashInfo        *flashInfo;

    #if 0
    status = Sciclient_loadFirmware((const uint32_t *) gSysFirmware);

    if (CSL_PASS != status)
    {
        printf ("sysfw loading failed \n");
        return -1;
    }
    else
    {
        printf ("Sysfw Loaded\n");
    }
    #endif

    boardCfg = BOARD_INIT_MODULE_CLOCK |
               BOARD_INIT_PINMUX_CONFIG |
               BOARD_INIT_UART_STDIO;

    status = Board_init(boardCfg);
    if(status != BOARD_SOK)
    {
        return -1;
    }

    /* Get the default OSPI init configurations */
    OSPI_socGetInitCfg(BOARD_OSPI_INSTANCE, &ospi_cfg);

    /* Modify the default OSPI configurations if necessary */
    /* Turning off interrupts for baremetal mode. May be re-enabled by app */
    ospi_cfg.intrEnable = false;
    ospi_cfg.dmaEnable  = false;
    ospi_cfg.phyEnable  = false;
    //ospi_cfg.phyEnable  = true;

    /* Set the default OSPI init configurations */
    OSPI_socSetInitCfg(BOARD_OSPI_INSTANCE, &ospi_cfg);

    gOspiHandle = Board_flashOpen(BOARD_FLASH_ID_MT35XU512ABA1G12,
                                  BOARD_OSPI_INSTANCE, NULL);
    if (!gOspiHandle)
    {
        printf("\n Board_flashOpen Failed. \n");
        return -1;
    }

    /* Enable PHY pipeline mode  even though DMA is not used */
    CSL_ospiPipelinePhyEnable((const CSL_ospi_flash_cfgRegs *)(ospi_cfg.baseAddr), TRUE);

    /* Reset Variables */
    offset = *(uint32_t *)OFFSET_ADDR;
    orig_fileSize = fileSize = *(uint32_t *)SIZE_ADDR;
    srcAddr = (uint8_t *)LOAD_ADDR;
    checkAddr = (uint8_t *)CHECK_ADDR;

    flashInfo = (Board_FlashInfo *)gOspiHandle;

    fileSize = ALIGN(fileSize, flashInfo->page_size);

    if (ospiFlashWrite(srcAddr, offset, fileSize))
    {
        printf ("Flash Write Failed\n");
        Board_flashClose(gOspiHandle);
        return -1;
    }

    if (ospiFlashRead(checkAddr, offset, fileSize))
    {
        printf ("Flash Read Failed\n");
        Board_flashClose(gOspiHandle);
        return -1;
    }

    if (memcmp(checkAddr, srcAddr, orig_fileSize) != 0)
    {
        printf ("memcmp failed \n");
        return -1;
    }

    memset((void *) LOAD_ADDR, (int32_t) 0U, fileSize);

    Board_flashClose(gOspiHandle);

    return 0;
}

int8_t UFP_uartConfig(uint32_t baudrate)
{
    /* dummy function */

    return 0;
}

/**
 *  \brief      This function generates delay in msec.
 *
 *  \param      delay_val   [IN]    delay value
 *
 */
void delay(uint16_t delay_val)
{
    uint32_t delayVal;
    delayVal = delay_val * 1000;
    while(delayVal--);
}

