;
; Copyright (c) 2021, Texas Instruments Incorporated
; All rights reserved.
;
;  Redistribution and use in source and binary forms, with or without
;  modification, are permitted provided that the following conditions
;  are met:
;
;  *  Redistributions of source code must retain the above copyright
;     notice, this list of conditions and the following disclaimer.
;
;  *  Redistributions in binary form must reproduce the above copyright
;     notice, this list of conditions and the following disclaimer in the
;     documentation and/or other materials provided with the distribution.
;
;  *  Neither the name of Texas Instruments Incorporated nor the names of
;     its contributors may be used to endorse or promote products derived
;     from this software without specific prior written permission.
;
;  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
;  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
;  THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;  PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
;  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
;  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
;  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
;  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
;  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
;  EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
;

        .cdecls C,NOLIST
%{
        #include "icssg_sddf.h"
%}
        .include "sddf_cfg.h"
        .include "sddf.h"
        .include "sddf_macros.h"
        .include "sddf_dbg.h"

;
; Compile-time Host event for SDDF samples available, pr0_pru_mst_intr[2]_intr_req
;;TRIGGER_HOST_SDDF_EVT           .set 18 ; 2+16
; R31 event interface mapping, add pru<n>_r31_vec_valid to system event number
TRIGGER_HOST_SDDF_IRQ   .set TRIGGER_HOST_SDDF_EVT + 16 ; 2+16+16 = 2+32 = 2 + 1<<5

; SPAD Bank for SD Ch context storage
BANK_CTXT               .set BANK0
; T0 context located in BANK locations 0-17
NUM_REGS_T0_CTXT        .set 18 ; Number of PRU registers for T0 context
BANK_SHIFT_T0_CTXT      .set 30 ; XIN/XOUT bank shift for T0 context
; T1_S0 state located in BANK locations 20-28
NUM_REGS_T1_S0_STATE    .set 9  ; Number of PRU registers for T1_S0 state
BANK_SHIFT_T1_S0_STATE  .set 11  ; XIN/XOUT bank shift for T1_S0 state

OUT_SAMP_MASK           .set 0x0FFFFFFF ; 28-bit mask applied to Integrator & Differentiator output
NUM_OUT_SAMP_HOST_INT   .set 3          ; triggered mode, number of SDDF output samples after trigger event to send interrupt to Host

;NC_SAMP_CNT .set 3 ; FL: download?
NC_SAMP_CNT .set 4
;NC_SAMP_CNT .set 6

        ; local interleaved NC output sample buffer
OUT_SAMP_BUF:   .usect  ".outSamps", ICSSG_NUM_SD_CH_FW, 4
        .retain         ".outSamps"
        .retainrefs     ".outSamps"
        
        .def    SDDF_ENTRY  ; global entry point
        .ref    dbg_setup_pinmux
        
        .sect   ".text"
        .retain ".text"
        .retainrefs ".text"
SDDF_ENTRY:
        ; Clear registers R0-R30
        ZERO    &R0, 124
        
        ; Disable Task Manager
        .word 0x32000000

        ; Clear Task Manager status which is sticky after debug halt
        LDI     TREG0.w0, 0x0fff
        SBCO    &TREG0.w0, CT_PRU_ICSSG_TM, 0, 2
        XIN     TM_YIELD_XID, &R0.b3,1
        LDI     TREG0.w0, 0
        SBCO    &TREG0.w0, CT_PRU_ICSSG_TM, 0, 2

        ; De-initialize IEP0
        JAL     RET_ADDR_REG, iep0_deinit

        ; Write C24 block index for access to FW registers
        WRITE_C24_BLK_INDEX C24_BLK_INDEX_FW_REGS_VAL

;
; Check PRU ID & set PRU ID acknowledge
;
check_pru_id:
        ; Wait for Host to update PRU ID
        LBCO    &TREG0.b0, CT_PRU_ICSSG_LOC_DMEM, FW_REG_SDDF_CTRL, FW_REG_SDDF_CTRL_SZ ; Load TR0.b0 <- FW_REG_SDDF_CTRL 
        AND     TREG0.b0, TREG0.b0, SDDF_CTRL_BF_PRU_ID_MASK                            ; Mask PRU_ID bit field in TR0.b0
        LDI     TREG1.b0, BF_PRU_ID_UNINIT<<SDDF_CTRL_BF_PRU_ID_SHIFT                   ; Load TR1.b0 <- PRU_ID_UNINIT at PRU_ID bit field location
        SUB     TREG2.b0, TREG0.b0, TREG1.b0                                            ; TR2.b0 = TR1.b0 - TR0.b0
        QBEQ    check_pru_id, TREG2.b0, 0                                               ; TR2.b0 == 0 indicates PRU_ID is uninitialized, 
                                                                                        ; TR2.b0 != 0 indicates PRU_ID is initialized
        ; Set PRU ID acknowledge
        LBCO    &TREG1.b0, CT_PRU_ICSSG_LOC_DMEM, FW_REG_SDDF_STAT, FW_REG_SDDF_STAT_SZ ; Load TR1.b0 <- FW_REG_SDDF_STAT 
        AND     TREG2.b0, TREG1.b0, SDDF_STAT_BF_PRU_ID_ACK_MASK                        ; Mask PRU_ID_ACK bit field in TR1.b0
        QBEQ    check_sddf_en, TREG0.b0, TREG2.b0                                       ; TR0.b0 == TR2.b0 indicates PRU_ID and PRU_ID_ACK bit fields are equal
        LDI     TREG2.b0, SDDF_STAT_BF_PRU_ID_ACK_MASK                                  ; Load TR2.b0 <- PRU_ID_ACK mask
        NOT     TREG2.b0, TREG2.b0                                                      ; TR2.b0 = ~(PRU_ID_ACK mask)
        AND     TREG1.b0, TREG1.b0, TREG2.b0                                            ; Clear PRU_ID_ACK bit field
        OR      TREG1.b0, TREG1.b0, TREG0.b0                                            ; Insert PRU_ID into PRU_ID_ACK bit field
        SBCO    &TREG1.b0, CT_PRU_ICSSG_LOC_DMEM, FW_REG_SDDF_STAT, FW_REG_SDDF_STAT_SZ ; Store TR1.b0 -> FW_REG_SDDF_STAT
       
;
; Check SDDF global enable & set SDDF global enable acknowledge.
; If SDDF global enable not set, allow re-selection of PRU ID.
;
check_sddf_en:
        ; Check SDDF global enable
        LBCO    &TREG0.b0, CT_PRU_ICSSG_LOC_DMEM, FW_REG_SDDF_CTRL, FW_REG_SDDF_CTRL_SZ ; Load TR0.b0 <- FW_REG_SDDF_CTRL 
        QBBC    check_pru_id, TREG0.b0, SDDF_CTRL_BF_SDDF_EN_SHIFT                      ; If SDDF_EN not set, re-check PRU_ID

        ; Set SDDF global enable acknowledge
        LBCO    &TREG0.b0, CT_PRU_ICSSG_LOC_DMEM, FW_REG_SDDF_STAT, FW_REG_SDDF_STAT_SZ ; Load TR0.b0 <- FW_REG_SDDF_STAT
        SET     TREG0, TREG0, SDDF_CTRL_BF_SDDF_EN_SHIFT                                ; Set SDDF_EN_ACK bit
        SBCO    &TREG0.b0, CT_PRU_ICSSG_LOC_DMEM, FW_REG_SDDF_STAT, FW_REG_SDDF_STAT_SZ ; Store FW_REG_SDDF_STAT -> TR0.b0

;
; Perform initialization
;
init_sddf:
        ; Enable XIN/XOUT shifting.
        ; Used for context & SD state save/restore in TM tasks.
        LBCO    &TREG0.b0, CT_PRU_ICSSG_CFG, ICSSG_CFG_SPPC, 1
        SET     TREG0, TREG0, XFR_SHIFT_EN_BN   ; ICSSG_SPP_REG:XFR_SHIFT_EN=1
        SBCO    &TREG0.b0, CT_PRU_ICSSG_CFG, ICSSG_CFG_SPPC, 1
        
        ; Initialize Task Manager
        JAL     RET_ADDR_REG, tm_init
        
        ; Enable Task Manager
        .word   0x32800000
        
        ; Initialize IEP0
        JAL     RET_ADDR_REG, iep0_init

    .if !$isdefed("_DBG_PRU_GPO")
        ; Initialize SD mode
        LDI32   TREG1, PR1_PRUn_GP_MUX_SEL_VAL<<PR1_PRUn_GP_MUX_SEL_SHIFT
        LBCO    &TREG0.b0, CT_PRU_ICSSG_LOC_DMEM, FW_REG_SDDF_CTRL, FW_REG_SDDF_CTRL_SZ ; Load TR0.b0 <- FW_REG_SDDF_CTRL
        QBBS    init_pru_id1, TREG0, SDDF_CTRL_BF_PRU_ID_SHIFT                          ; Check PRU ID 0 or 1
init_pru_id0:
        SBCO    &TREG1, CT_PRU_ICSSG_CFG, ICSSG_CFG_GPCFG0, 4                           ; Initialize PRU0 SD mode
        QBA     init_sddf_cont
init_pru_id1:
        SBCO    &TREG1, CT_PRU_ICSSG_CFG, ICSSG_CFG_GPCFG1, 4                           ; Initialize PRU1 SD mode
    .endif

init_sddf_cont:
    .if $isdefed("_DBG_PRU_GPO")
        ; debug, setup pinmux
        JAL     RET_ADDR_REG, dbg_setup_pinmux
    .endif

        ; Reset SDDF state
        JAL     RET_ADDR_REG, reset_sddf_state

        ; Set base pointer to FW Configuration registers
        LDI     SDDF_CFG_BASE_PTR_REG, FW_REG_SDDF_CFG_IEP_CFG
        ; Set base point to SD HW registers
        SET_SD_HW_REG_BASE_PTR SD_HW_BASE_PTR_REG 

        ; Initialize SD clock
        JAL     RET_ADDR_REG, init_sd_clock
        
        ; Configure OSR for SD channels
        JAL     RET_ADDR_REG, config_oc_osr
    
        ; Configure SD channels. For all channels, initialize:
        ;   ACC select: ACC3
        ;   Clock inversion
        ;   Clock source: pr1_pru<n>_pru_r31_in[16] Primary Input
        JAL     RET_ADDR_REG, config_sd_ch
    
        ; Global enable SD HW,
        ; reset SD channel HW
        JAL     RET_ADDR_REG, reset_sd_ch_hw
        SET     R30.t25 ; R30[25] channel_en = 1, all channels enabled

        ; Initialize dedicated registers:
        ;   MASK register, 
        ;   OC positive & negative thresholds, 
        ;   Local NC output sample buffer address, 
        ;   Clear NC sample count.
        LDI32   MASK_REG, OUT_SAMP_MASK
        LBBO    &OC_POS_THR, SDDF_CFG_BASE_PTR_REG, FW_REG_SDDF_CFG_OC_POS_THR_OFFSET, FW_REG_SDDF_CFG_OC_POS_THR_SZ
        LBBO    &OC_NEG_THR, SDDF_CFG_BASE_PTR_REG, FW_REG_SDDF_CFG_OC_NEG_THR_OFFSET, FW_REG_SDDF_CFG_OC_NEG_THR_SZ
        LDI32   OUT_SAMP_BUF_REG, OUT_SAMP_BUF
        LDI     SAMP_CNT_REG, 0

    .if $isdefed("_DBG_CB_SHMEM")
        ; Debug, Shared Memory capture buffer
        LDI32   DBG_BUF_ADDR, 0x10000 ; init shared memory pointer
        LDI     DBG_BUF_IDX, 0  ; init debug buffer index
    .endif

    .if $isdefed("_DBG_SOC_GPIO")
        ; debug GPIO, GPIO0_62 & GPIO0_52 outputs
        LDI32   GPIO0_REG_ADDR, 0x00600038 ; debug, GPIO0 GPIO_DIR23
        LBBO    &GPIO0_REG_VAL, GPIO0_REG_ADDR, 0, 4
        CLR     GPIO0_REG_VAL.t30 ; GPIO0_62 30+32=62
        CLR     GPIO0_REG_VAL.t20 ; GPIO0_52 20+32=52
        SBBO    &GPIO0_REG_VAL, GPIO0_REG_ADDR, 0, 4
        LDI     GPIO_TGL, 0
    .endif

    .if $isdefed("_DBG_OC_BM_PRU_CYCLE") | $isdefed("_DBG_NC_BM_PRU_CYCLE")
        ; Enable PRU counter
        LBCO    &TREG0, CT_PRU_ICSSG_CTRL, 0, 1
        SET     TREG0.b0.t3
        SBCO    &TREG0, CT_PRU_ICSSG_CTRL, 0, 1
    .endif

        ; Start IEP
        LBCO    &TREG0.b0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_GLOBAL_CFG_REG, 1
        SET     TREG0, TREG0, CNT_ENABLE_BN ; ICSSG_IEP_GLOBAL_CFG_REG:CNT_ENABLE=1
        SBCO    &TREG0.b0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_GLOBAL_CFG_REG, 1   

        ; Generate OC output samples
        ;
        ; TR0.b0: ch ID
        ; TR0.b1: channel ID FW register shift amount
        ; TR0.w2: channel ID FW register

        ; Load TR0.w1 <- SDDF_CFG_SD_CH_ID
        ;LBBO    &TREG0.w2, SDDF_CFG_BASE_PTR_REG, FW_REG_SDDF_CFG_SD_CH_ID_OFFSET, FW_REG_SDDF_CFG_SD_CH_ID_SZ
        LBBO    &SD_CH_ID, SDDF_CFG_BASE_PTR_REG, FW_REG_SDDF_CFG_SD_CH_ID_OFFSET, FW_REG_SDDF_CFG_SD_CH_ID_SZ
        LDI     TREG0.b1, 0 ; init Ch ID shift
        LDI     TREG1.w2, 0 ; init OC detect count

ts0_oc_loop:
    .if 0; $isdefed("_DBG_SOC_GPIO") ; drive GPIO0_52 high at OC start
        ; debug GPIO, GPIO0_52=1
        LDI32   GPIO0_REG_ADDR, 0x00600040 ; debug, GPIO0 GPIO_SET_DATA23
        LDI     GPIO0_REG_VAL, 0
        SET     GPIO0_REG_VAL.t20 ; GPIO0_52 20+32=52
        SBBO    &GPIO0_REG_VAL, GPIO0_REG_ADDR, 0, 4
    .endif

    .if $isdefed("_DBG_SOC_GPIO") ; toggle GPIO00_52 at OC start
        XOR     GPIO_TGL, GPIO_TGL, 1
        QBEQ    gpio_low, GPIO_TGL, 0
gpio_high:        
        ; debug GPIO, GPIO0_52=1
        LDI32   GPIO0_REG_ADDR, 0x00600040 ; debug, GPIO0 GPIO_SET_DATA23
        LDI     GPIO0_REG_VAL, 0
        SET     GPIO0_REG_VAL.t20 ; GPIO0_52 20+32=52
        SBBO    &GPIO0_REG_VAL, GPIO0_REG_ADDR, 0, 4
        QBA     gpio_cont
gpio_low:        
        ; debug GPIO, GPIO0_52=0
        LDI32   GPIO0_REG_ADDR, 0x00600044 ; debug, GPIO0 GPIO_CLR_DATA23
        LDI     GPIO0_REG_VAL, 0
        SET     GPIO0_REG_VAL.t20 ; GPIO0_52 20+32=52
        SBBO    &GPIO0_REG_VAL, GPIO0_REG_ADDR, 0, 4
gpio_cont:        
    .endif

    .if $isdefed("_DBG_NO_OC_POLL_SHDW_UPD")
        LDI     R30.w2, (SD_CH0_ID<<10 | 1<<9)
        NOP
wait_for_shadow_update_cont:        
        QBBC    wait_for_shadow_update_cont, R31, 28
    .endif

    .if $isdefed("_DBG_PRU_GPO_PROXY")
        SET     R30_PROXY.b0, R30_PROXY.b0, 1 ; PRU1_GPO1=1
        LDI     R0.b0, 0 ; set XIN/XOUT shift
        XOUT    BANK1, &R30_PROXY, 4
    .endif

    .if $isdefed("_DBG_OC_BM_PRU_CYCLE")
        ; Read ICSSG_PRU_CYCLE
        LBCO    &BM_PRU_CYCLE_START, CT_PRU_ICSSG_CTRL, 0xC, 4
    .endif

        ; Switch to SD HW to Ch X, 
        ; wait for shadow update flag, 
        ; clear shadow update flag
        ;LSR     TREG0.b0, TREG0.w2, TREG0.b1
        LSR     TREG0.b0, SD_CH_ID, TREG0.b1
        AND     TREG0.b0, TREG0.b0, BF_SD_CH0_ID_MASK
        M_WAIT_SHADOW_FLAG_AND_CLR TREG0.b0
        ; Load reg w/ SD HW ACC3 output sample
        AND     DN0, R31, MASK_REG
        ; Execute Sinc3 filtering
        M_ACC3_PROCESS ACC3_DN1_CH0, ACC3_DN3_CH0, ACC3_DN5_CH0
        ADD     TREG0.b1, TREG0.b1, 4 ; update Ch ID shift
        MOV     TREG2, CN5  ; save SD ch output for min,max OC SD sample calc

    .if $isdefed("_DBG_OC_OUT_SAMP_CB_SHMEM")
        ; Debug, Shared Memory capture buffer
        ; Capture ChX OC output samples
        LDI     TREG0.w2, DBG_BUF_SZ
        SUB     TREG0.w2, TREG0.w2, DBG_BUF_IDX
        QBEQ    dbg_no_cap, TREG0.w2, 0
        SBBO    &CN5, DBG_BUF_ADDR, DBG_BUF_IDX, 4
        ADD     DBG_BUF_IDX, DBG_BUF_IDX, 4
dbg_no_cap:        
    .endif

        ; Switch to SD HW to Ch Y, 
        ; wait for shadow update flag, 
        ; clear shadow update flag
        ;LSR     TREG0.b0, TREG0.w2, TREG0.b1
        LSR     TREG0.b0, SD_CH_ID, TREG0.b1
        AND     TREG0.b0, TREG0.b0, BF_SD_CH0_ID_MASK
        M_WAIT_SHADOW_FLAG_AND_CLR TREG0.b0
        ; Load reg w/ SD HW ACC3 output sample
        AND     DN0, R31, MASK_REG
        ; Execute Sinc3 filtering
        M_ACC3_PROCESS ACC3_DN1_CH1, ACC3_DN3_CH1, ACC3_DN5_CH1
        ADD     TREG0.b1, TREG0.b1, 4 ; update Ch ID shift
        MIN     TREG3, TREG2, CN5   ; compute min OC SD ch output
        MAX     TREG2, TREG2, CN5   ; compute max OC SD ch output

        ; Switch to SD HW to Ch Z, 
        ; wait for shadow update flag, 
        ; clear shadow update flag
        ;LSR     TREG0.b0, TREG0.w2, TREG0.b1
        LSR     TREG0.b0, SD_CH_ID, TREG0.b1
        AND     TREG0.b0, TREG0.b0, BF_SD_CH0_ID_MASK
        M_WAIT_SHADOW_FLAG_AND_CLR TREG0.b0
        ; Load reg w/ SD HW ACC3 output sample
        AND     DN0, R31, MASK_REG
        ; Execute Sinc3 filtering
        M_ACC3_PROCESS ACC3_DN1_CH2, ACC3_DN3_CH2, ACC3_DN5_CH2        
        LDI     TREG0.b1, 0  ; update Ch ID shift
        MIN     TREG3, TREG3, CN5   ; compute min OC SD ch output
        MAX     TREG2, TREG2, CN5   ; compute max OC SD ch output

        ; Check max SD channel output against OC positive threshold
        QBGE    over_current, OC_POS_THR, TREG2
        ; Check min SD channel output against OC negative threshold
        QBLE    over_current, OC_NEG_THR, TREG3
        QBA     skip_over_current
over_current:
        ; Currently track OC events in register.
        ; Update this to HW signal.
        ADD     TREG1.w2, TREG1.w2, 1
skip_over_current:

    .if 0; $isdefed("_DBG_SOC_GPIO") ; drive GPIO0_52 low at OC end
        ; debug GPIO, GPIO0_52=0
        LDI32   GPIO0_REG_ADDR, 0x00600044 ; debug, GPIO0 GPIO_CLR_DATA23
        LDI     GPIO0_REG_VAL, 0
        SET     GPIO0_REG_VAL.t20 ; GPIO0_52 20+32=52
        SBBO    &GPIO0_REG_VAL, GPIO0_REG_ADDR, 0, 4
    .endif

    .if $isdefed("_DBG_PRU_GPO_PROXY")
        CLR     R30_PROXY.b0, R30_PROXY.b0, 1 ; PRU1_GPO1=0
        LDI     R0.b0, 0 ; set XIN/XOUT shift
        XOUT    BANK1, &R30_PROXY, 4
    .endif

    .if $isdefed("_DBG_OC_BM_PRU_CYCLE")
        ; Read ICSSG_PRU_CYCLE
        LBCO    &BM_PRU_CYCLE_END, CT_PRU_ICSSG_CTRL, 0xC, 4

        ; Debug, Shared Memory capture buffer
        ; Capture time stamps
        LDI     TREG0.w2, DBG_BUF_SZ
        SUB     TREG0.w2, TREG0.w2, DBG_BUF_IDX
        QBEQ    dbg_no_cap, TREG0.w2, 0
        SBBO    &BM_PRU_CYCLE_START, DBG_BUF_ADDR, DBG_BUF_IDX, 2*4
        ADD     DBG_BUF_IDX, DBG_BUF_IDX, 2*4
dbg_no_cap:        
    .endif

        QBA     ts0_oc_loop

;
; Initialize Task Manager
;
tm_init:
; Configure Task Manager tasks
;
        ; TM general purpose mode
        ;   Enable T1_S0 : IEP0 CMP1 task
        ;   Enable T1_S1 : IEP0 CMP0 task (debug)
        ;LDI     TREG0.b0, (1b<<3 | 1b<<2| 11b<<0)   ; enable T1_S0, T1_S1
        LDI     TREG0.b0, (1b<<2| 11b<<0)           ; enable T1_S0
        ;LDI     TREG0.b0, 0                         ; disable all tasks
        SBCO    &TREG0.b0, CT_PRU_ICSSG_TM, 0, 2

        ; Set Task addresses
        ; set T1_S0 address
        LDI     TREG0.w0, $CODE(tm_t1_s0_nc)
        SBCO    &TREG0.w0, CT_PRU_ICSSG_TM, TASKS_MGR_TS1_PC_S0, 2
        ; set T1_S1 address
        LDI     TREG0.w0, $CODE(tm_t1_s1_iep0_cmp0)
        SBCO    &TREG0.w0, CT_PRU_ICSSG_TM, TASKS_MGR_TS1_PC_S1, 2

        ; Set Task triggers
        ; set T1_S0 trigger to IEP0 CMP1 event = 17
        ; set T1_S1 trigger to IEP0 CMP0 event = 16
        LDI     TREG0.w0, (16<<8 | 17<<0)
        SBCO    &TREG0.w0, CT_PRU_ICSSG_TM, TASKS_MGR_TS1_GEN_CFG1, 2

        JMP     RET_ADDR_REG


; TM T1_S0:
;   Triggered by IEP0 CMP1.
;   Normal Current (NC) sampling.
;
tm_t1_s0_nc:
    .if $isdefed("_DBG_PRU_GPO")
        SET     R30.b0, R30.b0, 1 ; GPU_GPO1=1
    .endif

    .if $isdefed("_DBG_PRU_GPO_PROXY")
        SET     R30_PROXY.b0, R30_PROXY.b0, 0 ; PRU1_GPO0=1
        LDI     R0.b0, 0 ; set XIN/XOUT shift
        XOUT    BANK1, &R30_PROXY, 4
    .endif

    .if $isdefed("_DBG_SOC_GPIO")
        ; Save T0 context
        LDI     R0.b0, 0 ; set XIN/XOUT shift
        XOUT    BANK1, &GPIO0_REG_ADDR, 4*3
        ; debug GPIO, GPIO0_62=1
        LDI32   GPIO0_REG_ADDR, 0x00600040 ; debug, GPIO0 GPIO_SET_DATA23
        LDI     GPIO0_REG_VAL, 0
        SET     GPIO0_REG_VAL.t30 ; GPIO0_62 30+32=62
        SBBO    &GPIO0_REG_VAL, GPIO0_REG_ADDR, 0, 4
    .endif

    .if $isdefed("_DBG_NC_BM_PRU_CYCLE")
        ; Save overwritten T0 context
        LDI     R0.b0, 0 ; set XIN/XOUT shift
        XOUT    BANK1, &TREG0, 4*1
        ; Read ICSSG_PRU_CYCLE
        LBCO    &BM_PRU_CYCLE_START, CT_PRU_ICSSG_CTRL, 0xC, 4
    .endif

        ; Save/restore context
        ;
        LDI     R0.b0, BANK_SHIFT_T0_CTXT       ; set XIN/XOUT shift
        MOV     R18.b0, R30.b3                  ; save T0 SD channel select
         ; save T0 context
        XOUT    BANK_CTXT, &T0_CTXT_BASE_REG, 4*NUM_REGS_T0_CTXT
        LDI     R0.b0, BANK_SHIFT_T1_S0_STATE   ; set XIN/XOUT shift
         ; restore T1_S0 state
        XIN     BANK_CTXT, &T1_S0_CTXT_BASE_REG, 4*NUM_REGS_T1_S0_STATE
        
    .if !$isdefed("_DBG_PRU_GPO")
        ; Clear IEP0 CMP1 events
        LDI		TREG0.b0, 0x02
        SBCO    &TREG0.b0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP_STATUS_REG, 1
        
        ; Set ChX SD snoop=1 & sample_counter_select=1
        ;LDI     TREG0.b2, (1b<<6 | 1b<<5)
        ;LSL     TREG0.b3, SD_CH_ID, 2
        ;AND     TREG0.b3, TREG0.b3, 1111b<<2
        ;SET     TREG0.b3.t1
        ;MOV     R30.w2, TREG0.w2
        LDI     R30.w2, (SD_CH0_ID<<10 | 1<<9 | 1<<6 | 1<<5)
        NOP
        
        ; Snoop read ChX sample_counter, 
        ; wait for ChX sample count+1.
        AND     TREG1, R31, 0xFF ; snoop read LSB ChX sample_counter
wait_sample_count_incr:
        AND     TREG2, R31, 0xFF ; snoop read LSB ChX sample_counter
        QBEQ    wait_sample_count_incr, TREG2, TREG1
        
        ; Snoop read ChX ACC3
        CLR     R30.t21                 ; set SD sample_counter_select=0
        NOP
        AND     DN0, R31, MASK_REG      ; DN0 = ChX SD HW ACC3 output sample
        CLR     R30.t22                 ; set SD snoop=0
        ;NOP
        
        ; Snoop read ACC3 for SD ChY
        ;LSR     TREG0.b3, SD_CH_ID, 2
        ;AND     TREG0.b3, TREG0.b3, 1111b<<2
        ;SET     TREG0.b3.t1
        ;MOV     R30.w2, TREG0.w2
        LDI     R30.w2, (SD_CH1_ID<<10 | 1<<9 | 1<<6)
        NOP
        AND     TREG1, R31, MASK_REG    ; TREG1 = ChY SD HW ACC3 output sample        
        CLR     R30.t22                 ; set SD snoop=0
        ;NOP
        
        ; Snoop read ACC3 for SD ChZ
        ;LSR     TREG0.b3, SD_CH_ID, 6
        ;AND     TREG0.b3, TREG0.b3, 1111b<<2
        ;SET     TREG0.b3.t1
        ;MOV     R30.w2, TREG0.w2
        LDI     R30.w2, (SD_CH2_ID<<10 | 1<<9 | 1<<6)
        NOP
        AND     TREG2, R31, MASK_REG    ; TREG1 = ChY SD HW ACC3 output sample
        CLR     R30.t22                 ; set SD snoop=0
        ;NOP

        ; Execute SINC3 differentiation for ChX
        M_ACC3_PROCESS ACC3_DN1_CH0, ACC3_DN3_CH0, ACC3_DN5_CH0
        ; Save NC output sample to local output sample buffer 
        SBBO    &CN5, OUT_SAMP_BUF_REG, 0, 4

        ; Execute SINC3 differentiation for ChY        
        MOV     DN0, TREG1 ; DN0 = ChY SD HW ACC3 output sample
        M_ACC3_PROCESS ACC3_DN1_CH1, ACC3_DN3_CH1, ACC3_DN5_CH1
        ; Save output sample to local output sample buffer 
        SBBO    &CN5, OUT_SAMP_BUF_REG, 4, 4

        ; Execute SINC3 differentiation for ChZ
        MOV     DN0, TREG2 ; DN0 = ChZ SD HW ACC3 output sample
        M_ACC3_PROCESS ACC3_DN1_CH2, ACC3_DN3_CH2, ACC3_DN5_CH2
        ; Save output sample to local output sample buffer 
        SBBO    &CN5, OUT_SAMP_BUF_REG, 8, 4   
    .endif

        QBLE    reset_nc_frame, SAMP_CNT_REG, NC_SAMP_CNT-1
        ; NC sample count < NC_SAMP_CNT-1
        ; update IEP0 CMP1
        LDI     TREG0,0
        LBCO    &TREG0, CT_PRU_ICSSG_LOC_DMEM, FW_REG_SDDF_CFG_NC_PRD_IEP_CNT, FW_REG_SDDF_CFG_NC_PRD_IEP_CNT_SZ
        LBCO    &TREG1, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP1_REG0, 4 ; FL: fix me
        ADD     TREG0, TREG1, TREG0
        LDI     TREG1, 6000
        QBLT    no_rollover, TREG1, TREG0
        SUB     TREG0, TREG0, TREG1
no_rollover: 
        SBCO    &TREG0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP1_REG0, 4
        
        ADD     SAMP_CNT_REG, SAMP_CNT_REG, 1   ; increment NC sample count
        
        QBA     nreset_nc_frame
reset_nc_frame:
        ; NC sample count >= NC_SAMP_CNT-1
    .if !$isdefed("_DBG_PRU_GPO")
        ; Write local interleaved output samples to Host buffer address
        LBCO    &TREG0, CT_PRU_ICSSG_LOC_DMEM, FW_REG_SDDF_CFG_OUT_SAMP_BUF, 4 ; Load TR1 <- SDDF_CFG_OUT_SAMP_BUF
        LBBO    &TREG1, OUT_SAMP_BUF_REG, 0, ICSSG_NUM_SD_CH_FW*4
        SBBO    &TREG1, TREG0, 0, ICSSG_NUM_SD_CH_FW*4
        ; Trigger interrupt
        LDI     R31.w0, TRIGGER_HOST_SDDF_IRQ
    .endif
        
        ; JR: Read CMP1 value and figure out which trigger to write (first set or second set)
        LDI     TREG0, 4439
        LBCO    &TREG1, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP1_REG0, 4
        QBEQ    second_set, TREG0, TREG1
        LDI     TREG0,1560
        QBA     ready_to_write
second_set
        LDI     TREG0,4560
ready_to_write:
        SUB     TREG0, TREG0, IEP_DEFAULT_INC ; subtract IEP default increment since IEP counts 0...CMP0
        SBCO    &TREG0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP1_REG0, 4

        ; Set IEP CMP1 value: IEP_CMP1_REG1:REG0 = 0:TRIG_SAMPLE_TIME
        ;LDI     TREG0,0
        ;LBCO    &TREG0, CT_PRU_ICSSG_LOC_DMEM, FW_REG_SDDF_CFG_TRIG_SAMPLE_TIME, FW_REG_SDDF_CFG_TRIG_SAMPLE_TIME_SZ
        ;SUB     TREG0, TREG0, IEP_DEFAULT_INC ; subtract IEP default increment since IEP counts 0...CMP0
        ;SBCO    &TREG0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP1_REG0, 4

        LDI     SAMP_CNT_REG, 0 ; reset NC sample count
nreset_nc_frame:

        ; Save/restore context
        ;
        LDI     R0.b0, BANK_SHIFT_T1_S0_STATE   ; set XIN/XOUT shift
        ; save T1_S0 state
        XOUT    BANK_CTXT, &T1_S0_CTXT_BASE_REG, 4*NUM_REGS_T1_S0_STATE
        LDI     R0.b0, BANK_SHIFT_T0_CTXT       ; set XIN/XOUT shift
        ; restore T0 context
        XIN     BANK_CTXT, &T0_CTXT_BASE_REG, 4*NUM_REGS_T0_CTXT
        MOV     R30.b3, R18.b0                  ; restore T0 SD channel select
        ;NOP

    .if $isdefed("_DBG_NC_BM_PRU_CYCLE")
        ; Read ICSSG_PRU_CYCLE
        LBCO    &BM_PRU_CYCLE_END, CT_PRU_ICSSG_CTRL, 0xC, 4
        
        ; Debug, Shared Memory capture buffer
        ; Capture time stamps
        LDI     TREG0.w2, DBG_BUF_SZ
        SUB     TREG0.w2, TREG0.w2, DBG_BUF_IDX
        QBEQ    dbg_no_cap, TREG0.w2, 0
        SBBO    &BM_PRU_CYCLE_START, DBG_BUF_ADDR, DBG_BUF_IDX, 2*4
        ADD     DBG_BUF_IDX, DBG_BUF_IDX, 2*4
        ; Restore overwritten T0 context
        LDI     R0.b0, 0 ; set XIN/XOUT shift
        XIN     BANK1, &TREG0, 4*1
dbg_no_cap:                
    .endif

    .if $isdefed("_DBG_SOC_GPIO")
        ; debug GPIO, GPIO0_62=0
        LDI32   GPIO0_REG_ADDR, 0x00600044 ; debug, GPIO0 GPIO_CLR_DATA23
        LDI     GPIO0_REG_VAL, 0
        SET     GPIO0_REG_VAL.t30 ; GPIO0_62 30+32=62
        SBBO    &GPIO0_REG_VAL, GPIO0_REG_ADDR, 0, 4
        ; Restore T0 context
        LDI     R0.b0, 0 ; set XIN/XOUT shift
        XIN     BANK1, &GPIO0_REG_ADDR, 4*3       
    .endif

    .if $isdefed("_DBG_PRU_GPO_PROXY")
        CLR     R30_PROXY.b0, R30_PROXY.b0, 0 ; PRU1_GPO0=0
        LDI     R0.b0, 0 ; set XIN/XOUT shift
        XOUT    BANK1, &R30_PROXY, 4
    .endif

    .if $isdefed("_DBG_PRU_GPO")
        CLR     R30.b0, R30.b0, 1 ; GPU_GPO1=0
    .endif

        XIN     TM_YIELD_XID, &R0.b3,1  ; exit task after two instructions/cycles
        NOP
        NOP

        
; TM T1_S1 (debug):
;   Triggered by IEP0 CMP0.
;   
tm_t1_s1_iep0_cmp0:
        SET     R30.b0, R30.b0, 0 ; GPU_GPO0=1

        ; Clear IEP0 CMP0 events
        LDI		TREG0.b0, 0x01
        SBCO    &TREG0.b0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP_STATUS_REG, 1

        NOP
        NOP
        NOP
        NOP
        CLR     R30.b0, R30.b0, 0 ; GPU_GPO0=0
        
        XIN     TM_YIELD_XID, &R0.b3,1  ; exit task after two instructions/cycles
        NOP
        NOP
    

; De-initialize IEP0.
;   
iep0_deinit:
        ; Disable IEP0 counter
        LBCO    &TREG0.b0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_GLOBAL_CFG_REG, 1
        AND     TREG0.b0, TREG0.b0, 0xFE
        SBCO    &TREG0.b0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_GLOBAL_CFG_REG, 1

        ; Set IEP0 counter to zero
        LDI     TREG0, 0
        SBCO    &TREG0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_COUNT_REG1, 4
        SBCO    &TREG0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_COUNT_REG0, 4

        ; Clear IEP0 CMP0 & CMP1 events
        LDI		TREG0.b0, 0x03
        SBCO    &TREG0.b0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP_STATUS_REG, 1
       
        JMP     RET_ADDR_REG


; Initialize IEP0.
;   
iep0_init:
        ; Disable IEP0 counter
        LBCO    &TREG0.b0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_GLOBAL_CFG_REG, 1
        AND     TREG0.b0, TREG0.b0, 0xFE
        SBCO    &TREG0.b0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_GLOBAL_CFG_REG, 1

        ; Set IEP0 counter to zero
        LDI     TREG0, 0
        SBCO    &TREG0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_COUNT_REG1, 4
        SBCO    &TREG0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_COUNT_REG0, 4

        ; Clear IEP0 CMP0 & CMP1 & CMP2 events
        LDI		TREG0.b0, 0x07
        SBCO    &TREG0.b0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP_STATUS_REG, 1
       
        ; Write IEP0 default increment
        LBCO    &TREG0.b0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_GLOBAL_CFG_REG, 1
        AND     TREG0.b0, TREG0.b0, 0x0F
        OR      TREG0.b0, TREG0.b0, IEP_DEFAULT_INC<<DEFAULT_INC_BN
        SBCO    &TREG0.b0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_GLOBAL_CFG_REG, 1
        
    .if !$isdefed("_DBG_TRIG_SIM")
        LBCO    &TREG0.b0, CT_PRU_ICSSG_IEP0_0x100, ICSSG_IEP_PWM_REG, 1
        SET     TREG0.b0.t0 ; IEP_PWM_REG:PWM0_RST_CNT_EN = 1, enable IEP0 counter reset on EPWM0 SYNCO event
        SBCO    &TREG0.b0, CT_PRU_ICSSG_IEP0_0x100, ICSSG_IEP_PWM_REG, 1
    .else
        ; Debug, set IEP0 CMP0 to simulate EPWM (FOC) loop period
        ; Set CMP0 compare value
        LDI     TREG0, 0
        SBCO    &TREG0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP0_REG1, 4
        LDI32   TREG0, CMP0_CNT_EPWM_PRD
        SUB     TREG0, TREG0, IEP_DEFAULT_INC ; subtract IEP default increment since IEP counts 0...CMP0
        SBCO    &TREG0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP0_REG0, 4
        ; Enable reset of IEP0 counter on CMP0 event, enable CMP0 event
        LBCO    &TREG0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP_CFG_REG, 4
        SET     TREG0.t0    ; CMP0_RST_CNT_EN=1, Enable the reset of the counter if a CMP0 event occurs
        SET     TREG0.t1    ; CMP_EN[0]=1 => CMP0 enabled
        SBCO    &TREG0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP_CFG_REG, 4
    .endif

        ; Initialize Trigger sample time
        ; Set IEP0 CMP1 value: IEP0_CMP1_REG1:REG0 = 0:TRIG_SAMPLE_TIME
        LDI     TREG0, 0
        SBCO    &TREG0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP1_REG1, 4
        LBCO    &TREG0, CT_PRU_ICSSG_LOC_DMEM, FW_REG_SDDF_CFG_TRIG_SAMPLE_TIME, FW_REG_SDDF_CFG_TRIG_SAMPLE_TIME_SZ
        LDI     TREG1, SDDF_CFG_TRIG_SAMP_TIME_BF_TRIG_SAMP_TIME_MASK
        AND     TREG0, TREG1, TREG0
        SUB     TREG0, TREG0, IEP_DEFAULT_INC ; subtract IEP default increment since IEP counts 0...CMP0
        SBCO    &TREG0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP1_REG0, 4
        ; Enable IEP0 CMP1
        LBCO    &TREG0.b0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP_CFG_REG, 1  ; TR0 <- Byte0 ICSSG_CMP_CFG_REG
        SET     TREG0.t2    ; CMP_EN[1]=1 => CMP1 enabled
        SBCO    &TREG0.b0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP_CFG_REG, 1  ; TR0 -> ICSSG_CMP_CFG_REG Byte0

        ; Set IEP0 CMP2 value: IEP0_CMP2_REG1:REG0 = 0:2999 (halfway point for 50KHz = 10us -> 3.33ns * 3000 = 10us)
        ; Set IEP0 CMP2 value: IEP0_CMP2_REG1:REG0 = 0:7499 (halfway point for 20KHz = 25us -> 3.33ns * 7500 = 25us)
        ; Experimentally measured 340ns delay (102 @ 3.33ns) between desired trigger and first EnDat CLK low
        LDI     TREG0, 0
        SBCO    &TREG0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP2_REG1, 4
        LDI     TREG0, (3000-102)
        SUB     TREG0, TREG0, IEP_DEFAULT_INC ; subtract IEP default increment since IEP counts 0...CMP0
        SBCO    &TREG0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP2_REG0, 4
        ; Enable IEP0 CMP2
        LBCO    &TREG0.b0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP_CFG_REG, 1  ; TR0 <- Byte0 ICSSG_CMP_CFG_REG
        SET     TREG0.t3    ; CMP_EN[2]=1 => CMP2 enabled
        SBCO    &TREG0.b0, CT_PRU_ICSSG_IEP0, ICSSG_IEP_CMP_CFG_REG, 1  ; TR0 -> ICSSG_CMP_CFG_REG Byte0

        JMP     RET_ADDR_REG


;
; Reset SDDF state
;
; Arguments: None
;
reset_sddf_state:
        ZERO    &T0_CTXT_BASE_REG, 4*NUM_REGS_T0_CTXT
        LDI     R0.b0, BANK_SHIFT_T0_CTXT ; set XIN/XOUT shift
        XOUT    BANK_CTXT, &T0_CTXT_BASE_REG, 4*NUM_REGS_T0_CTXT ; clear TS0 context

        ; T1_S0 state located in BANK locations 20-28
        ZERO    &T1_S0_CTXT_BASE_REG, 4*NUM_REGS_T1_S0_STATE
        LDI     R0.b0, BANK_SHIFT_T1_S0_STATE ; set XIN/XOUT shift
        XOUT    BANK_CTXT, &T1_S0_CTXT_BASE_REG, 4*NUM_REGS_T1_S0_STATE ; clear T1_S0 state

        JMP     RET_ADDR_REG


;
; Configure OC Over Sampling Rate for SD channels
;
; Arguments: 
;   SDDF_CFG_BASE_PTR_REG: base address of SD Configuration registers (&IEP_CFG_EPWM_PRD)
;   SD_HW_BASE_PTR_REG: base address of SD HW configuration registers (&ICSSG_PRUn_SD_CLK_SEL_REG0)
;
config_oc_osr:
        ; Load TR0.b0 <- FW_REG_SDDF_CFG_OSR
        LBBO    &TREG0.b0, SDDF_CFG_BASE_PTR_REG, FW_REG_SDDF_CFG_OSR_OFFSET, FW_REG_SDDF_CFG_OSR_SZ
        ; Load TR1.w0 <- SDDF_CFG_SD_CH_ID
        LBBO    &TREG1.w0, SDDF_CFG_BASE_PTR_REG, FW_REG_SDDF_CFG_SD_CH_ID_OFFSET, FW_REG_SDDF_CFG_SD_CH_ID_SZ
                
        LOOP    config_osr_loop_end, ICSSG_NUM_SD_CH_FW ; loop over SD channels
        AND     TREG2.b0, TREG1.b0, BF_SD_CH0_ID_MASK   ; LS byte is ID for Chi
        LSL     TREG2, TREG2, 3                         ; (ChID*8) bytes to Byte0 PRUn_SD_SAMPLE_SIZE_REG(ChID)
        ADD     TREG2, TREG2, 4                         ; 4 bytes added for Byte0 PRUn_SD_SAMPLE_SIZE_REG0
        
        SBBO    &TREG0.b0, SD_HW_BASE_PTR_REG, TREG2, 1 ; TR0.b0 -> PRUn_SD_SAMPLE_SIZE_REG(ChID)
        LSR     TREG1, TREG1, 4                         ; LS byte is ID for Ch(i+1) ; FL fix me
config_osr_loop_end:
        
        JMP     RET_ADDR_REG


;
; Configure SD channels. For SD channels, initialize:
;   ACC select = ACC3
;   Clock inversion
;   Clock source: pr1_pru<n>_pru_r31_in[16] Primary Input
;
;   SDDF_CFG_BASE_PTR_REG: base address of SD Configuration registers (&IEP_CFG_EPWM_PRD)
;   SD_HW_BASE_PTR_REG: base address of SD HW configuration registers (&ICSSG_PRUn_SD_CLK_SEL_REG0)
;
config_sd_ch:
        ; Load TR0.w0 <- FW_REG_SDDF_CFG_SD_CLK
        LBBO    &TREG0.w0, SDDF_CFG_BASE_PTR_REG, FW_REG_SDDF_CFG_SD_CLK_OFFSET, FW_REG_SDDF_CFG_SD_CLK_SZ
        ; Load TR1.w0 <- SDDF_CFG_SD_CH_ID
        LBBO    &TREG1.w0, SDDF_CFG_BASE_PTR_REG, FW_REG_SDDF_CFG_SD_CH_ID_OFFSET, FW_REG_SDDF_CFG_SD_CH_ID_SZ

        ; Select clock inversion
        LSR     TREG0.w0, TREG0.w0, SDDF_CFG_SD_CLK_BF_SD_CLK_INV_SHIFT-PRUn_SD_CLK_INVi_SHIFT  ; place SD_CLK_INV in HW reg bit field location
        AND     TREG0.b0, TREG0.b0, PRUn_SD_CLK_INVi_MASK<<PRUn_SD_CLK_INVi_SHIFT   ; TR1.b0 = SD_CLK_INV
        
        OR      TREG0.b0, TREG0.b0, PRUn_SD_CLK_SELi_VAL<<PRUn_SD_CLK_SELi_SHIFT    ; select clock source
        OR      TREG0.b0, TREG0.b0, PRUn_SD_ACC_SELi_VAL<<PRUn_SD_ACC_SELi_SHIFT    ; select to ACC source
        
        LOOP    config_sd_ch_loop_end, ICSSG_NUM_SD_CH_FW   ; loop over SD channels
        AND     TREG2.b0, TREG1.b0, BF_SD_CH0_ID_MASK       ; LS byte is ID for Chi
        LSL     TREG2, TREG2, 3                             ; (ChID*8) bytes to Byte0 PRUn_SD_CLK_SEL_REG(ChID)
        ADD     TREG2, TREG2, 0                             ; 0 bytes to Byte0 PRUn_SD_CLK_SEL_REG0

        SBBO    &TREG0.b0, SD_HW_BASE_PTR_REG, TREG2, 1     ; TR0.b0 -> PRUn_SD_CLK_SEL_REGi
        LSR     TREG1, TREG1, 4                             ; LS byte is ID for Ch(i+1) ; FL fix me
config_sd_ch_loop_end:
        
        JMP     RET_ADDR_REG


;
; Reset SD channel hardware
;
;   SDDF_CFG_BASE_PTR_REG: base address of SD Configuration registers (&IEP_CFG_EPWM_PRD)
;
reset_sd_ch_hw:
        ; Load T01.w0 <- SDDF_CFG_SD_CH_ID
        LBBO    &TREG0.w0, SDDF_CFG_BASE_PTR_REG, FW_REG_SDDF_CFG_SD_CH_ID_OFFSET, FW_REG_SDDF_CFG_SD_CH_ID_SZ

        LOOP    reset_sd_ch_hw_loop_end, ICSSG_NUM_SD_CH_FW ; loop over SD channels
        AND     TREG1.b0, TREG0.b0, BF_SD_CH0_ID_MASK       ; LS byte is ID for Chi

        ; Set R30[29-26]:channel_select = channel ID.
        ; Set R30[25]:channel_en=1 (set Global Channel enable).
        LSL     TREG1.b0, TREG1.b0, 2                       ; TR1.b0 = TR1.b0<<2 = (Ch ID)<<2
        SET     TREG1.b0.t1                                 ; R30[25] channel_enable=1
        MOV     R30.b3, TREG1.b0                            ; R30.b3 = TR1.b0
                                                            ; select SD Channel (Ch ID)
        LSR     TREG0, TREG0, 4                             ; LS byte is ID for Ch(i+1) ; FL fix me
        SET     R31.t23                                     ; R31[23] re_init=1
reset_sd_ch_hw_loop_end:

        JMP     RET_ADDR_REG


;
; Initialize SD (eCAP PWM) clock
;
; Arguments: 
;   SDDF_CFG_BASE_PTR_REG: base address of SD Configuration registers (&IEP_CFG_EPWM_PRD)
;
init_sd_clock:
        ; Set eCAP PWM mode
        LDI32   TREG0, (SYNCI_EN_VAL<<SYNCI_EN_SHIFT) | (SYNCO_SEL_VAL<<SYNCO_SEL_SHIFT) | (CAP_APWM_VAL<<CAP_APWM_SHIFT) | (APWMPOL_VAL<<APWMPOL_SHIFT)
        SBCO    &TREG0, CT_PRU_ICSSG_ECAP, ICSSG_eCAP_ECCTL1, 4

        ; Load TR0.w0 <- FW_REG_SDDF_CFG_SD_CLK
        LBBO    &TREG0.w0, SDDF_CFG_BASE_PTR_REG, FW_REG_SDDF_CFG_SD_CLK_OFFSET, FW_REG_SDDF_CFG_SD_CLK_SZ
        
        ; Set period count
        SUB     TREG1, TREG0.b0, 1  ; TR1 = SD_PRD_CLOCKS - 1
        SBCO    &TREG1, CT_PRU_ICSSG_ECAP, ICSSG_eCAP_CAP1, 4
        
        ; Compute & set Duty Cycle count.
        ; Divide period count by 2, biased rounding.
        ADD     TREG1, TREG0.b0, 1  ; TR1 = SD_PRD_CLOCKS + 1
        LSR     TREG1, TREG1, 1     ; TR1 = (SD_PRD_CLOCKS+1)/2
        SBCO    &TREG1, CT_PRU_ICSSG_ECAP, ICSSG_eCAP_CAP2, 4
        
        LDI     TREG0, 0x0
        SBCO    &TREG0, CT_PRU_ICSSG_ECAP, ICSSG_eCAP_CNTPHS, 4 ; clear counter phase
        SBCO    &TREG0, CT_PRU_ICSSG_ECAP, ICSSG_eCAP_TSCNT, 4  ; reset eCAP PWM Counter
        
        ; Enable eCAP PWM
        LBCO    &TREG0, CT_PRU_ICSSG_ECAP, ICSSG_eCAP_ECCTL1, 4
        SET     TREG0, TREG0, TSCNTSTP_BN   ; ICSSG_ECCTL2_ECCTL1:TSCNTSTP=1
        SBCO    &TREG0, CT_PRU_ICSSG_ECAP, ICSSG_eCAP_ECCTL1, 4        
        
        JMP     RET_ADDR_REG
