/*
 * Copyright (C) 2021 Texas Instruments Incorporated - http://www.ti.com/
 *
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *	* Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 *
 *	* Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the
 *	  distribution.
 *
 *	* Neither the name of Texas Instruments Incorporated nor the names of
 *	  its contributors may be used to endorse or promote products derived
 *	  from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _SDDF_INTF_H_
#define _SDDF_INTF_H_

#include <ti/csl/soc.h>
#include "icssg_sddf.h"


/* ========================================================================== */
/*                           Macros                                           */
/* ========================================================================== */

/* ICSSG DMEM0/1 base addresses */
#define PRU_ICSSG_DRAM0_SLV_RAM     ( CSL_PRU_ICSSG0_DRAM0_SLV_RAM_BASE )
#define PRU_ICSSG_DRAM1_SLV_RAM     ( CSL_PRU_ICSSG0_DRAM1_SLV_RAM_BASE )

/* SD channel control, channel disable/enable */
#define DEF_SD_CH_CTRL_CH_EN        ( 0 )       /* default all chs disabled */
#define BF_CH_EN_MASK               ( 0x1 )
#define SDDF_CH_CTRL_CH_EN_BF_CH0_EN_SHIFT   ( 0 )
#define SDDF_CH_CTRL_CH_EN_BF_CH0_EN_MASK    ( BF_CH_EN_MASK << SDDF_CH_CTRL_CH_EN_BF_CH0_EN_SHIFT )
#define SDDF_CH_CTRL_CH_EN_BF_CH1_EN_SHIFT   ( 1 )
#define SDDF_CH_CTRL_CH_EN_BF_CH1_EN_MASK    ( BF_CH_EN_MASK << SDDF_CH_CTRL_CH_EN_BF_CH1_EN_SHIFT )
#define SDDF_CH_CTRL_CH_EN_BF_CH2_EN_SHIFT   ( 2 )
#define SDDF_CH_CTRL_CH_EN_BF_CH2_EN_MASK    ( BF_CH_EN_MASK << SDDF_CH_CTRL_CH_EN_BF_CH2_EN_SHIFT )
#define SDDF_CH_CTRL_CH_EN_BF_CH3_EN_SHIFT   ( 3 )
#define SDDF_CH_CTRL_CH_EN_BF_CH3_EN_MASK    ( BF_CH_EN_MASK << SDDF_CH_CTRL_CH_EN_BF_CH3_EN_SHIFT )
#define SDDF_CH_CTRL_CH_EN_BF_CH4_EN_SHIFT   ( 4 )
#define SDDF_CH_CTRL_CH_EN_BF_CH4_EN_MASK    ( BF_CH_EN_MASK << SDDF_CH_CTRL_CH_EN_BF_CH4_EN_SHIFT )
#define SDDF_CH_CTRL_CH_EN_BF_CH5_EN_SHIFT   ( 5 )
#define SDDF_CH_CTRL_CH_EN_BF_CH5_EN_MASK    ( BF_CH_EN_MASK << SDDF_CH_CTRL_CH_EN_BF_CH5_EN_SHIFT )
#define SDDF_CH_CTRL_CH_EN_BF_CH6_EN_SHIFT   ( 6 )
#define SDDF_CH_CTRL_CH_EN_BF_CH6_EN_MASK    ( BF_CH_EN_MASK << SDDF_CH_CTRL_CH_EN_BF_CH6_EN_SHIFT )
#define SDDF_CH_CTRL_CH_EN_BF_CH7_EN_SHIFT   ( 7 )
#define SDDF_CH_CTRL_CH_EN_BF_CH7_EN_MASK    ( BF_CH_EN_MASK << SDDF_CH_CTRL_CH_EN_BF_CH7_EN_SHIFT )
#define SDDF_CH_CTRL_CH_EN_BF_CH8_EN_SHIFT   ( 8 )
#define SDDF_CH_CTRL_CH_EN_BF_CH8_EN_MASK    ( BF_CH_EN_MASK << SDDF_CH_CTRL_CH_EN_BF_CH8_EN_SHIFT )
#define SDDF_CH_CTRL_CH_EN_BF_CH9_EN_SHIFT   ( 9 )
#define SDDF_CH_CTRL_CH_EN_BF_CH9_EN_MASK    ( BF_CH_EN_MASK << SDDF_CH_CTRL_CH_EN_BF_CH9_EN_SHIFT )
#define SDDF_CH_CTRL_CH_EN_SHIFT             ( SDDF_CH_CTRL_CH_EN_BF_CH0_EN_SHIFT )
#define SDDF_CH_CTRL_CH_EN_MASK \
    ( SDDF_CH_CTRL_CH_EN_BF_CH0_EN_MASK | \
      SDDF_CH_CTRL_CH_EN_BF_CH1_EN_MASK | \
      SDDF_CH_CTRL_CH_EN_BF_CH2_EN_MASK | \
      SDDF_CH_CTRL_CH_EN_BF_CH3_EN_MASK | \
      SDDF_CH_CTRL_CH_EN_BF_CH4_EN_MASK | \
      SDDF_CH_CTRL_CH_EN_BF_CH5_EN_MASK | \
      SDDF_CH_CTRL_CH_EN_BF_CH6_EN_MASK | \
      SDDF_CH_CTRL_CH_EN_BF_CH7_EN_MASK | \
      SDDF_CH_CTRL_CH_EN_BF_CH8_EN_MASK | \
      SDDF_CH_CTRL_CH_EN_BF_CH9_EN_MASK )

/**    @brief    reinitialize PRU SDDF */
#define SDDF_RECFG_REINIT               ( SDDF_RECFG_BF_RECFG_REINIT_MASK )
/**    @brief    reconfigure SD clock */
#define SDDF_RECFG_CLK                  ( SDDF_RECFG_BF_RECFG_CLK_MASK )
/**    @brief    reconfigure SD OSR */
#define SDDF_RECFG_OSR                  ( SDDF_RECFG_BF_RECFG_OSR_MASK )
/**    @brief    reconfigure Trigger mode sample time */
#define SDDF_RECFG_TRIG_SAMP_TIME       ( SDDF_RECFG_BF_RECFG_TRIG_SAMPLE_TIME_MASK )
/**    @brief    reconfigure Trigger mode sample count */
#define SDDF_RECFG_TRIG_SAMP_CNT        ( SDDF_RECFG_BF_RECFG_TRIG_SAMPLE_CNT_MASK )
/**    @brief    reconfigure SD channel disable/enable */
#define SDDF_RECFG_CH_EN                ( 1<<6 )
/**    @brief    reconfigure SD channel disable/enable */
#define SDDF_RECFG_FD                   ( SDDF_RECFG_BF_RECFG_FD_MASK )
/**    @brief    reconfigure Trigger mode output sample buffer */
#define SDDF_RECFG_TRIG_OUT_SAMP_BUF    ( SDDF_RECFG_BF_RECFG_TRIG_OUT_SAMP_BUF_MASK )

/* SDDF output buffer size in 32-bit words */
#define ICSSG_SD_SAMP_CH_BUF_SZ (( SDDF_OUT_SAMP_BUF_SZ - SDDF_OUT_SAMP_BUF_IDX0 ) / SDDF_OUT_SAMP_SZ )

/* ========================================================================== */
/*                         Structures                                         */
/* ========================================================================== */

/**
 *    @brief    Structure defining SDDF clock configuration parameters.
 *
 *    @details  Firmware SD clock configuration interface exposed through PRU data <br>
 *              memory - used by driver to configure firmware parameters
 */
typedef struct Sddf_cfg_sd_clk_s
{
    volatile uint8_t  sd_prd_clocks;
    /**< clock count to generate SD clock (eCAP PWM) with desired frequency  */
    volatile uint8_t  sd_clk_inv;
    /**< invert SD clock post clock selection mux  */
} Sddf_cfg_sd_clk;

/**
 *    @brief    Structure defining SDDF triggered mode trigger times
 *
 *    @details  Firmware trigger fields exposed through PRU data <br>
 *              memory - used by driver to start sampling and    <br>
 *              generate optional output event after receiving   <br>
 *              input trigger
 */
typedef struct Sddf_cfg_trigger_s
{
    volatile uint16_t trig_samp_time;
    /**< time to start sampling after trigger input  */
    volatile uint16_t trig_opt_evt_out_time;
    /**< time to trigger optional output event after trigger input  */
} Sddf_cfg_trigger;

/**
 *    @brief    Structure defining SDDF FD Detect configuration interface
 *
 *    @details  Firmware FD configuration interface exposed through PRU data <br>
 *              memory - used by driver to configure firmware parameters
 */
typedef struct Sddf_fd_cfg_s
{
    volatile uint16_t fd_en;
    /**< bit-field to enable FD for individual SDDF channels, BitN:ChN, non-zero to enable */
    volatile uint8_t  fd_win_sz;
    /**< FD window length. 4,8,12,...,32  */
    volatile uint8_t  fd_one_min_thr;
    /**< FD minimum 1s threshold, 1 to configured Window Length  */
    volatile uint8_t  fd_one_max_thr;
    /**< FD maximum 1s threshold, 1 to configured Window Length  */
    volatile uint8_t  fd_zero_min_thr;
    /**< FD minimum 0s threshold, 1 to configured Window Length  */
    volatile uint8_t  fd_zero_max_thr;
    /**< FD maximum 0s threshold, 1 to configured Window Length  */
} Sddf_fd_cfg;

/**
 *    @brief    Structure defining SDDF configuration interface
 *
 *    @details  Firmware configuration interface exposed through PRU data <br>
 *              memory - used by driver to configure firmware parameters
 */
typedef struct Sddf_cfg_s
{
    volatile uint8_t    recfg;
    /**< bit-field containing flags indicating configurations to be executed, non-zero to enable */
    volatile uint8_t    resvd0;
    /**< reserved */  
    Sddf_cfg_sd_clk     sd_clk;
    /**< SD modulator clock, eCAP PWM period register value */
    volatile uint8_t    osr;
    /**< Over Sampling Rate (OSR) */
    volatile uint8_t    resvd1;
    /**< reserved */  
    Sddf_cfg_trigger    trigger;
    /**< triggered mode trigger times */
    volatile uint8_t    trig_sample_cnt;
    /**< triggered mode OSR loop count */
    Sddf_fd_cfg         sd_fd_cfg;
    /**< Fast Detect (FD) over current detection configuration */        
    volatile uint32_t   trig_out_samp_buf;
    /**< triggered mode host output buffer */
} Sddf_cfg;

/**
 *    @brief    Structure defining SDDF control fields
 *
 *    @details  Firmware control & status interface exposed through PRU data    <br>
 *              memory - used by driver to enable SDDF operations and to select <br>
 *              between continuous and triggered mode <br>
 */
typedef struct Sddf_ctrl_s
{
    volatile uint8_t  ctrl;
    /**< SDDF control */
    volatile uint8_t  stat;
    /**< SDDF status */
} Sddf_ctrl;

/**
 *    @brief    Structure defining SDDF channel control fields
 *
 *    @details  Used by driver to enable / disable individual SD <br>
 *              channels.
 */
typedef struct Sddf_ch_ctrl_s
{
    uint16_t    ch_en;
    /**< bit-field to enable individual SDDF channels, BitN:ChN, non-zero to enable */
} Sddf_ch_ctrl;

/**
 *    @brief    Structure defining SDDF per channel data interface
 *
 *    @details  Firmware data per channel interface exposed through PRU data <br>
 *              memory - used by driver to extract samples
 */
typedef struct Sddf_sample_ch_s
{
    volatile uint16_t     curr;
    /**< offset in 'buf' array, where next data sample would be written */
    volatile uint16_t     resvd[3];
    /**< reserved */
    volatile uint32_t     buf[ICSSG_SD_SAMP_CH_BUF_SZ];
    /**< circular sample buffer, sample previous to the one pointed by 'curr' has latest one */
} Sddf_sample_ch;

/**
 *    @brief    Structure defining SDDF data (or sample) interface
 *
 *    @details  Firmware data interface exposed through PRU data <br>
 *              memory - used by driver to extract samples
 */
typedef struct Sddf_sample_s
{
    Sddf_sample_ch  ch[ICSSG_NUM_SD_CH];
    /**< per channel data (or sample) interface */
} Sddf_sample;

/**
 *    @brief    Structure defining SDDF interface
 *
 *    @details  Firmware configuration, control, data and trigger interface exposed through PRU
 *
 */
typedef struct Sddf_s {
    uint8_t pru_id;
    /**< PRU ID */
    Sddf_ctrl       *p_sddf_ctrl;
    /**< control interface */
    Sddf_cfg        *p_sddf_cfg;
    /**< configuration interface */
    Sddf_ch_ctrl    sddf_ch_ctrl;
    /**< channel control interface */
    Sddf_sample     *p_sddf_sample;
    /**< data (or sample) interface */
} Sddf;

#endif /* #define _SDDF_INTF_H_ */
