/*
 *  Copyright (C) 2021 Texas Instruments Incorporated
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <kernel/dpl/DebugP.h>
#include "ti_drivers_config.h"
#include "ti_drivers_open_close.h"
#include "ti_board_open_close.h"

#include <stdint.h>
#include <drivers/hw_include/csl_types.h>
#include <drivers/pruicss.h>
#include <drivers/sciclient.h>

//#include "sddf_array.h"                 /* SDDF image data */
#include <gen/sddf_array.h>
#include "sddf.h"

/* ICSS INTC configuration */
static const PRUICSS_IntcInitData gPruicssIntcInitdata = PRUICSS_INTC_INITDATA;

/*
 *  ======== initIcss ========
 */
/* Initialize ICSSG */
int32_t initIcss(
    uint8_t icssInstId, 
    PRUICSS_Handle *pPruIcssHandle    
)
{
    PRUICSS_Handle pruIcssHandle;
    uint8_t i;
    int32_t status;
    
    /* Open ICSS PRU instance */
    pruIcssHandle = PRUICSS_open(icssInstId);
    if (pruIcssHandle == NULL) {
        return SDDF_ERR_INIT_ICSSG;
    }
    
    /* Disable PRUs & RTUs */
    for (i = 0; i < PRUICSS_NUM_CORES; i++)
    {
        status = PRUICSS_disableCore(pruIcssHandle, i);
        if (status != SystemP_SUCCESS) {
            return SDDF_ERR_INIT_ICSSG;
        }
    }
    
    /* Set ICSS pin mux to default */
    PRUICSS_setSaMuxMode(pruIcssHandle, PRUICSS_PINMUX_DEF);
    
    /* Initialize ICSS INTC */
    status = PRUICSS_intcInit(pruIcssHandle, &gPruicssIntcInitdata);
    if (status != SystemP_SUCCESS) {
        return SDDF_ERR_INIT_ICSSG;        
    }
    
    *pPruIcssHandle = pruIcssHandle;
    
    return SDDF_ERR_NERR;
}

/*
 *  ======== initPruSddf ========
 */
/* Initialize PRU for SDDF */
int32_t initPruSddf(
    PRUICSS_Handle pruIcssHandle,
    uint8_t pruInstId, 
    SddfPrms *pSddfPrms, 
    Sddf_handle *pHSddf
)
{
    int32_t size;
    const uint32_t *sourceMem;    /* Source memory[ Array of uint32_t ] */
    uint32_t offset;              /* Offset at which write will happen */
    uint32_t byteLen;             /* Total number of bytes to be written */
    //uint8_t pruId; // FL: remove Host FW init since FW changing
    int32_t status;
    
    /* Reset PRU */
    status = PRUICSS_resetCore(pruIcssHandle, pruInstId);
    if (status != SystemP_SUCCESS) {
        return SDDF_ERR_INIT_PRU_SDDF;
    }
    
    /* Initialize DMEM */
    size = PRUICSS_initMemory(pruIcssHandle, PRUICSS_DATARAM(pruInstId));
    if (size == 0) {
        return SDDF_ERR_INIT_PRU_SDDF;
    }
    
    /* Initialize IMEM */
    size = PRUICSS_initMemory(pruIcssHandle, PRUICSS_IRAM_PRU(pruInstId));
    if (size == 0)
    {
        return SDDF_ERR_INIT_PRU_SDDF;
    }

    /* Write DMEM */
    offset = 0;
    sourceMem = (uint32_t *)pru_sddf_image_1;
    byteLen = sizeof(pru_sddf_image_1);
    size = PRUICSS_writeMemory(pruIcssHandle, PRUICSS_DATARAM(pruInstId), offset, sourceMem, byteLen);
    if (size == 0)
    {
        return SDDF_ERR_INIT_PRU_SDDF;
    }

    /* Write IMEM */
    offset = 0;
    sourceMem = pru_sddf_image_0;
    byteLen = sizeof(pru_sddf_image_0);
    size = PRUICSS_writeMemory(pruIcssHandle, PRUICSS_IRAM_PRU(pruInstId), offset, sourceMem, byteLen);
    if (size == 0)
    {
        return SDDF_ERR_INIT_PRU_SDDF;
    }    
    
    /* Enable PRU */
    status = PRUICSS_enableCore(pruIcssHandle, pruInstId);
    if (status != SystemP_SUCCESS) {
        return SDDF_ERR_INIT_PRU_SDDF;
    }
#if 0 // FL: remove Host FW init since FW changing
    /* Translate PRU ID to SDDF API */
    if (pruInstId == PRUICSS_PRU0) {
        pruId = PRU_ID_0;
    }
    else if (pruInstId == PRUICSS_PRU1) {
        pruId = PRU_ID_1;
    }
    else {
        return SDDF_ERR_INIT_PRU_SDDF;
    }
    
    /* Initialize SDDF */
    status = initSddf(pruId, pSddfPrms, pHSddf);
    if (status != SDDF_ERR_NERR) {
        return SDDF_ERR_INIT_PRU_SDDF;
    }
#endif
    return SDDF_ERR_NERR;
}

/* Initialize SDDF */
int32_t initSddf(
    uint8_t pruId,
    SddfPrms *pSddfPrms, 
    Sddf_handle *pHSddf
)
{
    Sddf_handle hSddf;
    uint32_t trigOutSampBuf;
    uint8_t cfgMask;

    
    /* Initialize SDDF instance */
    hSddf = sddf_init(pruId);
    if (hSddf == NULL) {
        return SDDF_ERR_INIT_SDDF;
    }
    
    /* Set SDDF mode */
    if (pSddfPrms->sddfMode == SDDF_MODE_TRIG) {
        sddf_set_mode_triggered(hSddf);
    }
    else if (pSddfPrms->sddfMode == SDDF_MODE_CONT) {
        sddf_set_mode_continuous(hSddf);
    }
    else {
        return SDDF_ERR_INIT_SDDF;
    }
    
    /* Set non-default configuration */
    cfgMask = pSddfPrms->cfgMask;
    if (cfgMask & SDDF_CFG_CLK) {
        sddf_cmd_prep_clk_cfg(hSddf, &pSddfPrms->sdClkPrms);
    }
    if (cfgMask & SDDF_CFG_OSR) {
        sddf_cmd_prep_osr(hSddf, pSddfPrms->osr);
    }
    if (cfgMask & SDDF_CFG_CH_EN) {
        sddf_cmd_prep_ch_en(hSddf, pSddfPrms->chEn);
    }
    if (cfgMask & SDDF_CFG_FD) {
        sddf_cmd_prep_fd_cfg(hSddf, &pSddfPrms->fdPrms);
    }
    if (pSddfPrms->sddfMode == SDDF_MODE_TRIG) {
        /* Set non-default configuration for triggered mode */
        if (cfgMask & SDDF_CFG_TRIG_SAMP_TIME) {
            sddf_cmd_prep_trig_samp_time(hSddf, pSddfPrms->trigSampTime);
        }
        if (cfgMask & SDDF_CFG_TRIG_SAMP_CNT) {
            sddf_cmd_prep_trig_samp_cnt(hSddf, pSddfPrms->trigSampCnt);
        }
        if (cfgMask & SDDF_CFG_TRIG_OUT_SAMP_BUF) {
            trigOutSampBuf = CPU0_BTCM_SOCVIEW(pSddfPrms->trigOutSampBuf);
            sddf_cmd_prep_trig_out_samp_buf(hSddf, trigOutSampBuf);
        }
    }

    /* Enable (global) SDDF */
    sddf_enable(hSddf);
    
    *pHSddf = hSddf;
    
    return SDDF_ERR_NERR;
}
