/*
 * Copyright (C) 2021 Texas Instruments Incorporated - http://www.ti.com/
 *
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *	* Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 *
 *	* Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the
 *	  distribution.
 *
 *	* Neither the name of Texas Instruments Incorporated nor the names of
 *	  its contributors may be used to endorse or promote products derived
 *	  from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _SDDF_API_H_
#define _SDDF_API_H_


/**
 *    API usage (continuous mode):
 *    ----------
 *
 *    -> Obtain handle
 *
 *        sd_handle sd = sd_init(<config base>, <control base>, <channel control base>, <data base>);
 *
 *    -> Invoke sd_cmd_prep_* for relevent command preperation, save (and if required OR) returned flag
 *
 *        unsigned char flag = sd_cmd_prep_clk(sd, 10);
 *        flag |= sd_cmd_prep_osr(sd, 64);
 *
 *    -> Instruct firmware to execute the prepared commands
 *
 *        sd_cmd_go(sd, flag);
 *
 *    -> Start execution of the firmware
 *
 *    -> Enable continuous mode
 *
 *        sd_enable_continuous(sd);
 *
 *    -> Wait till firmware commands are executed
 *
 *        sd_cmd_wait(sd);
 *
 *    -> Read latest sample on a channel
 *
 *        sample = sd_get_cur_sample(sd, ch);
 *
 *    -> Invoke sd_cmd_prep_* for relevent command preperation, save (and if required OR) returned flag
 *
 *        flag = sd_cmd_prep_osr(sd, 128);
 *
 *    -> Instruct firmware to execute the prepared commands
 *
 *        sd_cmd_go(sd, flag);
 *
 *    -> Wait till firmware commands are executed
 *
 *        sd_cmd_wait(sd);
 *
 *    -> Read latest sample on a channel
 *
 *        sample = sd_get_cur_sample(sd, ch);
 *
 *
 *    API usage (triggered mode):
 *    ----------
 *
 *    -> Obtain handle
 *
 *        sd_handle sd = sd_init(<config base>, <control base>, <channel control base>, <data base>);
 *
 *    -> Invoke sd_cmd_prep_* for relevant command preparation, save (and if required OR) returned flag
 *
 *        unsigned char flag = sd_cmd_prep_t_sample(sd, <PWM period - pre-trigger time>);
 *        flag |= sd_cmd_prep_loop_cnt(sd, 31);
 *
 *    -> Instruct firmware to execute the prepared commands
 *
 *        sd_cmd_go(sd, flag);
 *
 *    -> Start execution of the firmware
 *
 *    -> Enable triggered mode
 *
 *        sd_enable_triggered(sd);
 *
 *
 *    Note: all reconfiguration commands perform hardware re-init.
 *
 *    Default firmware parameter values:
 *    ----------------------------------
 *    -> mode: continuous
 *    -> pwm_prd_clocks: 10 [i.e. clock - 20MHz for ICSSG CORE clock 200 MHz]
 *    -> osr: 64
 *    -> fd_en (all channels): disabled
 *
 */

//#include <ti/csl/tistdtypes.h>
#include <stdint.h>
#include "icssg_sddf.h"

/* Number of ICSSG PRUs */
#define NUM_PRU     ( 2 )
 
/* PRU IDs */
#define PRU_ID_0    ( 0 )   /* PRU 0 ID */
#define PRU_ID_1    ( 1 )   /* PRU 1 ID */

/* Number of SD channels */
#define NUM_SD_CH   ( ICSSG_NUM_SD_CH )
/* ICSSG INTC event */
#define SDDF_EVT    ( TRIGGER_HOST_SDDF_EVT )

/* SDDF handle */
typedef struct Sddf *Sddf_handle;

/* SDDF clock configuration parameters */
typedef struct Sddf_cfg_sd_clk_prms_s {
    uint8_t  sd_prd_clocks;
    /**< clock count to generate SD clock (eCAP PWM) with desired frequency  */
    uint8_t  sd_clk_inv;
    /**< invert SD clock post clock selection mux  */
} Sddf_cfg_sd_clk_prms;

/* SDDF FD Detect configuration parameters */
typedef struct Sddf_cfg_fd_prms_s {
    uint16_t fd_en;
    /**< FD disable/enable for individual SD channels. Bit field, BitN:ChN, non-zero to enable. */
    uint8_t  fd_win_sz;
    /**< FD window length. 4,8,12,...,32  */
    uint8_t  fd_one_min_thr;
    /**< FD minimum 1s threshold, 1 to configured Window Length  */
    uint8_t  fd_one_max_thr;
    /**< FD maximum 1s threshold, 1 to configured Window Length  */
    uint8_t  fd_zero_min_thr;
    /**< FD minimum 0s threshold, 1 to configured Window Length  */
    uint8_t  fd_zero_max_thr;
    /**< FD maximum 0s threshold, 1 to configured Window Length  */    
} Sddf_cfg_fd_prms;

/**
 *  @name   sddf_init
 *  @brief  Initialize SDDF instance
 *
 *  @param[in]  pru_id          PRU (SD) ID
 *
 *  @retval sddf  SDDF instance handle
 *
 */
Sddf_handle sddf_init(
    uint8_t pru_id
);

/**
 *  @name   sddf_set_mode_continuous
 *  @brief  Set SDDF to continuous sampling mode
 *
 *  @param[in]  h_sddf          SDDF handle
 *
 *  @retval none
 *
 */
void sddf_set_mode_continuous(
    Sddf_handle h_sddf
);

/**
 *  @name   sddf_set_mode_triggered
 *  @brief  Set SDDF to triggered sampling mode
 *
 *  @param[in]  h_sddf          SDDF handle
 *
 *  @retval none
 *
 */
void sddf_set_mode_triggered(
    Sddf_handle h_sddf
);

/**
 *  @name   sddf_enable
 *  @brief  SDDF global enable
 *
 *  @param[in]  h_sddf          SDDF handle
 *
 *  @retval none
 *
 */
void sddf_enable(
    Sddf_handle h_sddf
);

/**
 *  @name   sddf_cmd_prep_clk_cfg
 *  @brief  Prepare command for SD clock frequency and clock inversion. <br>
 *          SD clock frequency = (ICSSG CORE clock) / (sd_prd_clocks) <br>
 *          e.g. 200 MHz / 10 = 20 MHz
 *
 *  @param[in]  h_sddf                  SDDF handle
 *  @param[in]  p_sddf_cfg_sd_clk_prms  Pointer to SD clock configuration parameters
 *
 *  @retval flag        Clock command flag
 *
 */
uint8_t sddf_cmd_prep_clk_cfg(
    Sddf_handle h_sddf, 
    Sddf_cfg_sd_clk_prms *p_sddf_cfg_sd_clk_prms
);

/**
 *  @name   sddf_cmd_prep_osr
 *  @brief  Prepare command for SD OSR
 *
 *  @param      h_sddf          SDDF handle
 *  @param[in]  osr             OSR
 *
 *  @retval flag        OSR command flag
 *
 */
uint8_t sddf_cmd_prep_osr(
    Sddf_handle h_sddf, 
    uint8_t osr
);

/**
 *  @name   sddf_cmd_prep_trig_samp_time
 *  @brief  Prepare command for triggered mode sample time, typical value PWM period/2 - pre-trigger time from PWM reversal instant
 *
 *  @param[in]  h_sddf          SDDF handle
 *  @param[in]  trig_samp_time  Instant to trigger sample from external sync event
 *
 *  @retval flag    Trigger mode sample time command flag
 *
 */
uint8_t sddf_cmd_prep_trig_samp_time(
    Sddf_handle h_sddf, 
    uint16_t trig_samp_time
);

#if 0
/**
 *  @name   sd_cmd_prep_t_opt_event_out
 *  @brief  Prepare command for triggered mode optional output event time
 *
 *  @param[in]  h_sddf                  SDDF handle
 *  @param[in]  trig_opt_evt_out_time   Instant to generate event out from external sync event
 *
 *  @retval         Trigger mode optional event output time command flag 
 *
 */
uint8_t sddf_cmd_prep_trig_opt_evt_out_time(
    Sddf_handle h_sddf, 
    uint16_t trig_opt_evt_out_time
);
#endif

/**
 *  @name   sddf_cmd_prep_trig_samp_cnt
 *  @brief  Prepare command for triggered mode sample count
 *
 *  @param      h_sddf          SDDF handle
 *  @param[in]  trig_samp_cnt   Triggered mode sample
 *
 *  @retval flag         Trigger mode sample count command flag
 *
 */
uint8_t sddf_cmd_prep_trig_samp_cnt(
    Sddf_handle h_sddf, 
    uint16_t trig_samp_cnt
);

/**
 *  @name   sddf_cmd_prep_trig_out_samp_buf
 *  @brief  Prepare command for triggered mode output sample buffer
 *
 *  @param      h_sddf              SDDF handle
 *  @param[in]  trig_out_samp_buf   Triggered mode output sample buffer
 *
 *  @retval flag         Trigger mode output sample buffer command flag
 *
 */
uint8_t sddf_cmd_prep_trig_out_samp_buf(
    Sddf_handle h_sddf, 
    uint32_t trig_out_samp_buf
);

/**
 *  @name   sddf_cmd_prep_ch_en
 *  @brief  Prepare command for SD channel disable/enable.
 *
 *  @param[in]  h_sddf          SDDF handle
 *  @param[in]  ch_en           SD channel disable/enable. Bit field, BitN:ChN, non-zero to enable.
 *
 *  @retval flag        Channel enable command flag
 *
 */
uint8_t sddf_cmd_prep_ch_en(
    Sddf_handle h_sddf, 
    uint16_t ch_en
);

/**
 *  @name   sddf_cmd_prep_fd_cfg
 *  @brief  Prepare command for FD over current detection configuration
 *
 *  @param[in]  h_sddf              SDDF handle
 *  @param[in]  p_sddf_cfg_fd_prms  Pointer to FD over current detection parameters
 *
 *  @retval flag        FD command flag
 *
 */
uint8_t sddf_cmd_prep_fd_cfg(
    Sddf_handle h_sddf, 
    Sddf_cfg_fd_prms *p_fd_prms
);

/**
 *  @name   sddf_cmd_prep_reinit
 *  @brief  Prepare command for SDDF re-initialization. <br>
 *
 *  @param[in]  h_sddf          SDDF handle
 *
 *  @retval flag        Clock command flag
 *
 */
uint8_t sddf_cmd_prep_reinit(
    Sddf_handle h_sddf
);

/**
 *  @name   sddf_get_sample_base
 *  @brief  Get sample buffer base address for requested channel
 *
 *  @param[in]  h_sddf          SDDF handle
 *  @param[in]  ch              Channel number
 *
 *  @retval address     Sample buffer base address
 *
 */
void *sddf_get_sample_base(
    Sddf_handle h_sddf, 
    int32_t ch
);

/**
 *  @name    sddf_get_sample_offs_cur
 *  @brief   Get sample offset for the queried channel where the next sample would be written  <br>
 *        Circular buffer offset can vary from 8 to 508, each sample is 4 bytes, total 126  <br>
 *        samples. Last sample would be 4 bytes before offset (note the fact that buffer is <br>
 *        circular). User can track this offset and copy latest samples. After 126 samples  <br>
 *        are written to the buffer, buffer will be overwritten from the begining of the    <br>
 *        buffer. User, if requires all samples has to read from the buffer before it is    <br>
 *        rewritten. i.e. for OSR128 @20MHz new sample would be available in 6.4us, for 126 <br>
 *        samples it would take 806.4us, before this time all samples has to be captured so <br>
 *        that none are missed.
 *
 *  @param[in]  h_sddf          SDDF handle
 *  @param[in]  ch              Channel number
 *
 *  @retval offset      Sample offset where next sample would be written
 *
 */
uint16_t sddf_get_sample_offs_cur(
    Sddf_handle h_sddf, 
    int32_t ch
);

/**
 *  @name   sddf_get_cur_sample
 *  @brief  get current (or latest) sample for the specified channel
 *
 *  @param[in]  h_sddf          SDDF handle
 *  @param[in]  ch              channel number
 *
 *  @retval sample      value of current sample
 *
 */
uint32_t sddf_get_cur_sample(
    Sddf_handle h_sddf, 
    int32_t ch
);

/**
 *  @name   sddf_cmd_go
 *  @brief  Execute command based on flags returned from "sd_cmd_prep_*" functions <br>
 *          (multiple flags can be combined by OR-ing)
 *
 *  @param[in]  h_sddf          SDDF handle
 *  @param[in]  cmd             Command flags
 *
 *  @retval none
 *
 */
void sddf_cmd_go(
    Sddf_handle h_sddf, 
    uint8_t cmd
);

/**
 *  @name   sddf_cmd_wait
 *  @brief  Wait command completion
 *
 *  @param[in]  h_sddf          SDDF_handle
 *
 *  @retval none
 *
 */
void sddf_cmd_wait(
    Sddf_handle h_sddf
);

#endif /* _SDDF_API_H_ */
