/*
 * Copyright (C) 2021 Texas Instruments Incorporated - http://www.ti.com/
 *
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *	* Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 *
 *	* Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the
 *	  distribution.
 *
 *	* Neither the name of Texas Instruments Incorporated nor the names of
 *	  its contributors may be used to endorse or promote products derived
 *	  from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <ti/csl/tistdtypes.h>
#include "sddf_intf.h"
#include "sddf_api.h"

/* Internal structure for managing each PRU SD */
static Sddf gSddf[NUM_PRU] = {
    {PRU_ID_0, NULL, NULL, 0, NULL}, 
    {PRU_ID_1, NULL, NULL, 0, NULL}, 
};

/* Initialize SDDF instance */
Sddf_handle sddf_init(
    uint8_t pru_id
)
{
    Sddf *p_sddf;
    uint8_t sddf_ctrl;
    uint8_t sddf_stat;
    uint8_t pru_id_ack;
    
    if (pru_id == PRU_ID_0) {
        /* Initialize PRU 0 SD */
        
        p_sddf = &gSddf[pru_id];
    
        /* Initialize SDDF control address */
        p_sddf->p_sddf_ctrl = (Sddf_ctrl *)(PRU_ICSSG_DRAM0_SLV_RAM + ICSSG_SDDF_CTRL_BASE);
        /* Initialize SDDF configuration address */
        p_sddf->p_sddf_cfg  = (Sddf_cfg *)(PRU_ICSSG_DRAM0_SLV_RAM + ICSSG_SDDF_CFG_BASE);
        /* Initialize SDDF sample data base address */
        p_sddf->p_sddf_sample = (Sddf_sample *)(PRU_ICSSG_DRAM0_SLV_RAM + SDDF_OUT_SAMP_BUF_BASE);
        /* Initialize SDDF channel control */
        p_sddf->sddf_ch_ctrl.ch_en = DEF_SD_CH_CTRL_CH_EN;
        
        /* Set FW PRU ID */
        sddf_ctrl = p_sddf->p_sddf_ctrl->ctrl;
        sddf_ctrl &= ~SDDF_CTRL_BF_PRU_ID_MASK;
        sddf_ctrl |= pru_id << SDDF_CTRL_BF_PRU_ID_SHIFT;
        p_sddf->p_sddf_ctrl->ctrl = sddf_ctrl;
        
        /* Wait for FW PRU ID ACK */
        do {
            sddf_stat = p_sddf->p_sddf_ctrl->stat;
            pru_id_ack = (sddf_stat & SDDF_STAT_BF_PRU_ID_ACK_MASK) >> SDDF_STAT_BF_PRU_ID_ACK_SHIFT;
        } while (pru_id_ack != pru_id);
    }
    else if (pru_id == PRU_ID_1) {
        /* Initialize PRU 1 SD */
        
        p_sddf = &gSddf[pru_id];
        
        /* Initialize SDDF control address */
        p_sddf->p_sddf_ctrl = (Sddf_ctrl *)(PRU_ICSSG_DRAM1_SLV_RAM + ICSSG_SDDF_CTRL_BASE);
        /* Initialize SDDF configuration address */
        p_sddf->p_sddf_cfg = (Sddf_cfg *)(PRU_ICSSG_DRAM1_SLV_RAM + ICSSG_SDDF_CFG_BASE);
        /* Initialize SDDF sample data base address */
        p_sddf->p_sddf_sample = (Sddf_sample *)(PRU_ICSSG_DRAM1_SLV_RAM + SDDF_OUT_SAMP_BUF_BASE);        
        /* Initialize SDDF channel control */
        p_sddf->sddf_ch_ctrl.ch_en = DEF_SD_CH_CTRL_CH_EN;
        
        /* Set FW PRU ID */
        sddf_ctrl = p_sddf->p_sddf_ctrl->ctrl;
        sddf_ctrl &= ~SDDF_CTRL_BF_PRU_ID_MASK;
        sddf_ctrl |= pru_id << SDDF_CTRL_BF_PRU_ID_SHIFT;
        p_sddf->p_sddf_ctrl->ctrl = sddf_ctrl;
        
        /* Wait for FW PRU ID ACK */
        do {
            sddf_stat = p_sddf->p_sddf_ctrl->stat;
            pru_id_ack = (sddf_stat & SDDF_STAT_BF_PRU_ID_ACK_MASK) >> SDDF_STAT_BF_PRU_ID_ACK_SHIFT;
        } while (pru_id_ack != pru_id);
    }
    else {
        p_sddf = NULL;
    }
    
    return (Sddf_handle)p_sddf;
}

/* Set SDDF to continuous sampling mode */
void sddf_set_mode_continuous(
    Sddf_handle h_sddf
)
{
    Sddf *p_sddf;    
    Sddf_ctrl *p_sddf_ctrl;
    uint8_t sddf_ctrl;
    uint8_t sddf_stat;
    uint8_t sddf_mode_ack;

    p_sddf = (Sddf *)h_sddf;
    p_sddf_ctrl = p_sddf->p_sddf_ctrl;
    
    sddf_ctrl = p_sddf_ctrl->ctrl;
    sddf_ctrl &= ~SDDF_CTRL_BF_SDDF_MODE_MASK;
    sddf_ctrl |= BF_SDDF_MODE_CONT << SDDF_CTRL_BF_SDDF_MODE_SHIFT;
    p_sddf_ctrl->ctrl = sddf_ctrl;
    
    /* wait for ACK */
    do {
        sddf_stat = p_sddf_ctrl->stat;
        sddf_mode_ack = (sddf_stat & SDDF_STAT_BF_SDDF_MODE_ACK_MASK) >> SDDF_STAT_BF_SDDF_MODE_ACK_SHIFT;
    } while (sddf_mode_ack != BF_SDDF_MODE_CONT);    
}

/* Set SDDF to triggered sampling mode */
void sddf_set_mode_triggered(
    Sddf_handle h_sddf
)
{
    Sddf *p_sddf;    
    Sddf_ctrl *p_sddf_ctrl;
    uint8_t sddf_ctrl;
    uint8_t sddf_stat;
    uint8_t sddf_mode_ack;

    p_sddf = (Sddf *)h_sddf;
    p_sddf_ctrl = p_sddf->p_sddf_ctrl;
    
    sddf_ctrl = p_sddf_ctrl->ctrl;
    sddf_ctrl &= ~SDDF_CTRL_BF_SDDF_MODE_MASK;
    sddf_ctrl |= BF_SDDF_MODE_TRIG << SDDF_CTRL_BF_SDDF_MODE_SHIFT;
    p_sddf_ctrl->ctrl = sddf_ctrl;

    /* wait for ACK */
    do {
        sddf_stat = p_sddf_ctrl->stat;
        sddf_mode_ack = (sddf_stat & SDDF_STAT_BF_SDDF_MODE_ACK_MASK) >> SDDF_STAT_BF_SDDF_MODE_ACK_SHIFT;
    } while (sddf_mode_ack != BF_SDDF_MODE_TRIG);  
}

/* SDDF global enable */
void sddf_enable(
    Sddf_handle h_sddf
)
{
    Sddf *p_sddf;
    Sddf_ctrl *p_sddf_ctrl;
    uint8_t sddf_ctrl;
    uint8_t sddf_stat;
    uint8_t sddf_en_ack;
    
    p_sddf = (Sddf *)h_sddf;
    p_sddf_ctrl = p_sddf->p_sddf_ctrl;
    
    sddf_ctrl = p_sddf_ctrl->ctrl;
    sddf_ctrl &= ~SDDF_CTRL_BF_SDDF_EN_MASK;
    sddf_ctrl |= BF_SDDF_EN_ENABLE << SDDF_CTRL_BF_SDDF_EN_SHIFT;
    p_sddf_ctrl->ctrl = sddf_ctrl;
    
    /* wait for ACK */
    do {
        sddf_stat = p_sddf_ctrl->stat;
        sddf_en_ack = (sddf_stat & SDDF_STAT_BF_SDDF_EN_ACK_MASK) >> SDDF_STAT_BF_SDDF_EN_ACK_SHIFT;
    } while (sddf_en_ack != BF_SDDF_EN_ENABLE);
}

/* Prepare command for SDDF re-initialization */
uint8_t sddf_cmd_prep_reinit(
    Sddf_handle h_sddf
)
{
    return SDDF_RECFG_REINIT;    
}

/* Prepare command for changing SD clock frequency and clock inversion */
uint8_t sddf_cmd_prep_clk_cfg(
    Sddf_handle h_sddf, 
    Sddf_cfg_sd_clk_prms *p_sddf_cfg_sd_clk_prms
)
{
    Sddf *p_sddf;
    Sddf_cfg_sd_clk *p_sddf_cfg_sd_clk;

    p_sddf = (Sddf *)h_sddf;
    p_sddf_cfg_sd_clk = &p_sddf->p_sddf_cfg->sd_clk;
    
    /* Set SD PRD clock parameter */
    p_sddf_cfg_sd_clk->sd_prd_clocks &= ~BF_SD_PRD_CLOCKS_MASK;
    p_sddf_cfg_sd_clk->sd_prd_clocks |= p_sddf_cfg_sd_clk_prms->sd_prd_clocks;
    /* Set SD clock inversion parameter */
    p_sddf_cfg_sd_clk->sd_clk_inv &= ~BF_SD_CLK_INV_MASK;
    p_sddf_cfg_sd_clk->sd_clk_inv |= p_sddf_cfg_sd_clk_prms->sd_clk_inv;
    
    return SDDF_RECFG_CLK;
}

/* Prepare command for OSR */
uint8_t sddf_cmd_prep_osr(
    Sddf_handle h_sddf, 
    uint8_t osr
)
{
    Sddf *p_sddf;
    Sddf_cfg *p_sddf_cfg;

    p_sddf = (Sddf *)h_sddf;
    p_sddf_cfg = p_sddf->p_sddf_cfg;
    
    /* Set SD OSR parameter */
    p_sddf_cfg->osr &= ~SDDF_CFG_OSR_BF_OSR_MASK;
    p_sddf_cfg->osr |= (osr << SDDF_CFG_OSR_BF_OSR_SHIFT);
    
    return SDDF_RECFG_OSR;
}

/* Prepare command for triggered mode sample time */
uint8_t sddf_cmd_prep_trig_samp_time(
    Sddf_handle h_sddf, 
    uint16_t trig_samp_time
)
{
    Sddf *p_sddf;
    Sddf_cfg_trigger *p_sddf_cfg_trigger;

    p_sddf = (Sddf *)h_sddf;
    p_sddf_cfg_trigger = &p_sddf->p_sddf_cfg->trigger;
    
    /* Set Trigger mode sample time parameter */
    p_sddf_cfg_trigger->trig_samp_time &= ~SDDF_CFG_TRIG_SAMP_TIME_BF_TRIG_SAMP_TIME_MASK;
    p_sddf_cfg_trigger-> trig_samp_time |= (trig_samp_time << SDDF_CFG_TRIG_SAMP_TIME_BF_TRIG_SAMP_TIME_SHIFT);
    
    return SDDF_RECFG_TRIG_SAMP_TIME;
}

#if 0
/* Prepare command for triggered mode optional output event time */
uint8_t sddf_cmd_prep_trig_opt_evt_out_time(
    Sddf_handle h_sddf, 
    uint16_t trig_opt_evt_out_time
)
{
    return 0;
}
#endif

/* Prepare command for triggered mode sample count */
uint8_t sddf_cmd_prep_trig_samp_cnt(
    Sddf_handle h_sddf, 
    uint16_t trig_samp_cnt
)
{
    Sddf *p_sddf;
    Sddf_cfg *p_sddf_cfg;

    p_sddf = (Sddf *)h_sddf;
    p_sddf_cfg = p_sddf->p_sddf_cfg;
    
    /* Set Trigger mode sample time parameter */
    p_sddf_cfg->trig_sample_cnt &= ~SDDF_CFG_TRIG_SAMP_CNT_BF_TRIG_SAMP_CNT_MASK;
    p_sddf_cfg->trig_sample_cnt |= (trig_samp_cnt << SDDF_CFG_TRIG_SAMP_CNT_BF_TRIG_SAMP_CNT_SHIFT);
    
    return SDDF_RECFG_TRIG_SAMP_CNT;    
}

/**
 *  @name   sddf_cmd_prep_trig_out_samp_buf
 *  @brief  Prepare command for triggered mode output sample buffer
 *
 *  @param      h_sddf              SDDF handle
 *  @param[in]  trig_out_samp_buf   Triggered mode output sample buffer
 *
 *  @retval flag         Trigger mode output sample buffer command flag
 *
 */
uint8_t sddf_cmd_prep_trig_out_samp_buf(
    Sddf_handle h_sddf, 
    uint32_t trig_out_samp_buf
)
{
    Sddf *p_sddf;
    Sddf_cfg *p_sddf_cfg;

    p_sddf = (Sddf *)h_sddf;
    p_sddf_cfg = p_sddf->p_sddf_cfg;

    /* Set Trigger mode sample time parameter */
    p_sddf_cfg->trig_out_samp_buf &= ~SDDF_CFG_TRIG_OUT_SAMP_BUF_BF_TRIG_OUT_SAMP_BUF_MASK;
    p_sddf_cfg->trig_out_samp_buf |= (trig_out_samp_buf << SDDF_CFG_TRIG_OUT_SAMP_BUF_BF_TRIG_OUT_SAMP_BUF_SHIFT);

    return SDDF_RECFG_TRIG_OUT_SAMP_BUF;
}

/* Prepare command for SD channel disable/enable */
uint8_t sddf_cmd_prep_ch_en(
    Sddf_handle h_sddf, 
    uint16_t ch_en
)
{
    Sddf *p_sddf;
    Sddf_ch_ctrl *p_sddf_ch_ctrl;

    p_sddf = (Sddf *)h_sddf;
    p_sddf_ch_ctrl = &p_sddf->sddf_ch_ctrl;
    
    /* Set SDDF channel disable/enable parameter */
    p_sddf_ch_ctrl->ch_en &= ~SDDF_CH_CTRL_CH_EN_MASK;
    p_sddf_ch_ctrl->ch_en |= (ch_en << SDDF_CH_CTRL_CH_EN_SHIFT);
    
    return SDDF_RECFG_CH_EN;
}    

/* Prepare command for changing FD over current detection configuration */
uint8_t sddf_cmd_prep_fd_cfg(
    Sddf_handle h_sddf, 
    Sddf_cfg_fd_prms *p_fd_prms
)
{
    Sddf *p_sddf;
    Sddf_fd_cfg *p_sddf_fd_cfg;

    p_sddf = (Sddf *)h_sddf;
    p_sddf_fd_cfg = &p_sddf->p_sddf_cfg->sd_fd_cfg;
    
    /* Set FD window size */
    p_sddf_fd_cfg->fd_win_sz &= ~SDDF_CFG_FD_WIN_SZ_BF_FD_WIN_SZ_MASK;
    p_sddf_fd_cfg->fd_win_sz |= p_fd_prms->fd_win_sz;

    /* Set FD 1s min threshold,
       set FD 1s max threshold */
    p_sddf_fd_cfg->fd_one_min_thr &= ~SDDF_CFG_FD_ONE_MIN_THR_BF_FD_ONE_MIN_THR_MASK;
    p_sddf_fd_cfg->fd_one_min_thr |= (p_fd_prms->fd_one_min_thr << SDDF_CFG_FD_ONE_MIN_THR_BF_FD_ONE_MIN_THR_SHIFT);
    p_sddf_fd_cfg->fd_one_max_thr &= ~SDDF_CFG_FD_ONE_MIN_THR_BF_FD_ONE_MIN_THR_MASK;
    p_sddf_fd_cfg->fd_one_max_thr |= (p_fd_prms->fd_one_max_thr << SDDF_CFG_FD_ONE_MIN_THR_BF_FD_ONE_MIN_THR_SHIFT);

    /* Set FD 0s min threshold, 
       set FD 0s max threshold */
    p_sddf_fd_cfg->fd_zero_min_thr &= ~SDDF_CFG_FD_ZERO_MIN_THR_BF_FD_ZERO_MIN_THR_MASK;
    p_sddf_fd_cfg->fd_zero_min_thr |= (p_fd_prms->fd_zero_min_thr << SDDF_CFG_FD_ZERO_MIN_THR_BF_FD_ZERO_MIN_THR_SHIFT);
    p_sddf_fd_cfg->fd_zero_max_thr &= ~SDDF_CFG_FD_ZERO_MAX_THR_BF_FD_ZERO_MAX_THR_MASK;
    p_sddf_fd_cfg->fd_zero_max_thr |= (p_fd_prms->fd_zero_max_thr << SDDF_CFG_FD_ZERO_MAX_THR_BF_FD_ZERO_MAX_THR_SHIFT);

    /* Set FD disable/enable */
    p_sddf_fd_cfg->fd_en &= ~SDDF_CFG_FD_EN_MASK;
    p_sddf_fd_cfg->fd_en |= (p_fd_prms->fd_en << SDDF_CFG_FD_EN_SHIFT);

    return SDDF_RECFG_FD;
}

/* Get sample buffer base address for requested channel */
void *sddf_get_sample_base(
    Sddf_handle h_sddf, 
    int32_t ch
)
{
    Sddf *p_sddf = (Sddf *)h_sddf;

    return (void *)&p_sddf->p_sddf_sample->ch[ch].buf[0];
}

/* Get current (or latest) sample for the specified channel */
uint16_t sddf_get_sample_offs_cur(
    Sddf_handle h_sddf, 
    int32_t ch
)
{
    Sddf *p_sddf = (Sddf *)h_sddf;
    
    return p_sddf->p_sddf_sample->ch[ch].curr;
}

/* Get current (or latest) sample for the specified channel */
uint32_t sddf_get_cur_sample(
    Sddf_handle h_sddf, 
    int32_t ch
)
{
    Sddf *p_sddf = (Sddf *)h_sddf;    
    uint16_t offs;
    
    if ((p_sddf->sddf_ch_ctrl.ch_en >> ch) & BF_CH_EN_MASK) {
        /* Channel enabled */
        
        /* Get current sample offset */
        offs = sddf_get_sample_offs_cur(h_sddf, ch);
        
        /* Adjust offset for location of current sample, handle buffer wrap */
        offs = (offs == SDDF_OUT_SAMP_BUF_IDX0) ? (SDDF_OUT_SAMP_BUF_SZ - SDDF_OUT_SAMP_SZ) : (offs - SDDF_OUT_SAMP_SZ);
        offs = (offs - SDDF_OUT_SAMP_BUF_IDX0)/SDDF_OUT_SAMP_SZ;
        
        /* Return sample */
        return p_sddf->p_sddf_sample->ch[ch].buf[offs];
    }
    else {
        /* Channel disabled */
        
        return 0;
    }
}

/* Execute command */
void sddf_cmd_go(
    Sddf_handle h_sddf, 
    uint8_t cmd
)
{
    Sddf *p_sddf;    
    Sddf_cfg *p_sddf_cfg;
    uint8_t recfg;
    
    p_sddf = (Sddf *)h_sddf;
    p_sddf_cfg = p_sddf->p_sddf_cfg;
    
    recfg = p_sddf_cfg->recfg;
    recfg &= ~SDDF_RECFG_MASK;
    recfg |= (cmd << SDDF_RECFG_SHIFT);
    p_sddf_cfg->recfg = recfg;
}

/* Wait command completion */
void sddf_cmd_wait(
    Sddf_handle h_sddf
)
{
    Sddf *p_sddf;    
    Sddf_cfg *p_sddf_cfg;
    uint8_t recfg;

    p_sddf = (Sddf *)h_sddf;
    p_sddf_cfg = p_sddf->p_sddf_cfg;
    
    do {
        recfg = p_sddf_cfg->recfg;
        recfg &= SDDF_RECFG_MASK;
    } while (recfg != 0);
}
