/*
 *  Copyright (C) 2021 Texas Instruments Incorporated
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include <stdio.h>
#include <math.h>
#include <inttypes.h>
#include <kernel/dpl/ClockP.h>
#include <kernel/dpl/SemaphoreP.h>
#include <drivers/ipc_notify.h>
#include <drivers/pinmux.h>
#include <kernel/dpl/AddrTranslateP.h>
#include <motor_control/position_sense/endat/include/endat_drv.h>
#include <motor_control/position_sense/endat/firmware/endat_master_bin.h>
#include <motor_control/position_sense/endat/firmware/endat_master_multi_bin.h>
#include "ti_drivers_open_close.h"
#include "ti_board_open_close.h"
#include "DRV8350_defs.h"
#include "pwm.h"
#include "settings.h"
#include "ti_r5fmath_trig.h"
#include <cmsis/DSP/Include/arm_math.h>

#include "tisddf_pruss_intc_mapping.h"
#include "sddf.h"
#include <sddf_api.h>


/* EPWM ISR information */
typedef struct _AppEPwmIsrInfo_t
{
    uint32_t            epwmBaseAddr;
    SemaphoreP_Object   *pEpwmSyncSemObject;
} AppEPwmIsrInfo_t;
/* Interrupt globals */
static HwiP_Object       gEpwm0HwiObject;
static SemaphoreP_Object gEpwm0SyncSemObject;
/* variable to hold EPWM base address and semaphore address for ISR */
AppEPwmIsrInfo_t gAppEPwm0Info;

/* variables to hold EPWM base addresses */
uint32_t gEpwm0BaseAddr;
uint32_t gEpwm1BaseAddr;
uint32_t gEpwm2BaseAddr;

/* EPWM output frequency */
volatile uint32_t gEpwmOutFreq = APP_EPWM_OUTPUT_FREQ;
/* EPWM output frequency set value */
volatile uint32_t gEpwmOutFreqSet = APP_EPWM_OUTPUT_FREQ;
/* EPWM Rising Edge Delay */
uint32_t gDbRisingEdgeDelay = APP_EPWM_DB_RED_COUNT;
/* EPWM Falling Edge Delay */
uint32_t gDbFallingEdgeDelay = APP_EPWM_DB_FED_COUNT;

volatile Bool gSetEPwmOutFreq8K = TRUE;
volatile Bool gSetEPwmOutFreq16K = TRUE;
volatile Bool gSetEPwmOutFreq4K = TRUE;

// debug, mainloop count
uint32_t gLoopCnt = 0;

// debug, ISR counts
volatile uint32_t gEpwmIsrCnt = 0;
/* EPWM period */
volatile uint32_t gEpwmPrdVal;

/* For handling up-down count alternating period
   when period isn't divisible by 2 */
Bool gToggleEpwmPrd=FALSE;          /* Flag for EPWMs in alternating period mode */
uint8_t gToggleEpwmPrdState=0;      /* Alternating period state: 'Lower' or 'Upper' period written on alternating ISRs */
uint32_t gEpwmPrdValL;              /* 'Lower' EPWM period value written in 'Lower' state */
uint32_t gEpwmPrdValU;              /* 'Upper' EPWM period value written in 'Upper' state */

volatile Bool gRunFlag = TRUE;      /* Flag for continuing to execute test */

/* ICSS INTC configuration */
static const PRUICSS_IntcInitData gPruicssIntcInitdata = PRUICSS_INTC_INITDATA;

volatile uint32_t gPruSddfIrqCnt=0;         /* SDDF PRU FW IRQ count */
/* HWI global variables */
static HwiP_Object gIcssgSddfHwiObject;     /* ICSSG SDDF PRU FW HWI */

/* ICSSG Interrupt settings */
#define ICSSG_PRU_SDDF_INT_NUM          ( CSLR_R5FSS0_CORE0_INTR_PRU_ICSSG0_PR1_HOST_INTR_PEND_0 )  /* interrupt number */

#if (DEBUG_LEVEL == DEBUG_BUFFERS_ON)
/* debug, SDDF sample capture buffers */
#define DEBUG_BUFF_SZ     (65536) //(196608)//( 128 )

__attribute__((section(".gDebugBuff0"))) float gDebugBuff0[DEBUG_BUFF_SZ] = { 0 };
__attribute__((section(".gDebugBuff1"))) float gDebugBuff1[DEBUG_BUFF_SZ] = { 0 };
__attribute__((section(".gDebugBuff2"))) float gDebugBuff2[DEBUG_BUFF_SZ] = { 0 };
__attribute__((section(".gDebugBuff3"))) float gDebugBuff3[DEBUG_BUFF_SZ] = { 0 };
__attribute__((section(".gDebugBuff4"))) float gDebugBuff4[DEBUG_BUFF_SZ] = { 0 };
__attribute__((section(".gDebugBuff5"))) float gDebugBuff5[DEBUG_BUFF_SZ] = { 0 };
uint32_t gDebugBuffIdx = 0;
#endif

uint32_t gCapSddfChSampsChNum = 0;
__attribute__((section(".gCtrlVars"))) float gSddfChOffsets[3] = { 0 };

/* SDDF output samples, written by PRU */
__attribute__((section(".gSddfChSampsRaw"))) uint32_t gSddfChSamps[ICSSG_NUM_SD_CH] = { 0 };
__attribute__((section(".gSddfChSampsScaled"))) float gSddfChSampsScaled[ICSSG_NUM_SD_CH] = { 0 };

struct endat_priv *priv;

/** \brief Global Structure pointer holding PRUSS1 memory Map. */
PRUICSS_Handle gPruIcss0Handle;

/* ADC offset calculation complete */
uint8_t gSDDFOffsetComplete = FALSE;

#define ENDAT_MULTI_CH0 (1 << 0)
#define ENDAT_MULTI_CH1 (1 << 1)
#define ENDAT_MULTI_CH2 (1 << 2)

#define ENDAT_INPUT_CLOCK_UART_FREQUENCY 192000000
/* use uart clock only to start with */
#define ENDAT_INPUT_CLOCK_FREQUENCY ENDAT_INPUT_CLOCK_UART_FREQUENCY

#define ENDAT_RX_SAMPLE_SIZE    7
#define ENDAT_RX_SESQUI_DIV (1 << 15)

static int gEndat_is_multi_ch;
static unsigned char gEndat_multi_ch_mask;
static unsigned int gEndat_prop_delay[3];
static unsigned int gEndat_prop_delay_max;

__attribute__((section(".gEncChData"))) struct endat_pruss_xchg local_pruss_xchg;

volatile Bool gUpdOutIsr = FALSE;   /* Flag for updating PWM output in ISR */

arm_pid_instance_f32 gPiId;
arm_pid_instance_f32 gPiIq;
arm_pid_instance_f32 gPiSpd;
arm_pid_instance_f32 gPiPos;
volatile Bool gConstantsChanged = 0;

void enable_pwm_buffers(Bool enable)
{
    if(enable) {
        GPIO_pinWriteLow(MTR_1_PWM_EN_BASE_ADDR, MTR_1_PWM_EN_PIN);
        GPIO_pinWriteLow(MTR_2_PWM_EN_BASE_ADDR, MTR_2_PWM_EN_PIN);
        GPIO_pinWriteLow(MTR_3_PWM_EN_BASE_ADDR, MTR_3_PWM_EN_PIN);
    }
    else {
        GPIO_pinWriteHigh(MTR_1_PWM_EN_BASE_ADDR, MTR_1_PWM_EN_PIN);
        GPIO_pinWriteHigh(MTR_2_PWM_EN_BASE_ADDR, MTR_2_PWM_EN_PIN);
        GPIO_pinWriteHigh(MTR_3_PWM_EN_BASE_ADDR, MTR_3_PWM_EN_PIN);
    }
}

unsigned long endat_get_fw_version(void)
{
    return *((unsigned long *)EnDatFirmware + 2);
}

void endat_pruss_load_run_fw(void)
{
    PRUICSS_disableCore(gPruIcss0Handle, PRUICSS_PRUx);

    /*Load firmware. Set buffer = write to Pru memory */
    if(gEndat_is_multi_ch)
    {
        PRUICSS_writeMemory(gPruIcss0Handle, PRUICSS_IRAM_PRU(PRUICSS_PRUx),
                            0, (uint32_t *) EnDatFirmwareMulti,
                            sizeof(EnDatFirmwareMulti));
    }
    else
    {
        PRUICSS_writeMemory(gPruIcss0Handle, PRUICSS_IRAM_PRU(PRUICSS_PRUx),
                            0, (uint32_t *) EnDatFirmware,
                            sizeof(EnDatFirmware));
    }

    PRUICSS_resetCore(gPruIcss0Handle, PRUICSS_PRUx);
    /*Run firmware */
    PRUICSS_enableCore(gPruIcss0Handle, PRUICSS_PRUx);
}

static int endat_calc_clock(unsigned freq, struct endat_clk_cfg *clk_cfg)
{
    unsigned ns;

    if(freq > 16000000 || (freq > 12000000 && freq < 16000000))
    {
        DebugP_log("\r| ERROR: frequency above 16MHz, between 12 & 16MHz not allowed\n|\n|\n");
        return -1;
    }

    if((freq != 16000000) && (ENDAT_INPUT_CLOCK_FREQUENCY % (freq * 8)))
        DebugP_log("\r| WARNING: exact clock divider is not possible, frequencies set would be tx: %u\trx: %u\n",
                    ENDAT_INPUT_CLOCK_FREQUENCY / (ENDAT_INPUT_CLOCK_FREQUENCY / freq),
                    ENDAT_INPUT_CLOCK_FREQUENCY / (ENDAT_INPUT_CLOCK_FREQUENCY / (freq * 8)));

    ns = 2 * 1000000000 / freq; /* rx arm >= 2 clock */

    /* should be divisible by 5 */
    if(ns % 5)
    {
        ns /= 5, ns += 1,  ns *= 5;
    }

    clk_cfg->tx_div = ENDAT_INPUT_CLOCK_FREQUENCY / freq - 1;
    clk_cfg->rx_div = ENDAT_INPUT_CLOCK_FREQUENCY / (freq * 8) - 1;
    clk_cfg->rx_en_cnt = ns;
    clk_cfg->rx_div_attr = ENDAT_RX_SAMPLE_SIZE;

    if(freq == 16000000)
    {
        clk_cfg->rx_div_attr |= ENDAT_RX_SESQUI_DIV;
    }

    DebugP_logInfo("\r| clock config values - tx_div: %u\trx_div: %u\trx_en_cnt: %u\trx_div_attr: %x\n",
                    clk_cfg->tx_div, clk_cfg->rx_div, clk_cfg->rx_en_cnt, clk_cfg->rx_div_attr);

    return 0;
}

static void endat_handle_prop_delay(struct endat_priv *priv,
                                    unsigned short prop_delay)
{
    if(prop_delay > priv->rx_en_cnt)
    {
        unsigned short dis = (prop_delay - priv->rx_en_cnt) * 2 / priv->rx_en_cnt;

        endat_config_rx_arm_cnt(priv, prop_delay);
        /* propagation delay - 2T */
        endat_config_rx_clock_disable(priv, dis);
    }
    else
    {
        endat_config_rx_arm_cnt(priv, priv->rx_en_cnt);
        endat_config_rx_clock_disable(priv, 0);
    }
}

static void endat_print_encoder_info(struct endat_priv *priv)
{
    DebugP_log("EnDat 2.%d %s encoder\tID: %u %s\tSN: %c %u %c\n\n",
                priv->cmd_set_2_2 ? 2 : 1,
                (priv->type == rotary) ? "rotary" : "linear",
                priv->id.binary, (char *)&priv->id.ascii,
                (char)priv->sn.ascii_msb, priv->sn.binary, (char)priv->sn.ascii_lsb);
    DebugP_log("\rPosition: %d bits ", priv->pos_res);

    if(priv->type == rotary)
    {
        DebugP_log("(singleturn: %d, multiturn: %d) ", priv->single_turn_res,
                    priv->multi_turn_res);
    }

    DebugP_log("[resolution: %d %s]", priv->step,
                priv->type == rotary ? "M/rev" : "nm");
    DebugP_log("\r\n\nPropagation delay: %dns",
                gEndat_prop_delay[priv->channel]);
    DebugP_log("\n\n\n");
}

static unsigned endat_do_sanity_tst_delay(unsigned delay)
{
    /* (unsigned short)~0 is also a multiple of 5 */
    if(delay > (unsigned short)~0)
    {
        DebugP_log("\r| ERROR: delay greater than %uns, enter lesser value\n|\n|\n",
                    (unsigned short)~0);
        return delay;
    }

    if(delay % 5)
    {
        delay += 5, delay /= 5, delay *= 5;
        DebugP_log("\r| WARNING: delay not multiple of 5ns, rounding to %uns\n|\n|\n",
                    delay);
    }

    return delay;
}

static void endat_init_clock(uint32_t frequency, struct endat_priv *priv) {
    struct endat_clk_cfg clk_cfg;
    int32_t j;
    uint32_t delay;
    uint16_t d;

    if(endat_calc_clock(frequency, &clk_cfg) < 0)
    {
        return;
    }

    endat_config_clock(priv, &clk_cfg);

    priv->rx_en_cnt = clk_cfg.rx_en_cnt;

    for(j = 0; j < 3; j++)
    {
        if(gEndat_multi_ch_mask & 1 << j)
        {
            endat_multi_channel_set_cur(priv, j);
            endat_handle_prop_delay(priv, gEndat_prop_delay[priv->channel]);
            d = gEndat_prop_delay_max - gEndat_prop_delay[j];
            endat_config_wire_delay(priv, d);
        }
    }

    /* set tST to 2us if frequency > 1MHz, else turn it off */
    if(frequency >= 1000000)
    {
        frequency = 2000;
    }
    else
    {
        frequency = 0;
    }

    delay = endat_do_sanity_tst_delay(frequency);

    if(delay <= (unsigned short)~0)
    {
        endat_config_tst_delay(priv, (unsigned short) delay);
    }
}
volatile float gAngle = 0;
volatile uint32_t gRevolution = 0;

static void endat_handle_rx(struct endat_priv *priv, int cmd)
{
    unsigned crc;
    union endat_format_data endat_format_data;

    endat_recvd_process(priv, cmd,  &endat_format_data);
    crc = endat_recvd_validate(priv, cmd, &endat_format_data);
    gAngle = (float) endat_format_data.position_addinfo.position.position / priv->step * 360.0;
    gRevolution = endat_format_data.position_addinfo.position.revolution;

    return;
}


volatile uint32_t gPruEncoderIrqCnt=0;      /* EnDat PRU FW IRQ count */
static HwiP_Object gIcssgEncoderHwiObject;  /* ICSSG EnDat PRU FW HWI */

/* ICSSG Interrupt settings */
#define ICSSG_PRU_ENDAT_INT_NUM         ( CSLR_R5FSS0_CORE0_INTR_PRU_ICSSG0_PR1_HOST_INTR_PEND_1 )  /* interrupt number */

__STATIC_FORCEINLINE void space_vector_f32(
float32_t Ialpha,
float32_t Ibeta,
float32_t * pIa,
float32_t * pIb,
float32_t * pIc)
{
    float32_t tmp1, tmp2, tmp3;
    uint8_t vector = 3;

    tmp1 = Ibeta;
    tmp2 = (Ibeta / 2) + (0.8660254039f * Ialpha);
    tmp3 = tmp2 - tmp1;

    vector = (tmp2 > 0) ? vector - 1: vector;
    vector = (tmp3 > 0) ? vector - 1: vector;
    vector = (tmp1 < 0) ? (7 - vector): vector;

    if(vector == 1 || vector == 4){
        *pIa = tmp2;
        *pIb = tmp1 - tmp3;
        *pIc = -tmp2;
    }
    else if (vector == 2 || vector == 5){
        *pIa = tmp3 + tmp2;
        *pIb = tmp1;
        *pIc = -tmp1;
    }
    else {
        *pIa = tmp3;
        *pIb = -tmp3;
        *pIc = -(tmp1 + tmp2);
    }
}

volatile float gIq = 0.0;
volatile float gIqTarget, gIqSetPoint, gIqRef = 0.0;
__attribute__((section(".gCtrlVars"))) volatile float gIqArray[8] = {0,0.5,1.0,1.5,0,-.5,-1,0};

volatile float gId = 0.0;
volatile float gIdRef = 0.0;

volatile float gSpdTarget, gSpdSetPoint, gSpdRef = 0;
__attribute__((section(".gCtrlVars"))) volatile float gSpdArray[8] = {0,500,750,-500,-750,0,0,0};

volatile float gPosTarget, gPosSetPoint, gPosRef = 0;
__attribute__((section(".gCtrlVars"))) volatile float gPosArray[8] = {180,180,359.5,359.5,0.5,0.5,180,180};

volatile uint8_t count = 0;

volatile uint8_t gInferFromLargestPhases = TRUE;
volatile uint8_t gInferA, gInferB = FALSE;

__attribute__((section(".gCtrlVars"))) float gMechAngleOffset = 0;
float gLastMechTheta = 0;
uint16_t gLastMultiTurn = 0;
uint16_t gStartingMultiTurn = 0;

uint8_t localEnDatGetSingleMulti(float32_t * mechTheta, uint16_t * multiTurn){
    uint32_t pos, rev;

    /* Check CRC from the EnDat PRU */
    if(!(local_pruss_xchg.ch[0].crc.status & ENDAT_CRC_DATA))
        return -1;

    /* Set CRC to 0 to ensure that the PRU re-populates it every time and we aren't reading old CRC flags*/
    local_pruss_xchg.ch[0].crc.status = 0;

    /* grab position word 0/1 from the TCM */
    pos = local_pruss_xchg.ch[0].pos_word0;
    rev = local_pruss_xchg.ch[0].pos_word1;
    /* Reverse the bits since they arrive at the PRU in reverse order */
    asm("rbit %0,%1" : "=r"(pos) : "r"(pos));
    asm("rbit %0,%1" : "=r"(rev) : "r"(rev));

    /* Cobble the multiturn data together from pos0 and pos1 and create singleturn by shifting out F1/F2 and masking the multiturn bits */
    rev = ((rev & 0x07F00000) >> 15) | ((pos & 0xF8000000) >> 27);
    pos = (pos >> 2) & 0x1FFFFFF;

    *mechTheta = (float32_t) pos / (float32_t) priv->step * 360.0;
    *multiTurn = (uint16_t) rev;

    return 0;
}

/* Arm CMSIS 'arm_pid_f32' function modified here to remove the derivative portion since it is unused */
__STATIC_FORCEINLINE float32_t arm_pi_f32(
arm_pid_instance_f32 * S,
float32_t in)
{
  float32_t out;

  /* y[n] = y[n-1] + A0 * x[n] + A1 * x[n-1] */
  out = (S->A0 * in) + (S->A1 * S->state[0]) + (S->state[2]);

  /* Update state */
  S->state[1] = S->state[0];
  S->state[0] = in;
  S->state[2] = out;

  /* return to application */
  return (out);
}

uint32_t gLastEnDatCMP2 = 2897;

float32_t gClarkeAlphaMeasured, gClarkeBetaMeasured;

struct pdo_tx_data {
    uint16_t controlWord;
    int8_t modeOfOperation;
    int32_t targetPosition;
    int32_t targetVelocity;
    int16_t targetTorque;
};

struct pdo_rx_data {
    uint16_t statusWord;
    int8_t modeOfOperationDisplay;
    int32_t actualPosition;
    int32_t actualVelocity;
    int16_t actualTorque;
};

/* Target data from the PLC */
__attribute__((section(".gTxDataSection"))) struct pdo_tx_data gTxData;
/* Actual data to send to the PLC */
__attribute__((section(".gRxDataSection"))) struct pdo_rx_data gRxData;

/* EnDat PRU FW IRQ handler */
__attribute__((section(".critical_code"))) void pruEncoderIrqHandler(void *args)
{
    float32_t fdbkCurPhA, fdbkCurPhB;
    float32_t mechTheta, elecTheta, relativeMechTheta = 0;
    uint16_t multiTurn = 0;
    float32_t elecThetaSinCos[2];
    float32_t parkIdOut, parkIqOut, parkIdMeasured, parkIqMeasured;
    float32_t iparkAlphaOut, iparkBetaOut;
    float32_t spcVectAOut, spcVectBOut, spcVectCOut;
    float dc0, dc1, dc2;        /* EPWM duty cycle values */
    uint16_t cmp0, cmp1, cmp2;  /* EPWM CMPA values */
    float speed, angleDelta;
    float halfPeriod;

    /* Hard-coded for now to trigger the EnDat reading twice, not benchmarked because this should be moved to the PRU in the future */
    if(gLastEnDatCMP2 == 2897) {
        HW_WR_REG32(0x3002E088, 5897);
        gLastEnDatCMP2 = 5897;
    }
    else {
        HW_WR_REG32(0x3002E088, 2897);
        gLastEnDatCMP2 = 2897;
    }

    /* debug, show ISR timing */
    GPIO_pinWriteHigh(SDDF_DEBUG_BASE_ADDR, SDDF_DEBUG_PIN);

    /* Clear interrupt at source */
    PRUICSS_clearEvent(gPruIcss0Handle, ENDAT_EVENT);

    if (gSDDFOffsetComplete) {
        /* debug, increment PRU SDDF IRQ count */
        gPruEncoderIrqCnt++;

        /* Start FOC loop and unlock the rotor */
        if (gPruEncoderIrqCnt > OFFSET_FINISHED) {
            /* Get the latest mechanical theta and multi-turn position from the encoder */
            if (localEnDatGetSingleMulti(&mechTheta, &multiTurn) != 0)
                   return;
            /* Use calculated offset from electrical 0, 4 pole pairs */
            elecTheta = mechTheta - gMechAngleOffset;
            if(elecTheta < 0)
                elecTheta += 90;
            elecTheta *= 4.0;

            /* ti_r5fmath_sincos expects input in the range of 0-2pi radians so normalize here to 0-360 degrees */
            while (elecTheta > 360.0)
                elecTheta -= 360.0;

            angleDelta = mechTheta - gLastMechTheta;
            /* Max angle change per period @ 3200RPMs with 50KHz period should be 0.384 degrees, outside that is rollover
             * Use 5 as the bound to provide plenty of room for max acceleration cases */
            if (angleDelta < -5)
                angleDelta += 360;
            else if (angleDelta > 5)
                angleDelta -= 360;

            /* Compute instantaneous RPMs using the change in angle and the PWM frequency */
            speed = (angleDelta / 360.0) / (ISR_PRD_IN_MINUTES);

#if (PRECOMPUTE_LEVEL == NO_PRECOMPUTE)
            /* Try to avoid the INA240 switching noise by selecting the 2 phases with the largest duty cycles */
            if (gInferA) {
                fdbkCurPhB = (-((float)gSddfChSamps[1] - gSddfChOffsets[1] - SDDF_HALF_SCALE_FLT) / SDDF_HALF_SCALE_FLT) * 30.0;
                fdbkCurPhA = -fdbkCurPhB - ((-((float)gSddfChSamps[2] - gSddfChOffsets[2] - SDDF_HALF_SCALE_FLT) / SDDF_HALF_SCALE_FLT) * 30.0);

            }
            else if (gInferB){
                fdbkCurPhA = (-((float)gSddfChSamps[0] - gSddfChOffsets[0] - SDDF_HALF_SCALE_FLT) / SDDF_HALF_SCALE_FLT) * 30.0;
                fdbkCurPhB = -fdbkCurPhA - ((-((float)gSddfChSamps[2] - gSddfChOffsets[2] - SDDF_HALF_SCALE_FLT) / SDDF_HALF_SCALE_FLT) * 30.0);
            }
            else {
                fdbkCurPhA = (-((float)gSddfChSamps[0] - gSddfChOffsets[0] - SDDF_HALF_SCALE_FLT) / SDDF_HALF_SCALE_FLT) * 30.0;
                fdbkCurPhB = (-((float)gSddfChSamps[1] - gSddfChOffsets[1] - SDDF_HALF_SCALE_FLT) / SDDF_HALF_SCALE_FLT) * 30.0;
            }

            /* Clarke transform */
            arm_clarke_f32(fdbkCurPhA, fdbkCurPhB, &gClarkeAlphaMeasured, &gClarkeBetaMeasured);
#endif
            /* Calculate sine and cosine of the electrical angle (elecTheta converted to radians) */
            ti_r5fmath_sincos((elecTheta * TI_R5FMATH_PIOVER180), ti_r5fmath_sincosPIconst, ti_r5fmath_sincosCoef, elecThetaSinCos);
            /* Park transform */
            arm_park_f32(gClarkeAlphaMeasured, gClarkeBetaMeasured, &parkIdMeasured, &parkIqMeasured, elecThetaSinCos[0], elecThetaSinCos[1]);

// Open loop Iq
#if (BUILDLEVEL == OPEN_LOOP_IQ_ID)
            if(gIq < IQ_TESTING)
                gIq += 0.01;

            parkIdOut = 0.0;
            parkIqOut = gIq;
#endif
//Closed loop Iq and Id tuning
#if (BUILDLEVEL == CLOSED_LOOP_IQ_ID)
            gIqTarget = gIqArray[count];
            gIqSetPoint = gIqTarget;

            if((gPruEncoderIrqCnt - OFFSET_FINISHED) % CYCLES_PER_TARGET == 0){
                count++;
                if(count == 8)
                    count = 0;
            }

            parkIqOut = arm_pi_f32(&gPiIq, gIqSetPoint - parkIqMeasured);
            parkIdOut = arm_pi_f32(&gPiId, 0.0 - parkIdMeasured);
#endif
//Closed loop speed
#if (BUILDLEVEL == CLOSED_LOOP_SPEED)
            gSpdTarget = gSpdArray[count];
            if(gSpdSetPoint < gSpdTarget)
                gSpdSetPoint += MAX_SPD_CHANGE;
            else if (gSpdSetPoint > gSpdTarget)
                gSpdSetPoint -= MAX_SPD_CHANGE;

            if((gPruEncoderIrqCnt - OFFSET_FINISHED) % CYCLES_PER_TARGET == 0){
                count++;
                if(count == 8)
                    count = 0;
            }

            gIqRef = arm_pi_f32(&gPiSpd, gSpdSetPoint - speed);
            parkIqOut = arm_pi_f32(&gPiIq, gIqRef - parkIqMeasured);
            parkIdOut = arm_pi_f32(&gPiId, 0.0 - parkIdMeasured);
#endif

//Closed loop position
#if (BUILDLEVEL == CLOSED_LOOP_POSITION)
            /* Create a 0-1080 degree reference frame using the multiturn data
             * 0-360 is overflow with a counter-clockwise turn
             * 360-720 is no overflow condition (same revolution as start point)
             * 720-1080 is overflow with a clockwise turn
             */
            if (multiTurn == gStartingMultiTurn) /* no overflow, move to center of the reference frame */
                relativeMechTheta = mechTheta + 360;
            else if ((multiTurn == (gStartingMultiTurn + 1)) || ((gStartingMultiTurn > 1) && (multiTurn == 0))) /* clockwise overflow */
                relativeMechTheta = mechTheta + 720;
            else /* counter-clockwise overflow, leave in the lower end of the reference frame */
                relativeMechTheta = mechTheta;

            gPosTarget = gPosArray[count];
            if(gPosSetPoint < gPosTarget) {
                gPosSetPoint += MAX_POS_CHANGE;
                if (gPosSetPoint > gPosTarget) /* Increment went over the request */
                    gPosSetPoint = gPosTarget;
            }
            else if (gPosSetPoint > gPosTarget) {
                gPosSetPoint -= MAX_POS_CHANGE;
                if (gPosSetPoint < gPosTarget) /* Decrement went below the request */
                    gPosSetPoint = gPosTarget;
            }
            if((gPruEncoderIrqCnt - OFFSET_FINISHED) % CYCLES_PER_TARGET == 0){
                count++;
                if(count == 8)
                    count = 0;
            }

            /* Move the setpoint to the 360-720 reference frame and compare */
            gSpdRef = arm_pi_f32(&gPiPos, (gPosSetPoint + 360) - relativeMechTheta);
            gIqRef = arm_pi_f32(&gPiSpd, gSpdRef - speed);
            parkIqOut = arm_pi_f32(&gPiIq, gIqRef - parkIqMeasured);
            parkIdOut = arm_pi_f32(&gPiId, 0.0 - parkIdMeasured);
#endif

//Closed loop CiA402 Data
#if (BUILDLEVEL == CLOSED_LOOP_CIA402)
            switch(gTxData.modeOfOperation){
            case CYCLIC_SYNC_POSITION_MODE:
                gRxData.actualPosition = (int32_t) (mechTheta * 1000.0);
                /* Create a 0-1080 degree reference frame using the multiturn data
                 * 0-360 is overflow with a counter-clockwise turn
                 * 360-720 is no overflow condition (same revolution as start point)
                 * 720-1080 is overflow with a clockwise turn
                 */
                if (multiTurn == gStartingMultiTurn) /* no overflow, move to center of the reference frame */
                    relativeMechTheta = mechTheta + 360;
                else if ((multiTurn == (gStartingMultiTurn + 1)) || ((gStartingMultiTurn > 1) && (multiTurn == 0))) /* clockwise overflow */
                    relativeMechTheta = mechTheta + 720;
                else /* counter-clockwise overflow, leave in the lower end of the reference frame */
                    relativeMechTheta = mechTheta;

                gPosTarget = (float) gTxData.targetPosition / 1000.0;
                if(gPosSetPoint < gPosTarget) {
                    gPosSetPoint += MAX_POS_CHANGE;
                    if (gPosSetPoint > gPosTarget) /* Increment went over the request */
                        gPosSetPoint = gPosTarget;
                }
                else if (gPosSetPoint > gPosTarget) {
                    gPosSetPoint -= MAX_POS_CHANGE;
                    if (gPosSetPoint < gPosTarget) /* Decrement went below the request */
                        gPosSetPoint = gPosTarget;
                }

                /* Move the setpoint to the 360-720 reference frame and compare*/
                gSpdRef = arm_pi_f32(&gPiPos, (gPosSetPoint + 360) - relativeMechTheta);
                gIqRef = arm_pi_f32(&gPiSpd, gSpdRef - speed);
                parkIqOut = arm_pi_f32(&gPiIq, gIqRef - parkIqMeasured);
                parkIdOut = arm_pi_f32(&gPiId, 0.0 - parkIdMeasured);
                break;
            case CYCLIC_SYNC_VELOCITY_MODE:
                gRxData.actualVelocity = (int32_t) (speed * 1000.0);
                gSpdTarget = (float) gTxData.targetVelocity / 1000.0;
                if(gSpdSetPoint < gSpdTarget)
                    gSpdSetPoint += MAX_SPD_CHANGE;
                else if (gSpdSetPoint > gSpdTarget)
                    gSpdSetPoint -= MAX_SPD_CHANGE;

                gIqRef = arm_pi_f32(&gPiSpd, gSpdSetPoint - speed);
                parkIqOut = arm_pi_f32(&gPiIq, gIqRef - parkIqMeasured);
                parkIdOut = arm_pi_f32(&gPiId, 0.0 - parkIdMeasured);
                break;
            case CYCLIC_SYNC_TORQUE_MODE:
                gRxData.actualTorque = (int16_t) (parkIqOut * 1000.0);
                gIqTarget = (float) gTxData.targetTorque / 1000.0;
                gIqSetPoint = gIqTarget;

                parkIqOut = arm_pi_f32(&gPiIq, gIqSetPoint - parkIqMeasured);
                parkIdOut = arm_pi_f32(&gPiId, 0.0 - parkIdMeasured);
                break;
            default:
                break;
            }
#endif

            /* Inverse Park transform */
            arm_inv_park_f32(parkIdOut, parkIqOut, &iparkAlphaOut, &iparkBetaOut, elecThetaSinCos[0], elecThetaSinCos[1]);
            /* Space Vector Generation */
            space_vector_f32(iparkAlphaOut, iparkBetaOut, &spcVectAOut, &spcVectBOut, &spcVectCOut);

            /* Write next CMPA values. Swap cmp0 and cmp2 because the HW connect PWM0 to Phase C and PWM2 to Phase A */
            halfPeriod = (float)gEpwmPrdVal / 2.0;
            writeCmpA(gEpwm0BaseAddr, (uint16_t) ((1 - spcVectCOut) * halfPeriod));
            writeCmpA(gEpwm1BaseAddr, (uint16_t) ((1 - spcVectBOut) * halfPeriod));
            writeCmpA(gEpwm2BaseAddr, (uint16_t) ((1 - spcVectAOut) * halfPeriod));

            asm("    dsb");

            /* Determine if Phase A or Phase B currents need to be inferred in the next cycle based on the two largest phases */
            gInferA = FALSE;
            gInferB = FALSE;

            if (spcVectCOut > spcVectAOut || spcVectCOut > spcVectBOut) {   /* If C is larger than either A or B */
                if (spcVectAOut > spcVectBOut) {                            /* A and C are the largest, infer B in the next cycle */
                    gInferB = TRUE;
                }
                else {                                                      /* B and C are the largest, infer A in the next cycle */
                    gInferA = TRUE;
                }
            }

            gLastMechTheta = mechTheta;
            gLastMultiTurn = multiTurn;

#if (DEBUG_LEVEL == DEBUG_BUFFERS_ON)
#if (BUILDLEVEL == OPEN_LOOP_IQ_ID)
            gDebugBuff0[gDebugBuffIdx] = gClarkeAlphaMeasured;
            gDebugBuff1[gDebugBuffIdx] = gClarkeBetaMeasured;
            gDebugBuff2[gDebugBuffIdx] = parkIqMeasured;
            gDebugBuff3[gDebugBuffIdx] = spcVectAOut;
            gDebugBuff4[gDebugBuffIdx] = spcVectBOut;
            gDebugBuff5[gDebugBuffIdx] = spcVectCOut;
#endif
#if (BUILDLEVEL == CLOSED_LOOP_IQ_ID)
            gDebugBuff0[gDebugBuffIdx] = gClarkeAlphaMeasured;
            gDebugBuff1[gDebugBuffIdx] = gClarkeBetaMeasured;
            gDebugBuff2[gDebugBuffIdx] = parkIqMeasured;
            gDebugBuff3[gDebugBuffIdx] = parkIdMeasured;
            gDebugBuff4[gDebugBuffIdx] = gIqRef;
            gDebugBuff5[gDebugBuffIdx] = speed;
#endif
#if (BUILDLEVEL == CLOSED_LOOP_SPEED)
            gDebugBuff0[gDebugBuffIdx] = gClarkeAlphaMeasured;
            gDebugBuff1[gDebugBuffIdx] = gClarkeBetaMeasured;
            gDebugBuff2[gDebugBuffIdx] = parkIqMeasured;
            gDebugBuff3[gDebugBuffIdx] = parkIdMeasured;
            gDebugBuff4[gDebugBuffIdx] = gIqRef;
            gDebugBuff5[gDebugBuffIdx] = speed;
#endif
#if (BUILDLEVEL == CLOSED_LOOP_POSITION)
            gDebugBuff0[gDebugBuffIdx] = gClarkeAlphaMeasured;
            gDebugBuff1[gDebugBuffIdx] = gClarkeBetaMeasured;
            gDebugBuff2[gDebugBuffIdx] = parkIqMeasured;
            gDebugBuff3[gDebugBuffIdx] = gIqRef;
            gDebugBuff4[gDebugBuffIdx] = speed;
            gDebugBuff5[gDebugBuffIdx] = mechTheta;
#endif
#endif

#if (PID_TUNE_LEVEL == PID_TUNING)
//Tune Id and Iq PID parameters
#if (BUILDLEVEL == CLOSED_LOOP_IQ_ID)
            /* debug, update capture buffers index */
            gDebugBuffIdx++;
            if (gDebugBuffIdx >= TARGET_BUFF_SIZE) {
                gDebugBuffIdx = 0;

                if(gConstantsChanged) {
                    arm_pid_init_f32(&gPiIq, 1);
                    arm_pid_init_f32(&gPiId, 1);
                    gConstantsChanged = 0;
                }
                /* Reset the last mechanical theta in case R5F paused for tuning while PRU continues to update the position */
                  localEnDatGetSingleMulti(&gLastMechTheta, &gLastMultiTurn);
            }
#endif

//Tune Speed PID parameters
#if (BUILDLEVEL == CLOSED_LOOP_SPEED)
            /* debug, update capture buffers index */
            gDebugBuffIdx++;
            if (gDebugBuffIdx >= TARGET_BUFF_SIZE) {
                gDebugBuffIdx = 0;

                if(gConstantsChanged) {
                    arm_pid_init_f32(&gPiIq, 1);
                    arm_pid_init_f32(&gPiId, 1);
                    arm_pid_init_f32(&gPiSpd, 1);
                    gConstantsChanged = 0;
                }
                /* Reset the last mechanical theta in case R5F paused for tuning while PRU continues to update the position */
                localEnDatGetSingleMulti(&gLastMechTheta, &gLastMultiTurn);
            }
#endif


//Tune Position PID parameters
#if (BUILDLEVEL == CLOSED_LOOP_POSITION)
            /* debug, update capture buffers index */
            gDebugBuffIdx++;
            if ((gDebugBuffIdx >= TARGET_BUFF_SIZE) {
                gDebugBuffIdx = 0;

                if( gConstantsChanged) {
                    arm_pid_init_f32(&gPiIq, 1);
                    arm_pid_init_f32(&gPiId, 1);
                    arm_pid_init_f32(&gPiSpd, 1);
                    arm_pid_init_f32(&gPiPos, 1);
                    gConstantsChanged = 0;
                }
                /* Reset the last mechanical theta in case R5F paused for tuning while PRU continues to update the position */
                localEnDatGetSingleMulti(&gLastMechTheta, &gLastMultiTurn);
            }
#endif
#elif (PID_TUNE_LEVEL == NO_TUNING) /* No tuning */
#if (DEBUG_LEVEL == DEBUG_BUFFERS_ON)
#if (DEBUG_WRAP_TYPE == DEBUG_BUFFER_WRAP) /* No tuning, wrap buffer */
            /* debug, update capture buffers index */
            gDebugBuffIdx++;
            if (gDebugBuffIdx >= DEBUG_BUFF_SZ) {
                gDebugBuffIdx = 0;
            }
#elif (DEBUG_WRAP_TYPE == DEBUG_BUFFER_SINGLE) /* No tuning, single buffer store */

            /* debug, update capture buffers index */
            gDebugBuffIdx++;
            if (gDebugBuffIdx >= DEBUG_BUFF_SZ) {
                gDebugBuffIdx = DEBUG_BUFF_SZ - 1;
            }
#endif
#endif
#endif
        }
        /* Calculate mechanical/electrical angle offset */
        else {
            /* Get the latest mechanical theta from the encoder */
            localEnDatGetSingleMulti(&mechTheta, &multiTurn);
            if (gPruEncoderIrqCnt > SETTLING_COUNT)
                gMechAngleOffset += mechTheta;
            if (gPruEncoderIrqCnt == OFFSET_FINISHED) {
                gMechAngleOffset /= SETTLING_COUNT;
                while (gMechAngleOffset > 90)
                    gMechAngleOffset -= 90;

                /* Electrical Angle offset complete, turn off all phases */
                computeCmpx(0.0, gEpwmPrdVal, &dc0, &cmp0);
                computeCmpx(0.0, gEpwmPrdVal, &dc1, &cmp1);
                computeCmpx(0.0, gEpwmPrdVal, &dc2, &cmp2);

                /* Write next CMPA values. Swap cmp0 and cmp2 because the HW connect PWM0 to Phase C and PWM2 to Phase A */
                writeCmpA(gEpwm0BaseAddr, cmp2);
                writeCmpA(gEpwm1BaseAddr, cmp1);
                writeCmpA(gEpwm2BaseAddr, cmp0);

                gLastMechTheta = mechTheta;
                gLastMultiTurn = multiTurn;
                gStartingMultiTurn = multiTurn;

                /* Set initial position information */
                gPosSetPoint = mechTheta;
                gTxData.targetPosition = mechTheta;
            }
        }
    }
    GPIO_pinWriteLow(SDDF_DEBUG_BASE_ADDR, SDDF_DEBUG_PIN);
}

void init_encoder(){
    int i, j;
    void *pruss_cfg;
    HwiP_Params hwiPrms;
    int32_t status;

    /* Configure g_mux_en to 1 in ICSSG_SA_MX_REG Register. */
    HW_WR_REG32((CSL_PRU_ICSSG0_PR1_CFG_SLV_BASE+0x40), (0x80));

    i = endat_get_fw_version();

    DebugP_log("\n\n\n");
    DebugP_log("EnDat firmware \t: %x.%x.%x (%s)\n\n", (i >> 24) & 0x7F,
                (i >> 16) & 0xFF, i & 0xFFFF, i & (1 << 31) ? "internal" : "release");

    /* Register & enable ICSSG EnDat PRU FW interrupt */
    HwiP_Params_init(&hwiPrms);
    hwiPrms.intNum      = ICSSG_PRU_ENDAT_INT_NUM;
    hwiPrms.callback    = &pruEncoderIrqHandler;
    hwiPrms.args        = 0;
    hwiPrms.isPulse     = FALSE;
    hwiPrms.isFIQ       = FALSE;
    status              = HwiP_construct(&gIcssgEncoderHwiObject, &hwiPrms);
    DebugP_assert(status == SystemP_SUCCESS);

    gPruIcss0Handle = PRUICSS_open(CONFIG_PRU_ICSS0);

    /* Set in constant table C30 to shared RAM 0x40300000 */
    PRUICSS_setConstantTblEntry(gPruIcss0Handle, PRUICSS_PRUx, PRUICSS_CONST_TBL_ENTRY_C30, ((0x40300000 & 0x00FFFF00) >> 8));

    /* clear ICSS0 PRU1 data RAM */
    PRUICSS_initMemory(gPruIcss0Handle, PRUICSS_DATARAM(PRUICSS_PRUx));

    PRUICSS_disableCore(gPruIcss0Handle, PRUICSS_PRUx);

    gEndat_is_multi_ch = 1;
    gEndat_multi_ch_mask = ENDAT_MULTI_CH0;

    DebugP_log("\r\nchannels %s %s %s selected\n\r\n\n",
                gEndat_multi_ch_mask & ENDAT_MULTI_CH0 ? "0" : "",
                gEndat_multi_ch_mask & ENDAT_MULTI_CH1 ? "1" : "",
                gEndat_multi_ch_mask & ENDAT_MULTI_CH2 ? "2" : "");

    pruss_cfg = (void *)(((PRUICSS_HwAttrs *)(gPruIcss0Handle->hwAttrs))->cfgRegBase);

    priv = endat_init((struct endat_pruss_xchg *)((PRUICSS_HwAttrs *)(
                          gPruIcss0Handle->hwAttrs))->pru1DramBase, pruss_cfg);

    endat_config_host_trigger(priv);
    endat_config_multi_channel_mask(priv, gEndat_multi_ch_mask);
    endat_pruss_load_run_fw();

    /* check initialization ack from firmware, with a timeout of 5 second */
    i = endat_wait_initialization(priv, 5000);

    if(i < 0)
    {
        DebugP_log("\rERROR: EnDat initialization failed -\n\n");

        if(gEndat_is_multi_ch)
        {
            unsigned char tmp;

            tmp = endat_multi_channel_detected(priv) & gEndat_multi_ch_mask;
            tmp ^= gEndat_multi_ch_mask;
            DebugP_log("\r\tunable to detect encoder in channel %s %s %s\n",
                        tmp & ENDAT_MULTI_CH0 ? "0" : "",
                        tmp & ENDAT_MULTI_CH1 ? "1" : "",
                        tmp & ENDAT_MULTI_CH2 ? "2" : "");
        }
        else
        {
            DebugP_log("\r\tcheck whether encoder is connected and ensure proper connections\n");
        }

        DebugP_log("\rexit %s due to failed firmware initialization\n", __func__);
        return;
    }

    /* read encoder info at low frequency so that cable length won't affect */
    endat_init_clock(200*1000, priv);

    for(j = 0; j < 3; j++)
    {
        if(gEndat_multi_ch_mask & 1 << j)
        {
            endat_multi_channel_set_cur(priv, j);

            if(endat_get_encoder_info(priv) < 0)
            {
                DebugP_log("\rEnDat initialization channel %d failed\n", j);
                DebugP_log("\rexit %s due to failed initialization\n", __func__);
                return;
            }

            gEndat_prop_delay[priv->channel] = endat_get_prop_delay(priv);
            DebugP_log("\n\t\t\t\tCHANNEL %d\n\n", j);
            endat_print_encoder_info(priv);
        }
    }

    gEndat_prop_delay_max = gEndat_prop_delay[0] > gEndat_prop_delay[1] ?
                           gEndat_prop_delay[0] : gEndat_prop_delay[1];
    gEndat_prop_delay_max = gEndat_prop_delay_max > gEndat_prop_delay[2] ?
                           gEndat_prop_delay_max : gEndat_prop_delay[2];

    /* default frequency - 16MHz for 2.2 encoders, 1MHz for 2.1 encoders */
    endat_init_clock(16 * 1000 * 1000, priv);
    /* JR: Hard code propagation delay to make 16MHz work @300MHz PRU */
    endat_handle_prop_delay(priv, 265);

    /* JR: Configures EnDat to trigger based on IEP CMP2 event */
    endat_config_periodic_trigger(priv);

    DebugP_assert(endat_command_process(priv, 8, NULL) >= 0);
    endat_multi_channel_set_cur(priv, 0);
    DebugP_log("\r|\n|\t\t\t\tCHANNEL %d\n", 0);
    endat_handle_rx(priv, 8);
}

/* Test SDDF handle */
Sddf_handle gHSddf;

/* Test SDDF parameters */
SddfPrms gTestSddfPrms = {
    SDDF_MODE_TRIG,     // sddfMode
    {0, 0, 0, 0, 0, 0}, // fdPrms
    0x1FF,              // chEn
    3,                  // trigSampCnt
    //31,                  // trigSampCnt
    //10,                  // trigSampCnt
    //45200,              // trigSampTime
    3000,
    //0,              // trigSampTime
    //4500,              // trigSampTime
    63,                 // osr
    {0, 0},             // sdClkPrms
    (uint32_t)gSddfChSamps,       // trigOutSampBuf
    SDDF_CFG_OSR | SDDF_CFG_TRIG_SAMP_TIME | SDDF_CFG_TRIG_SAMP_CNT | SDDF_CFG_CH_EN | SDDF_CFG_TRIG_OUT_SAMP_BUF // cfgMask
};

/* SDDF PRU FW IRQ handler */
void pruSddfIrqHandler(void *args)
{
    /* debug, show ISR timing */
    //GPIO_pinWriteHigh(SDDF_DEBUG_BASE_ADDR, SDDF_DEBUG_PIN);

    /* debug, increment PRU SDDF IRQ count */
    gPruSddfIrqCnt++;
    if(gPruSddfIrqCnt > SETTLING_COUNT && gPruSddfIrqCnt <= OFFSET_FINISHED){
        gSddfChOffsets[0] += (int32_t) gSddfChSamps[0] - SDDF_HALF_SCALE;
        gSddfChOffsets[1] += (int32_t) gSddfChSamps[1] - SDDF_HALF_SCALE;
        gSddfChOffsets[2] += (int32_t) gSddfChSamps[2] - SDDF_HALF_SCALE;

        if(gPruSddfIrqCnt == OFFSET_FINISHED) {
            float dc0, dc1, dc2;
            uint16_t cmp0, cmp1, cmp2;

            gSddfChOffsets[0] = (gSddfChOffsets[0] / SETTLING_COUNT);
            gSddfChOffsets[1] = (gSddfChOffsets[1] / SETTLING_COUNT);
            gSddfChOffsets[2] = (gSddfChOffsets[2] / SETTLING_COUNT);

            /* ADC offset complete, lock the rotor to electrical 0 */
            computeCmpx(0.1, gEpwmPrdVal, &dc0, &cmp0);
            computeCmpx(-0.1, gEpwmPrdVal, &dc1, &cmp1);
            computeCmpx(-0.1, gEpwmPrdVal, &dc2, &cmp2);

            /* Write next CMPA values. Swap cmp0 and cmp2 because the HW connect PWM0 to Phase C and PWM2 to Phase A */
            writeCmpA(gEpwm0BaseAddr, cmp2);
            writeCmpA(gEpwm1BaseAddr, cmp1);
            writeCmpA(gEpwm2BaseAddr, cmp0);
            gSDDFOffsetComplete = TRUE;
#if (PRECOMPUTE_LEVEL == NO_PRECOMPUTE)
            HwiP_disableInt(ICSSG_PRU_SDDF_INT_NUM);
#endif
        }

    }
#if (PRECOMPUTE_LEVEL == PRECOMPUTE_CLARKE)
    /* Offset has been calculated, pull the value and convert to float and scale it */
    else {
        float32_t fdbkCurPhA, fdbkCurPhB;

       /* Try to avoid the INA240 switching noise by selecting the 2 phases with the largest duty cycles */
       if (gInferA) {
           fdbkCurPhB = (-((float)gSddfChSamps[1] - gSddfChOffsets[1] - SDDF_HALF_SCALE_FLT) / SDDF_HALF_SCALE_FLT) * 30.0;
           fdbkCurPhA = -fdbkCurPhB - ((-((float)gSddfChSamps[2] - gSddfChOffsets[2] - SDDF_HALF_SCALE_FLT) / SDDF_HALF_SCALE_FLT) * 30.0);

       }
       else if (gInferB){
           fdbkCurPhA = (-((float)gSddfChSamps[0] - gSddfChOffsets[0] - SDDF_HALF_SCALE_FLT) / SDDF_HALF_SCALE_FLT) * 30.0;
           fdbkCurPhB = -fdbkCurPhA - ((-((float)gSddfChSamps[2] - gSddfChOffsets[2] - SDDF_HALF_SCALE_FLT) / SDDF_HALF_SCALE_FLT) * 30.0);
       }
       else {
           fdbkCurPhA = (-((float)gSddfChSamps[0] - gSddfChOffsets[0] - SDDF_HALF_SCALE_FLT) / SDDF_HALF_SCALE_FLT) * 30.0;
           fdbkCurPhB = (-((float)gSddfChSamps[1] - gSddfChOffsets[1] - SDDF_HALF_SCALE_FLT) / SDDF_HALF_SCALE_FLT) * 30.0;
       }

       /* Clarke transform */
       arm_clarke_f32(fdbkCurPhA, fdbkCurPhB, &gClarkeAlphaMeasured, &gClarkeBetaMeasured);
    }
#endif

    /* Clear interrupt at source */
    /* Write 18 to ICSSG_STATUS_CLR_INDEX_REG
        Firmware:   TRIGGER_HOST_SDDF_IRQ defined as 18
        Host:       SDDF_EVT defined as TRIGGER_HOST_SDDF_IRQ
        18 = 16+2, 2 is Host Interrupt Number. See AM654x TRM, Table 6-391.
    */
    PRUICSS_clearEvent(gPruIcss0Handle, SDDF_EVT);

    /* debug, show ISR timing */
    //GPIO_pinWriteLow(SDDF_DEBUG_BASE_ADDR, SDDF_DEBUG_PIN);
}

void init_sddf(){
    HwiP_Params hwiPrms;
    int32_t status;

    /* Configure g_mux_en to 1 in ICSSG_SA_MX_REG Register. */
    HW_WR_REG32((CSL_PRU_ICSSG0_PR1_CFG_SLV_BASE+0x40), (0x80));

    gPruIcss0Handle = PRUICSS_open(CONFIG_PRU_ICSS0);
    /* Sigma Delta running in PRU0 */
    PRUICSS_disableCore(gPruIcss0Handle, PRUICSS_PRU0);
    PRUICSS_setSaMuxMode(gPruIcss0Handle, PRUICSS_SA_MUX_MODE_SD_ENDAT);
    PRUICSS_intcInit(gPruIcss0Handle, &gPruicssIntcInitdata);

    /* Register & enable ICSSG SDDF PRU FW interrupt */
    HwiP_Params_init(&hwiPrms);
    hwiPrms.intNum      = ICSSG_PRU_SDDF_INT_NUM;
    hwiPrms.callback    = &pruSddfIrqHandler;
    hwiPrms.args        = 0;
    hwiPrms.isPulse     = FALSE;
    hwiPrms.isFIQ       = FALSE;
    status              = HwiP_construct(&gIcssgSddfHwiObject, &hwiPrms);
    DebugP_assert(status == SystemP_SUCCESS);

    initPruSddf(gPruIcss0Handle, PRUICSS_PRU0, &gTestSddfPrms, &gHSddf);
}

/* This example shows message exchange between multiple cores.
 *
 * One of the core is designated as the 'main' core
 * and other cores are desginated as `remote` cores.
 *
 * The main core initiates IPC with remote core's by sending it a message.
 * The remote cores echo the same message to the main core.
 *
 * The main core repeats this for gMsgEchoCount iterations.
 *
 * In each iteration of message exchange, the message value is incremented.
 *
 * When iteration count reaches gMsgEchoCount, a semaphore is posted and
 * the pending thread/task on that core is unblocked.
 *
 * When a message or its echo is received, a user registered callback is invoked.
 * The message is echoed from within the user callback itself.
 *
 * This is a example message exchange, in final systems, user can do more
 * sophisticated message exchanges as needed for their applications.
 */

/* number of iterations of message exchange to do */
uint32_t gMsgEchoCount = 1000000u;
/* client ID that is used to send and receive messages */
uint32_t gClientId = 4u;
/* main core that starts the message exchange */
uint32_t gMainCoreId = CSL_CORE_ID_R5FSS1_0;
/* remote cores that echo messages from main core, make sure to NOT list main core in this list */
uint32_t gRemoteCoreId[] = {
    CSL_CORE_ID_R5FSS0_0,
    CSL_CORE_ID_M4FSS0_0,
    CSL_CORE_ID_MAX /* this value indicates the end of the array */
};

/* semaphore's used to indicate a main core has finished all message exchanges */
SemaphoreP_Object gMainDoneSem[CSL_CORE_ID_MAX];

/* semaphore used to indicate a remote core has finished all message xchange */
SemaphoreP_Object gRemoteDoneSem;

void ipc_notify_msg_handler_remote_core(uint32_t remoteCoreId, uint16_t localClientId, uint32_t msgValue, void *args)
{
    /* on remote core, we have registered handler on the same client ID and current core client ID */
    IpcNotify_sendMsg(remoteCoreId, localClientId, msgValue, 1);

    /* if all messages received then post semaphore to exit */
    if(msgValue == (gMsgEchoCount-1))
    {
        SemaphoreP_post(&gRemoteDoneSem);
    }
}

int32_t rw_drv8350_register(uint16_t r_w, uint16_t reg_addr, uint16_t reg_val, uint16_t* read_val)
{
    MCSPI_Transaction   spiTransaction;
    uint16_t mcspiTxBuffer, mcspiRxBuffer;
    int32_t transferOK;

    /* Initiate SPI transfer parameters */
    spiTransaction.channel  = 0U;
    spiTransaction.count    = 1;
    spiTransaction.txBuf    = &mcspiTxBuffer;
    spiTransaction.rxBuf    = &mcspiRxBuffer;
    spiTransaction.args     = NULL;

    mcspiTxBuffer = r_w | (reg_addr << DRV8350_ADDR_SHIFT) | reg_val;
    GPIO_pinWriteLow(MTR_1_CS_BASE_ADDR, MTR_1_CS_PIN);
    transferOK = MCSPI_transfer(gMcspiHandle[CONFIG_MCSPI0], &spiTransaction);
    GPIO_pinWriteHigh(MTR_1_CS_BASE_ADDR, MTR_1_CS_PIN);

    if (read_val != NULL)
        *read_val = mcspiRxBuffer;

    return transferOK;
}

int32_t init_drv8350()
{
    int32_t status = SystemP_SUCCESS;
    DRV8350_Vars drv8350;
    uint16_t reg_vals[6];

    /* Build the desired register values for the four control registers */
    drv8350.driver_control.all = 0;
    drv8350.gate_drive_hs.all = 0;
    drv8350.gate_drive_ls.all = 0;
    drv8350.ocp_control.all = 0;

    drv8350.driver_control.bit.OCP_ACT = DRV8350_OCP_disable_three_phases;

    drv8350.gate_drive_hs.bit.IDRIVEN_HS = DRV8350_idriveN_600mA;
    drv8350.gate_drive_hs.bit.IDRIVEP_HS = DRV8350_idriveP_300mA;
    drv8350.gate_drive_hs.bit.LOCK = DRV8350_lock_disable;

    drv8350.gate_drive_ls.bit.CBC = DRV8350_OCP_ClrFaults_Cycle_by_Cycle_Yes;
    drv8350.gate_drive_ls.bit.IDRIVEN_LS = DRV8350_idriveN_600mA;
    drv8350.gate_drive_ls.bit.IDRIVEP_LS = DRV8350_idriveP_300mA;
    drv8350.gate_drive_ls.bit.TDRIVE = DRV8350_tdrive_4000nS;

    drv8350.ocp_control.bit.DEAD_TIME = DRV8350_deadTime_50nS;
    drv8350.ocp_control.bit.OCP_MODE = DRV8350_OCP_Latch_Fault;
    drv8350.ocp_control.bit.OCP_DEG = DRV8350_deglitch_1us;
    drv8350.ocp_control.bit.VDS_LVL = DRV8350_VDS_LVL_1000mV;

    /* Write the control registers */
    rw_drv8350_register(DRV8350_WRITE, DRV8350_DRIVER_CONTROL_ADDR, drv8350.driver_control.all, NULL);
    rw_drv8350_register(DRV8350_WRITE, DRV8350_GATE_DRIVE_HS_ADDR, drv8350.gate_drive_hs.all, NULL);
    rw_drv8350_register(DRV8350_WRITE, DRV8350_GATE_DRIVE_LS_ADDR, drv8350.gate_drive_ls.all, NULL);
    rw_drv8350_register(DRV8350_WRITE, DRV8350_OCP_CONTROL_ADDR, drv8350.ocp_control.all, NULL);

    /* Read back the control registers to check for correctness */
    rw_drv8350_register(DRV8350_READ, DRV8350_DRIVER_CONTROL_ADDR, 0, &reg_vals[DRV8350_DRIVER_CONTROL_ADDR]);
    rw_drv8350_register(DRV8350_READ, DRV8350_GATE_DRIVE_HS_ADDR, 0, &reg_vals[DRV8350_GATE_DRIVE_HS_ADDR]);
    rw_drv8350_register(DRV8350_READ, DRV8350_GATE_DRIVE_LS_ADDR, 0, &reg_vals[DRV8350_GATE_DRIVE_LS_ADDR]);
    rw_drv8350_register(DRV8350_READ, DRV8350_OCP_CONTROL_ADDR, 0, &reg_vals[DRV8350_OCP_CONTROL_ADDR]);

    /* Check against desired values */
    if ((drv8350.driver_control.all != reg_vals[DRV8350_DRIVER_CONTROL_ADDR]) ||
        (drv8350.gate_drive_hs.all != reg_vals[DRV8350_GATE_DRIVE_HS_ADDR]) ||
        (drv8350.gate_drive_ls.all != reg_vals[DRV8350_GATE_DRIVE_LS_ADDR]) ||
        (drv8350.ocp_control.all != reg_vals[DRV8350_OCP_CONTROL_ADDR])) {
        status = SystemP_FAILURE;
        DebugP_log("[Single Chip Servo] DRV8350 configuration failed!\r\n");
    }
    else {
        DebugP_log("[Single Chip Servo] DRV8350 configured!\r\n");
    }

    /* Lock the control registers of the DRV8350 to ensure there are no accidental writes */
    drv8350.gate_drive_hs.bit.LOCK = DRV8350_lock_enable;
    rw_drv8350_register(DRV8350_WRITE, DRV8350_GATE_DRIVE_HS_ADDR, drv8350.gate_drive_hs.all, NULL);
    rw_drv8350_register(DRV8350_READ, DRV8350_GATE_DRIVE_HS_ADDR, 0, &reg_vals[DRV8350_GATE_DRIVE_HS_ADDR]);

    if (drv8350.gate_drive_hs.all == reg_vals[DRV8350_GATE_DRIVE_HS_ADDR])
        DebugP_log("[Single Chip Servo] DRV8350 Control registers locked!\r\n");

    return status;
}

int32_t check_adapter_board_power()
{
    int32_t status = SystemP_SUCCESS;

    /* Check the nRESET_SVS pin to ensure power on the 3-axis board is good */
    if(!GPIO_pinRead(NRESET_SVS_BASE_ADDR, NRESET_SVS_PIN)) {
        DebugP_log("[Single Chip Servo] Adapter board is not powered or 3.3V rail is out of specification.\r\n");
        status = SystemP_FAILURE;
    }
    else {
        DebugP_log("[Single Chip Servo] Adapter board power is good.\r\n");
    }

    return status;
}

void init_gpio_state()
{
    /* Set the initial GPIO output state before configuring direction. This avoids ambiguity on outputs. */
    GPIO_pinWriteHigh(MTR_1_CS_BASE_ADDR, MTR_1_CS_PIN);
    GPIO_pinWriteHigh(MTR_2_CS_BASE_ADDR, MTR_2_CS_PIN);
    GPIO_pinWriteHigh(MTR_3_CS_BASE_ADDR, MTR_3_CS_PIN);
    GPIO_pinWriteHigh(MTR_1_PWM_EN_BASE_ADDR, MTR_1_PWM_EN_PIN);
    GPIO_pinWriteHigh(MTR_2_PWM_EN_BASE_ADDR, MTR_2_PWM_EN_PIN);
    GPIO_pinWriteHigh(MTR_3_PWM_EN_BASE_ADDR, MTR_3_PWM_EN_PIN);
    GPIO_pinWriteLow(SDDF_DEBUG_BASE_ADDR, SDDF_DEBUG_PIN);
    GPIO_pinWriteLow(PWM_DEBUG_BASE_ADDR, PWM_DEBUG_PIN);

    GPIO_setDirMode(NRESET_SVS_BASE_ADDR, NRESET_SVS_PIN, NRESET_SVS_DIR);
    GPIO_setDirMode(MTR_1_CS_BASE_ADDR, MTR_1_CS_PIN, MTR_1_CS_DIR);
    GPIO_setDirMode(MTR_2_CS_BASE_ADDR, MTR_2_CS_PIN, MTR_2_CS_DIR);
    GPIO_setDirMode(MTR_3_CS_BASE_ADDR, MTR_3_CS_PIN, MTR_3_CS_DIR);
    GPIO_setDirMode(MTR_1_PWM_EN_BASE_ADDR, MTR_1_PWM_EN_PIN, MTR_1_PWM_EN_DIR);
    GPIO_setDirMode(MTR_2_PWM_EN_BASE_ADDR, MTR_2_PWM_EN_PIN, MTR_2_PWM_EN_DIR);
    GPIO_setDirMode(MTR_3_PWM_EN_BASE_ADDR, MTR_3_PWM_EN_PIN, MTR_3_PWM_EN_DIR);
    GPIO_setDirMode(SDDF_DEBUG_BASE_ADDR, SDDF_DEBUG_PIN, SDDF_DEBUG_DIR);
    GPIO_setDirMode(PWM_DEBUG_BASE_ADDR, PWM_DEBUG_PIN, PWM_DEBUG_DIR);
}

void misc_pinmux(){
    /* Sigma Delta pin mux */
    Pinmux_PerCfg_t miscPinMuxCfg[] = {
       /* PRG0_ECAP0_IN_APWM_OUT,
          PRG0_PRU1_GPO15, PRG0_ECAP0_IN_APWM_OUT, U5, J2.C11  */
       {
           PIN_PRG0_PRU1_GPO15,
           ( PIN_MODE(10) | PIN_PULL_DISABLE )
       },
       /* SD8_CLK,
          PRG0_PRU0_GPI16, SD8_CLK,    U4, J2E:P9 */
       {
           PIN_PRG0_PRU0_GPO16,
           ( PIN_MODE(1) | PIN_INPUT_ENABLE | PIN_PULL_DISABLE )
       },
       /* SD0_D,
         PRG0_PRU0_GPI1,   SD0_D,      R4, J2E:P8 */
       {
           PIN_PRG0_PRU0_GPO1,
           ( PIN_MODE(1) | PIN_INPUT_ENABLE | PIN_PULL_DISABLE )
       },
       /* SD1_D,
          PRG0_PRU0_GPI3m, SD1_D,      V2, J2A:P9  */
       {
           PIN_PRG0_PRU0_GPO3,
           ( PIN_MODE(1) | PIN_INPUT_ENABLE | PIN_PULL_DISABLE )
       },
       /* SD2_D,
          PRG0_PRU0_GPI5,  SD2_D,      R3, J2C:P6  */
       {
           PIN_PRG0_PRU0_GPO5,
           ( PIN_MODE(1) | PIN_INPUT_ENABLE | PIN_PULL_DISABLE )
       },
       {PINMUX_END, PINMUX_END}
    };

    Pinmux_config(miscPinMuxCfg, PINMUX_DOMAIN_ID_MAIN);
}

/* EPWM ISR */
__attribute__((section(".critical_code"))) static void App_epwmIntrISR(void *args)
{
    AppEPwmIsrInfo_t  *pAppEpwmIsrInfo;
    uint32_t epwmBaseAddr;
    SemaphoreP_Object *pEpwmSyncSemObject;
    volatile uint16_t status;
    uint32_t rem;
    float epwmPrdVal_f;
    uint32_t epwmPrdVal;

    GPIO_pinWriteHigh(PWM_DEBUG_BASE_ADDR, PWM_DEBUG_PIN);

    /* Get EPWM info */
    pAppEpwmIsrInfo = (AppEPwmIsrInfo_t *)args;
    epwmBaseAddr = pAppEpwmIsrInfo->epwmBaseAddr;
    pEpwmSyncSemObject = pAppEpwmIsrInfo->pEpwmSyncSemObject;

    status = Epwm_etIntrStatus(epwmBaseAddr);
    if (status & EPWM_ETFLG_INT_MASK) {
        if (gUpdOutIsr == TRUE) {
            gEpwmIsrCnt++;

            /* Check for EPWM period toggle */
            if (gToggleEpwmPrd == TRUE) {
                gEpwmPrdVal = (gToggleEpwmPrdState == 0) ? gEpwmPrdValL : gEpwmPrdValU;
                gToggleEpwmPrdState ^= 0x1;

                /* Write next period count */
                writeTbPrd(gEpwm0BaseAddr, gEpwmPrdVal);
                writeTbPrd(gEpwm1BaseAddr, gEpwmPrdVal);
                writeTbPrd(gEpwm2BaseAddr, gEpwmPrdVal);
            }

            /* Check for PWM frequency change */
            if (gEpwmOutFreq != gEpwmOutFreqSet) {
                /* Compute new EPWM period count,
                   divide by 2 for Up-Down Count. */
                //gEpwmPrdVal = APP_EPWM_TB_FREQ / gEpwmOutFreqSet / 2;
                epwmPrdVal_f = (float)EPWM0_FCLK / gEpwmOutFreqSet;
                epwmPrdVal_f = roundf(epwmPrdVal_f);
                epwmPrdVal = (uint32_t)epwmPrdVal_f;
                rem = epwmPrdVal - epwmPrdVal/2*2;
                if (rem == 0) {
                    /* Period is divisible by 2,
                       alternating period not employed */
                    gToggleEpwmPrd = FALSE;
                    gEpwmPrdVal = epwmPrdVal/2;
                } else {
                    /* Period is not divisible by 2,
                       alternating period employed to provide correct average EPWM frequency:
                        EPWM period 2*n     : TBPRD <- 'Lower' period
                        EPWM period 2*n+1   : TBPRD <- 'Upper' period
                    */
                    gToggleEpwmPrd = TRUE;
                    gToggleEpwmPrdState = 1;
                    gEpwmPrdValL = epwmPrdVal/2;
                    gEpwmPrdValU = epwmPrdVal/2+1;
                    gEpwmPrdVal = gEpwmPrdValL;
                }

                /* Write next period count */
                writeTbPrd(gEpwm0BaseAddr, gEpwmPrdVal);
                writeTbPrd(gEpwm1BaseAddr, gEpwmPrdVal);
                writeTbPrd(gEpwm2BaseAddr, gEpwmPrdVal);

                gEpwmOutFreq = gEpwmOutFreqSet;
            }
            GPIO_pinWriteLow(PWM_DEBUG_BASE_ADDR, PWM_DEBUG_PIN);
        }

        SemaphoreP_post(pEpwmSyncSemObject);
        Epwm_etIntrClear(epwmBaseAddr);
    }

    return;
}

void init_pwms(){
    int32_t         status;
    AppEPwmCfg_t    appEpwmCfg;
    HwiP_Params     hwiPrms;
    uint32_t        epwm0PrdVal, epwm1PrdVal, epwm2PrdVal;
    uint32_t        epwm0CmpAVal, epwm1CmpAVal, epwm2CmpAVal;

    /* Address translate */
    gEpwm0BaseAddr = (uint32_t)AddrTranslateP_getLocalAddr(EPWM0_BASE_ADDR);
    gEpwm1BaseAddr = (uint32_t)AddrTranslateP_getLocalAddr(EPWM1_BASE_ADDR);
    gEpwm2BaseAddr = (uint32_t)AddrTranslateP_getLocalAddr(EPWM2_BASE_ADDR);


    SOC_controlModuleUnlockMMR(SOC_DOMAIN_ID_MAIN, 1);
    /* Configure the SYNCI/SYNCO mapping to tie the three PWM groups together and have PWM0 SYNC from Time Sync Router 38 */
    CSL_REG32_WR(CSL_CTRL_MMR0_CFG0_BASE + CSL_MAIN_CTRL_MMR_CFG0_EPWM0_CTRL, (2 << CSL_MAIN_CTRL_MMR_CFG0_EPWM0_CTRL_SYNCIN_SEL_SHIFT));
    /* Time Sync Router input 29 (ICSSG1 IEP0 SYNC0) -> Time Sync Router output 38 (0x26 + 4 = 0x2A + Time Sync Router Base */
    CSL_REG32_WR(CSL_TIMESYNC_EVENT_INTROUTER0_CFG_BASE + ((38 * 4) + 4), (0x10000 | 29));

    /* Configure the ISR triggered by EPWM0 */
    /* Create semaphore */
    status = SemaphoreP_constructBinary(&gEpwm0SyncSemObject, 0);
    DebugP_assert(SystemP_SUCCESS == status);

    /* Register & enable EPWM0 interrupt */
    gAppEPwm0Info.epwmBaseAddr = gEpwm0BaseAddr;
    gAppEPwm0Info.pEpwmSyncSemObject = &gEpwm0SyncSemObject;
    HwiP_Params_init(&hwiPrms);
    hwiPrms.intNum      = EPWM0_INTR;
    hwiPrms.callback    = &App_epwmIntrISR;
    hwiPrms.args        = &gAppEPwm0Info;
    hwiPrms.isPulse     = EPWM0_INTR_IS_PULSE;
    status              = HwiP_construct(&gEpwm0HwiObject, &hwiPrms);
    DebugP_assert(status == SystemP_SUCCESS);

    /* Configure PWMs */
    /* Configure EPWM0 (Phase C) */
    appEpwmCfg.epwmBaseAddr = gEpwm0BaseAddr;
    appEpwmCfg.epwmCh = EPWM_OUTPUT_CH_A;
    appEpwmCfg.epwmFuncClk = EPWM0_FCLK;
    appEpwmCfg.epwmTbFreq = EPWM0_FCLK;
    appEpwmCfg.epwmOutFreq = gEpwmOutFreq;
    appEpwmCfg.epwmDutyCycle = 50;
    appEpwmCfg.epwmTbCounterDir = EPWM_TB_COUNTER_DIR_UP_DOWN;
    appEpwmCfg.cfgTbSyncIn = TRUE;
    appEpwmCfg.tbPhsValue = 7;
    appEpwmCfg.tbSyncInCounterDir = EPWM_TB_COUNTER_DIR_DOWN;
    appEpwmCfg.cfgTbSyncOut = TRUE;
    appEpwmCfg.tbSyncOutMode = EPWM_TB_SYNC_OUT_EVT_CNT_EQ_ZERO;
    appEpwmCfg.aqCfg.zeroAction = EPWM_AQ_ACTION_DONOTHING;
    appEpwmCfg.aqCfg.prdAction = EPWM_AQ_ACTION_DONOTHING;
    appEpwmCfg.aqCfg.cmpAUpAction = EPWM_AQ_ACTION_HIGH;
    appEpwmCfg.aqCfg.cmpADownAction = EPWM_AQ_ACTION_LOW;
    appEpwmCfg.aqCfg.cmpBUpAction = EPWM_AQ_ACTION_DONOTHING;
    appEpwmCfg.aqCfg.cmpBDownAction = EPWM_AQ_ACTION_DONOTHING;
    appEpwmCfg.cfgDb = TRUE;
    appEpwmCfg.dbCfg.inputMode = EPWM_DB_IN_MODE_A_RED_A_FED;
    appEpwmCfg.dbCfg.outputMode = EPWM_DB_OUT_MODE_A_RED_B_FED;
    appEpwmCfg.dbCfg.polaritySelect = EPWM_DB_POL_SEL_ACTV_HIGH_COMPLEMENTARY;
    appEpwmCfg.dbCfg.risingEdgeDelay = gDbRisingEdgeDelay;
    appEpwmCfg.dbCfg.fallingEdgeDelay = gDbFallingEdgeDelay;
    appEpwmCfg.cfgEt = TRUE;
    appEpwmCfg.intSel = EPWM_ET_INTR_EVT_CNT_EQ_PRD;
    appEpwmCfg.intPrd = EPWM_ET_INTR_PERIOD_FIRST_EVT;
    App_epwmConfig(&appEpwmCfg, &epwm0PrdVal, &epwm0CmpAVal);

    /* Configure EPWM1 (Phase B), EPWM2 (Phase A) */
    appEpwmCfg.tbPhsValue = 0;
    appEpwmCfg.tbSyncInCounterDir = EPWM_TB_COUNTER_DIR_UP;
    appEpwmCfg.cfgEt = FALSE;
    appEpwmCfg.epwmBaseAddr = gEpwm1BaseAddr;
    App_epwmConfig(&appEpwmCfg, &epwm1PrdVal, &epwm1CmpAVal);
    appEpwmCfg.epwmBaseAddr = gEpwm2BaseAddr;
    App_epwmConfig(&appEpwmCfg, &epwm2PrdVal, &epwm2CmpAVal);

    DebugP_assert(epwm0PrdVal == epwm1PrdVal);
    DebugP_assert(epwm1PrdVal == epwm2PrdVal);

    /* Period is the same for all EPWMs */
    gEpwmPrdVal = epwm0PrdVal;

    /* Force SW sync for EPWM0. Other PWMs will be sync'd through HW sync daisy-chain. */
    Epwm_tbTriggerSwSync(gEpwm0BaseAddr);

    /* Enable the PWM output buffer until gate driver configured and PWM signals at 50% duty cycle */
    enable_pwm_buffers(FALSE);
}

void init_pids(){
    /* 50KHz PWM frequency PID constants */
#if 1
    gPiId.Kp = 0.2;
    gPiId.Ki = 0.001;
    gPiId.Kd = 0;
    arm_pid_init_f32(&gPiId, 1);

    gPiIq.Kp = 0.2;
    gPiIq.Ki = 0.0005;
    gPiIq.Kd = 0;
    arm_pid_init_f32(&gPiIq, 1);

    /* Mmax 0.12 RPM step between periods */
    gPiSpd.Kp = 0.08;
    gPiSpd.Ki = 0.0003;
    gPiSpd.Kd = 0;
    arm_pid_init_f32(&gPiSpd, 1);

    /* Max 0.06 angle step between periods (500 RPM) */
    gPiPos.Kp = 30;
    gPiPos.Ki = 0;
    gPiPos.Kd = 0;
    arm_pid_init_f32(&gPiPos, 1);
#endif

    /* 20KHz PWM frequency PID constants */
#if 0
    gPiId.Kp = 0.225;
    gPiId.Ki = 0.005;
    gPiId.Kd = 0;
    arm_pid_init_f32(&gPiId, 1);

    gPiIq.Kp = 0.2;
    gPiIq.Ki = 0.005;
    gPiIq.Kd = 0;
    arm_pid_init_f32(&gPiIq, 1);

    /* Max 0.3 RPM step between periods */
    gPiSpd.Kp = 0.045;
    gPiSpd.Ki = 0.0001;
    gPiSpd.Kd = 0;
    arm_pid_init_f32(&gPiSpd, 1);

    /* Max 0.15 angle step between periods (500 RPM) */
    gPiPos.Kp = 13;
    gPiPos.Ki = 0.000025;
    gPiPos.Kd = 0;
    arm_pid_init_f32(&gPiPos, 1);
#endif
}

void single_chip_servo_remote_core_start()
{
    int32_t status;

#if (DEBUG_LEVEL == DEBUG_BUFFERS_ON)
    /* Make sure that the Debug buffer size is larger than the target buffer size so that there are no index out of bounds issues */
    DebugP_assert(DEBUG_BUFF_SZ > TARGET_BUFF_SIZE);
#endif

    /* Initialize GPIO state */
    init_gpio_state();

    /* Check the power on the adapter board */
    status = check_adapter_board_power();
    DebugP_assert(status==SystemP_SUCCESS);

    /* Disable the PWM output buffer until gate driver configured and PWM signals at 50% duty cycle */
    enable_pwm_buffers(FALSE);

    /* Initialize the DRV8350 gate driver */
    status = init_drv8350();
    DebugP_assert(status==SystemP_SUCCESS);

    misc_pinmux();

    /* Configure PWMs for complementary 3-phase and set duty cycle to 50% */
    init_pwms();

    init_encoder();
    init_sddf();

    init_pids();

#if (BUILDLEVEL == CLOSED_LOOP_CIA402)
    /* Default the CiA402 commands to torque control with a target value of 0 */
    gTxData.modeOfOperation = CYCLIC_SYNC_TORQUE_MODE;
    gTxData.targetPosition = 0;
    gTxData.targetVelocity = 0;
    gTxData.targetTorque = 0;
#endif

    /* Enable the PWM output buffers */
    enable_pwm_buffers(TRUE);

    SemaphoreP_constructBinary(&gRemoteDoneSem, 0);

    /* register a handler to receive messages */
    status = IpcNotify_registerClient(gClientId, ipc_notify_msg_handler_remote_core, NULL);
    DebugP_assert(status==SystemP_SUCCESS);

    /* wait for all cores to be ready */
    IpcNotify_syncAll(SystemP_WAIT_FOREVER);

    DebugP_log("[Single Chip Servo] All cores synchronized...\r\n");

    while(gRunFlag == TRUE)
        {
            SemaphoreP_pend(&gEpwm0SyncSemObject, SystemP_WAIT_FOREVER);
            gLoopCnt++;
        }

    DebugP_log("Exiting!\r\n");
}

void single_chip_servo_main(void *args)
{
    Drivers_open();
    Board_driversOpen();

    single_chip_servo_remote_core_start();

    Board_driversClose();
    /* We dont close drivers to let the UART driver remain open and flush any pending messages to console */
    /* Drivers_close(); */
}
