//#############################################################################
//! \file /F28377D_TeslaCLAHandsOn_Problem/main.c
//!
//! \brief  CLA Hands-on Workshop
//! \author Vishal Coelho
//! \date   May 10, 2016
//!
//! In this workshop we setup EPWM1A to run at 10 KHz (period = 0.1 ms) to
//! trigger a start-of-conversion on ADC channel A0. This channel will,
//! in turn, sample EPWM4A which is set to run at 1KHz.
//! At the end-of-conversion the ADC interrupt is fired. The interrupt signal
//! will be used to trigger a CLA task that runs an FIR filter. The filter
//! is designed to be low pass with a cutoff frequency of 1KHz; it will remove
//! the odd harmonics in the input signal smoothing the square wave to a
//! sinusoidal shape.
//!
//! \b External \b Connections \n
//!  - connect A0 (Pin 9) to EPWM4A (Pin 54)
//!
//! \b Watch \b Variables \n
//!  - None
//!
//  Group:             C2000
//  Target Family:     C28x + CLA
//
// (C)Copyright 2016, Texas Instruments, Inc.
//#############################################################################
//$TI Release: PACKAGE NAME $
//$Release Date: PACKAGE RELEASE DATE $
//#############################################################################

//*****************************************************************************
// includes
//*****************************************************************************
#include "cla_shared.h"
#include "F28x_Project.h"
#include "fpu_rfft.h"
#include <string.h>
#include "F2837xD_Cla_defines.h"

//*****************************************************************************
// defines
//*****************************************************************************
#define IMPULSE				(1 << 0)
#define LPF					(1 << 1)
#define USE_FILTER		  	IMPULSE

#define	RFFT_STAGES			10U
#define	RFFT_SIZE			(1 << RFFT_STAGES)

#define SYS_FREQ			200000000UL
#define EPWM_CLKDIV			32UL

#define EPWM1_FREQ			10000UL
#define EPWM4_FREQ			1000UL
#define EPWM1_PERIOD		(uint16_t)(SYS_FREQ/(EPWM_CLKDIV*EPWM1_FREQ))
#define EPWM4_PERIOD		(uint16_t)(SYS_FREQ/(EPWM_CLKDIV*EPWM4_FREQ))
#define EPWM4_DUTY_CYCLE	(EPWM4_PERIOD/2)

//*****************************************************************************
// globals
//*****************************************************************************
// FFT Variables
#pragma DATA_SECTION(IOBuffer,"IOBuffer");   //Buffer alignment for the input array,
float IOBuffer[RFFT_SIZE];                   //RFFT_f32u(optional), RFFT_f32(required)
                                             //Output of FFT overwrites input if 
                                             //RFFT_STAGES is ODD
#pragma DATA_SECTION(IOBuffer2,"IOBuffer");
float IOBuffer2[RFFT_SIZE];                  //Output of FFT here if RFFT_STAGES is EVEN

#pragma DATA_SECTION(RFFTmagBuff,"RFFTmag");
float RFFTmagBuff[RFFT_SIZE/2+1];            //Additional Buffer used in Magnitude calc
 
#pragma DATA_SECTION(RFFTF32Coef,"RFFTtwiddles");
float RFFTF32Coef[RFFT_SIZE];                 //Twiddle buffer

volatile uint16_t fftBufIndex = 0;            // index for the FFT buffer
RFFT_F32_STRUCT rfft; // The FFT object

// Filter variables
#if defined(C28_CODE)
volatile uint16_t runFFT = 0;				  // Flag set in the ISR to signal FFT computation
int16_t filter_in;
float filter_out;
float D[FILTER_LENGTH];
// Generated using fdatool
// LPF FIR Equiripple (Order = 99)
// Fs = 1 KHz
// Fp = 100 Hz
// Fs = 120 Hz
// Ap = 1dB
// As = 60dB
const float coeffs[FILTER_LENGTH] = {
#if USE_FILTER == LPF
  -0.001411611331,-0.002494140761,-0.003871886292,-0.004853332415,-0.004885642324,
  -0.003591896733,-0.0009863324231, 0.002419462427, 0.005700702779, 0.007805929519,
   0.007954596542, 0.005997032858, 0.002571551129,-0.001051861909,-0.003419536399,
   -0.00348134269,-0.001083443058, 0.002867734758, 0.006688602734, 0.008577809669,
   0.007407161873, 0.003307572333,-0.002240737667,-0.006916393992,-0.008508284576,
   -0.00593526056,0.0001190453913, 0.007291257847,  0.01239662617,  0.01277281065,
   0.007563922089,-0.001641549403, -0.01118932944, -0.01672186702, -0.01505199634,
  -0.005786245223, 0.008053471334,  0.02082409337,  0.02633191273,   0.0204805769,
   0.003468860174, -0.01943019591, -0.03901529685, -0.04502049461, -0.02979215793,
   0.008525118232,  0.06452515721,   0.1264797896,    0.179519549,   0.2100705355,
	 0.2100705355,    0.179519549,   0.1264797896,  0.06452515721, 0.008525118232,
   -0.02979215793, -0.04502049461, -0.03901529685, -0.01943019591, 0.003468860174,
	 0.0204805769,  0.02633191273,  0.02082409337, 0.008053471334,-0.005786245223,
   -0.01505199634, -0.01672186702, -0.01118932944,-0.001641549403, 0.007563922089,
	0.01277281065,  0.01239662617, 0.007291257847,0.0001190453913, -0.00593526056,
  -0.008508284576,-0.006916393992,-0.002240737667, 0.003307572333, 0.007407161873,
   0.008577809669, 0.006688602734, 0.002867734758,-0.001083443058, -0.00348134269,
  -0.003419536399,-0.001051861909, 0.002571551129, 0.005997032858, 0.007954596542,
   0.007805929519, 0.005700702779, 0.002419462427,-0.0009863324231,-0.003591896733,
  -0.004885642324,-0.004853332415,-0.003871886292,-0.002494140761,-0.001411611331
#else //IMPULSE
  1.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
  0.0, 0.0, 0.0, 0.0, 0.0,
#endif
};

#elif defined(CLA_CODE)
// <Step 1>: Assign the CLA variables to their respective memory spaces
//  - Input/Output variables should go into the Cla1ToCpuMsgRam/CpuToCla1MsgRam
//  - Global variables in the .cla file are automatically placed in the ".bss_cla" so
//    no manual placement is required.
//  - The same goes for constants in the .cla file; they are placed in the
//    ".const_cla" section but you will need to load it to FLASH and then copy
//    to RAM at runtime
//  - You can manually place shared variables in a user defined section, e.g. Cla1DataRam0,
//     to allow the C28x to also access them
//Task 1 (C) Variables
#pragma DATA_SECTION(<S1:REPLACE_ME>,"Cla1ToCpuMsgRAM");
<S1:REPLACE_ME>;
#pragma DATA_SECTION(<S1:REPLACE_ME>,"Cla1DataRam");
<S1:REPLACE_ME>
#endif

// Linker Defined variables
#if defined(CLA_CODE)
// <Step 2>: The CLA can only operate out of RAM i.e. its program and data spaces are restricted
//           to RAM. Depending on the device, the CLA may be configured to access the entire lower
//           64K of RAM (F2837x) or its prorgam and data spaces are restricted to certain RAMs
//           (F2806x, F2803x, F2805x program -> RAML3, data -> RAML0,1,/2)
//  - In standalone mode, you must load the CLA code and constants to FLASH  and then copy
//    these over to RAM at runtime
//  - The linker defines placement variables for the section "Cla1Prog" and ".const_cla"
//  - In order to use these variables in a memcpy() we need to tell the C compiler that
//    these variables are declared externally (by the linker). We use the "extern" storage
//    qualifier to do just that
extern uint32_t <S2:REPLACE_ME>;
extern uint32_t <S2:REPLACE_ME>;
#endif
//*****************************************************************************
// Function Prototypes
//*****************************************************************************
void initADC( void );
void initEPWM( void );
#if defined(C28_CODE)
__attribute__((interrupt))  void adca1intIsr( void );
#elif defined(CLA_CODE)
// Step 3: Setup the CLA in initCLA() and its memory resources in
//         configCLAMemory()
//  - see the initCLA() and configCLAMemory() for more steps
//  - add the End-of-Task 1 prototype here
void initCLA( void );
void configCLAMemory(void);
__attribute__((interrupt))  void cla1Isr1( void );
#endif // C28_CODE
//*****************************************************************************
// main
//*****************************************************************************
//!
//! \brief main routine for the cla hands on workshop example
//! \return returns a 1
//!
//! This example shows how to take a control loop in a C28x only environment
//! and migrate the code over to the CLA
//!
int main( void )
{
    // Locals
    //*************************************************************************
    // Initialize the system
    // Copy the ramfuncs section from Flash to RAM
    //*************************************************************************
    InitSysCtrl(); // will turn on all the peripheral clocks
    memcpy((uint32_t *)&RamfuncsRunStart, (uint32_t *)&RamfuncsLoadStart,
        (uint32_t)&RamfuncsLoadSize );
    InitFlash();
    InitGpio();
    DINT;						//Globally disable maskable CPU interrupts
	InitPieCtrl();				//Clear and disable all PIE interrupts
	IER = 0x0000;				//Individually disable maskable CPU interrupts
	IFR = 0x0000;				//Clear all CPU interrupt flags
	InitPieVectTable();			//Populate the PIE interrupt vector table with 
                                //shell ISRs
    //*************************************************************************
    // Setup the FFT
    //*************************************************************************
    //! \b Setup \b the \b FFT
    //! Initialize the elements of the FFT object, clear out the buffers and
    //! generate the twiddle factors in memory
    //! \note In this version, IOBuffer and IOBuffer2 are used in
    //!       ping-pong fashion. The input data is first stored in IOBuffer.
    //!       The FFT is then calculated, including bit reversing, and
    //!       when done, the cfft.CurrentInPtr pointer points to
    //!       buffer which has the result. Depending on the FFT size,
    //!       it will either be in IOBuffer or IOBuffer2.
    //*************************************************************************
    memset(&IOBuffer, 0, sizeof(IOBuffer));
    memset(&IOBuffer2, 0, sizeof(IOBuffer2));
    memset(&RFFTmagBuff, 0, sizeof(RFFTmagBuff));
    rfft.FFTSize   = RFFT_SIZE;
    rfft.FFTStages = RFFT_STAGES;   
    rfft.InBuf     = &IOBuffer[0];     //Input buffer
    rfft.OutBuf    = &IOBuffer2[0];    //Output buffer
    rfft.CosSinBuf = &RFFTF32Coef[0];  //Twiddle factor buffer
    rfft.MagBuf    = &RFFTmagBuff[0];  //Magnitude buffer

    RFFT_f32_sincostable(&rfft);       //Calculate twiddle factor
 
    //*************************************************************************
    // Setup the CLA, peripherals and interrupt service routines
    //*************************************************************************
    //! Setup the CLA, ADC and ePWM
    //*************************************************************************
#if defined(CLA_CODE)
    configCLAMemory();
    initCLA();
#endif
    initADC();

    EALLOW;
    CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 0;
    EDIS;
    initEPWM();
    EALLOW;
    CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 1;
    EDIS;
    

    EALLOW;
#if defined(C28_CODE)
    PieVectTable.ADCA1_INT         = &adca1intIsr;
	PieCtrlRegs.PIEIER1.bit.INTx1  = 0x01;
	IER |= (M_INT1);
#elif defined(CLA_CODE)
	// <Step 7> : We want task 1 to trigger an end-of-task interrupt in the PIE
	//            so we create an end-of-task1 interrupt service routine and we
	//            assign it to the PieVectTable (see TRM, Table 1-118)
	//   - Enable the PIE group and subgroup for that particular interrupt
    PieVectTable.<S7:REPLACE_ME>               = &<S7:REPLACE_ME>;
	PieCtrlRegs.PIEIER11.bit.<S7:REPLACE_ME>   = <S7:REPLACE_ME>;
	IER |= (<S7:REPLACE_ME>);
#endif
    EDIS;
    EINT;    // Enable Global interrupt INTM
	ERTM;    // Enable Global realtime interrupt DBGM

    //*************************************************************************
    // Running the FFT
    //*************************************************************************
    //! The FFT will be run each time 1024 samples have been accumulated in
	//! the input buffer by the ISR. The ADC or CLA end-of-task ISR, depending
	//! on which project configuration is active, will set "runFFT" to 1 each
	//! time the buffer is full
    //*************************************************************************
	for(;;){
#if defined(C28_CODE)
		if(runFFT == 1){    
			RFFT_f32(&rfft);		    // Calculate FFT
			RFFT_f32_mag_TMU0(&rfft);	// Calculate magnitude, result stored
			                            // in CurrentOutPtr
            runFFT = 0;                 // reset flag
		}
#else
		if(runFFT == 2){
			RFFT_f32_mag_TMU0(&rfft);	// Calculate magnitude, result stored
			                            // in CurrentOutPtr
            runFFT = 0;                 // reset flag
		}
#endif //C28_CODE
	}

    // Code never gets to this point. You can suppress the warning
    // this generates using a pragma
	#pragma  diag_suppress=112
    return 1;
}
// End of main

//*****************************************************************************
// function definitions
//*****************************************************************************
#if defined(CLA_CODE)
//! \brief Initialize the CLA
//! This function will
//! -# Initialize the MVECTs
//! -# setup task triggers
//! -# enable each individual task
//! -# run one-time initialization task
//! \note The CLA can only run code, access data out of RAM
//
    void initCLA( void ){
    	// <Step 4>: Now that the code and data is in place, the CLA needs to be able to
        //           jump to a particular task whenever its triggered - this is done through
        //           the MVECTx registers, 1 per task. The MVECTx register holds the OFFSET
        //           of the task from the start of programming memory (i.e. 0x9000)
        //           When a task is triggered this vector is loaded into the MPC (CLA program
        //           counter) and the CLA starts executing from that point
    	EALLOW;
    	Cla1Regs.MVECT1 = (uint16_t)(&<S4:REPLACE_ME>);
    	Cla1Regs.MVECT8 = (uint16_t)(&<S4:REPLACE_ME>);

    	// <Step 5> : Decide what triggers each task and which tasks are enabled
    	//  - The CLA1TASKSRCSEL(1/2) register has a 4-bit field for each task where the
    	//    user configures which peripheral will trigger the task. If you wish
    	//    to force a task in software, make sure you set its trigger source to
    	//    NONE. You can find a list of macros in "F2837xD_Cla_defines.h". Also
    	//    see section 5.2.4 and 2.14.7.3, 2.14.7.4 of the TRM for the register
    	//    description
    	// -  You will have to globally enable the tasks you want active through their
    	//    respective bits in the MIER (enable register). See 5.7.2.15 of the TRM
    	//    for a description of this register
    	DmaClaSrcSelRegs.CLA1TASKSRCSEL1.bit.TASK1 	= <S5:REPLACE_ME>;
    	DmaClaSrcSelRegs.CLA1TASKSRCSEL2.bit.TASK8 	= <S5:REPLACE_ME>;
    	Cla1Regs.MIER.all                           = <S5:REPLACE_ME>;

    	// <Step 6>: Enable software forcing and run task 8 once
    	//    - The MCTL register allows for a task to be triggered by software
    	//      see TRM 5.7.2.9
    	//    - Run any one-time tasks by using the software force. "F2806x_Cla_defines.h"
    	//      has some useful macro functions to force each task
    	Cla1Regs.MCTL.bit.<S6:REPLACE_ME>  = 1;
    	EDIS;
    	<S6:REPLACE_ME>();
    }

    //! \brief Configure the RAMs that will be the program and data spaces
    //! for the CLA
    //! This function will
    //! -# copy over code and const from flash to CLA program and data ram respectively
    //! -# map program and data spaces to the CLA
    //! \note The CLA can only run code, access data out of RAM
    //
    void configCLAMemory(void)
    {
    	// <Step 2>: use void * memcpy ( void * destination, const void * source, size_t num )
    	//           to copy over the "Cla1Prog" and ".const_cla" sections from Flash to RAM
        memcpy(<S2:REPLACE_ME>);
        memcpy(<S2:REPLACE_ME>);
        EALLOW;

    	// Initialize and wait for CLA1ToCPUMsgRAM
    	MemCfgRegs.MSGxINIT.bit.INIT_CLA1TOCPU = 1;
    	while(MemCfgRegs.MSGxINITDONE.bit.INITDONE_CLA1TOCPU != 1){};

    	// Initialize and wait for CPUToCLA1MsgRAM
    	MemCfgRegs.MSGxINIT.bit.INIT_CPUTOCLA1 = 1;
    	while(MemCfgRegs.MSGxINITDONE.bit.INITDONE_CPUTOCLA1 != 1){};

    	// <Step 3> : Give CLA control over program and data RAM(s)
    	//    - The MemCfgRegs register is described in TRM 2.14.17
    	// Configure LS0RAM and LS1RAM as program spaces for the CLA
    	// First configure the CLA to be the master for LS0(1) and then
        // set the spaces to be program blocks
    	MemCfgRegs.LSxMSEL.bit.MSEL_LS0 = <S3:REPLACE_ME>;
    	MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS0 = <S3:REPLACE_ME>;

    	MemCfgRegs.LSxMSEL.bit.MSEL_LS1 = <S3:REPLACE_ME>;
    	MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS1 = <S3:REPLACE_ME>;

    	//Next configure RAMLS2 to RAMLS5 as data spaces for the CLA
    	// First configure the CLA to be the master for each block and then
        // set the spaces to be data blocks
    	MemCfgRegs.LSxMSEL.bit.MSEL_LS2 = <S3:REPLACE_ME>;
    	MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS2 = <S3:REPLACE_ME>;

    	MemCfgRegs.LSxMSEL.bit.MSEL_LS3 = <S3:REPLACE_ME>;
    	MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS3 = <S3:REPLACE_ME>;

    	MemCfgRegs.LSxMSEL.bit.MSEL_LS4 = <S3:REPLACE_ME>;
    	MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS4 = <S3:REPLACE_ME>;

    	MemCfgRegs.LSxMSEL.bit.MSEL_LS5 = <S3:REPLACE_ME>;
    	MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS5 = <S3:REPLACE_ME>;

    	EDIS;
    }
#endif

//! \brief Configure the ADC
//! The ADC start-of-conversion is trigger by EPWM1SOCA at a rate
//! of EPWM1_FREQ Hz. Channel A0 will sample EPWM4A continuously. At the
//! end of conversion it will assert ADCA_INT1 which will trigger
//! either a C28x ISR or a CLA task depending on the project
//! configuration
//
    void initADC( void ){
        EALLOW;
    
        //write configurations
        AdcaRegs.ADCCTL2.bit.PRESCALE = 6; //set ADCCLK divider to /4
        AdcSetMode(ADC_ADCA, ADC_RESOLUTION_12BIT, ADC_SIGNALMODE_SINGLE);
        
        //Set pulse positions to late
        AdcaRegs.ADCCTL1.bit.INTPULSEPOS = 1;

        //power up the ADC
        AdcaRegs.ADCCTL1.bit.ADCPWDNZ = 1;
    
        //delay for 1ms to allow ADC time to power up
        DELAY_US(1000);
        
        //
        // ADC interrupt comes at the end of conversion
        // EOC0 will trigger ADCA1_INT
        // Enable ADCA_INT1
        // Enable ADCA_INT1 Continuous mode
        //
        AdcaRegs.ADCINTSEL1N2.bit.INT1SEL   = 0;
        AdcaRegs.ADCINTSEL1N2.bit.INT1E     = 1;
        AdcaRegs.ADCINTSEL1N2.bit.INT1CONT  = 1;

        //
        //determine minimum acquisition window (in SYSCLKS) based on 
        // resolution
        //
        if(ADC_RESOLUTION_12BIT == AdcaRegs.ADCCTL2.bit.RESOLUTION){
            AdcaRegs.ADCSOC0CTL.bit.ACQPS    = 14; //75ns
        }                                    
        else { //resolution is 16-bit        
            AdcaRegs.ADCSOC0CTL.bit.ACQPS    = 63; //320ns
        }

        //
        // set SOC0 channel select to ADCINA0
        // set SOC0 start trigger on EPWM1A interrupt  	    
        //
        AdcaRegs.ADCSOC0CTL.bit.CHSEL    = 0;
        AdcaRegs.ADCSOC0CTL.bit.TRIGSEL  = 5;

        EDIS;
    }
    
//! \brief Configure the EPWMs
//! EPWM1A will run at EPWM1_FREQ Hz and serves as the sampling clock for ADC channel
//! A1 which will be sampling the slower running EPWM4A (at EPWM4_FREQ Hz)
//! The default time base for the EPWM module is half the system clock
//! i.e. TBCLK = SYSCLKOUT/2
//! EPWM1A will be setup in count-up mode and an event generated every period
//! this event will trigger the ADC to start sampling on channel A1
//! EPWM4A is setup for updown count mode with CMPA level at the half period
//! point giving a 50% duty cycle square wave at EPWM4_FREQ Hz
//! For a desired PWM frequency F_pwm (1/T_pwm), we have
//! T_pwm = 2 x TBPRD / TBCLK
//! TBPRD = TBCLK/(2*F_pwm)
//! For e.g. F_pwm = 10KHz
//! TBPRD = 45e6/(2*10e3)
//!       = 2250
//! F_pwm = 1KHz
//! TBPRD = 45e6/(2*1e3)
//!       = 22500
//! For e.g. F_pwm = 100Hz
//! TBPRD = 45e6/(2*10e2)
//!       = 225000
//
    void initEPWM( void ){
        //
        // EALLOW: is needed to write to EALLOW protected registers
        // EDIS: is needed to disable write to EALLOW protected registers   
        //
        // Enable start of conversion (SOC) on A
        // An SOC event will occur when the ePWM counter is zero
        // The ePWM will generate an SOC on the first event
        // 
        // Set the period for ePWM1  
        // By default TBPRD = 1/2 SYSCLKOUT 
        // Set the counter for up count mode
        //
    	EPwm1Regs.TBCTL.bit.CLKDIV	= 4;  // CLKDIV = /16 HSPCLKDIV =/2
    	                                  // TBCLK = SYSCLK/32
        EPwm1Regs.TBPRD 			= EPWM1_PERIOD;
        EPwm1Regs.TBCTL.bit.CTRMODE = TB_COUNT_UP;	   
        EPwm1Regs.ETSEL.bit.SOCAEN	= 1;	   
        EPwm1Regs.ETSEL.bit.SOCASEL	= ET_CTR_ZERO;	   
        EPwm1Regs.ETPS.bit.SOCAPRD 	= ET_1ST;	   

        //
        // For testing - monitor the EPWM1A output
        // which toggles once every ePWM period (i.e
        // at the start of conversion)
        //
        EALLOW;
        GpioCtrlRegs.GPAMUX1.bit.GPIO0 = 1;
        EDIS;
        EPwm1Regs.AQCTLA.bit.ZRO = AQ_TOGGLE;
        EPwm1Regs.TBCTL.bit.FREE_SOFT = 3;     

        //
        // Disable the timer (counter mode is halt)
        // Set the free/soft emulation bits to ignore the
        // emulation halt (PWM will continue to count when
        // the CPU is halted)
        //

        EPwm4Regs.TBCTL.bit.CTRMODE = 0x3;		
        EPwm4Regs.TBCTL.bit.FREE_SOFT = 3; 

        //
        // Clear the counter
        // Set the period and timer phase
        // Specify when the compare A event will occur
        //
    	EPwm4Regs.TBCTL.bit.CLKDIV	= 4;  // CLKDIV = /16 HSPCLKDIV =/2
    	                                  // TBCLK = SYSCLK/32
        EPwm4Regs.TBCTR = 0x0000;			   
        EPwm4Regs.TBPRD = EPWM4_PERIOD;
        EPwm4Regs.TBPHS.bit.TBPHS = 0x0000;
        EPwm4Regs.CMPA.bit.CMPA = EPWM4_DUTY_CYCLE;

        //
        // On compare A, when counting up, pull the EPWM A output high
        // On compare A, when counting down, pull the EPWM A output low
        // Set the counter to up/down count mode
        //
        EPwm4Regs.AQCTLA.bit.CAU = AQ_SET;
        EPwm4Regs.AQCTLA.bit.CAD = AQ_CLEAR;
        EPwm4Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;	
    
        // EALLOW: is needed to write to EALLOW protected registers
        // EDIS: is needed to disable write to EALLOW protected registers
        // Configure the GPIO6 pin to be EPWM4A output
        EALLOW;
        GpioCtrlRegs.GPAMUX1.bit.GPIO6 = 1;
        EDIS;
    }
//*****************************************************************************
// ISR
//*****************************************************************************
#if defined(C28_CODE)
#pragma CODE_SECTION(adca1intIsr, "ramfuncs")
__attribute__((interrupt))  void adca1intIsr()
{
	//**********************
	// <<LOW_PASS_FIR_FILTER
	//**********************
	//Local Variables
    int16_t i;
    filter_out = 0.0;
    filter_in  = AdcaResultRegs.ADCRESULT0 - (1 << 11); // Read ADC value minus bias
    
    //Shift the delay line to the right by 1
    #pragma UNROLL(FILTER_LENGTH-1)
    for(i = FILTER_LENGTH-1; i > 0; i--){
        D[i] = D[i-1];
        //Multiply coefficients with the delay line and accumulate
        filter_out   += coeffs[i] * D[i];
    }
    
    //Get adc input into start of the delay line
    //note the adc input is simulated in software
    D[0] = (float)filter_in; //convert to single precision float
    filter_out  += coeffs[0] * D[0];
    //**********************
    // LOW_PASS_FIR_FILTER>>
    //**********************
    
    // store the filtered output
    IOBuffer[fftBufIndex++] = filter_out;
    if(fftBufIndex == RFFT_SIZE){
    // if the full FFT size is reached, set the runFFT flag
        runFFT = 1;
        fftBufIndex = 0;
    }
    // Acknowledge the end-of-task interrupt for ADCA_INT1
	PieCtrlRegs.PIEACK.all = M_INT1;
}

#elif defined(CLA_CODE)
#pragma CODE_SECTION(cla1Isr1, "ramfuncs")
__attribute__((interrupt))  void cla1Isr1 ()
{
	// Clear the ADC interrupt flag so the next SOC can occur
	AdcaRegs.ADCINTFLGCLR.bit.ADCA_INT1 = 1;

    // store the filtered output
    <S9:REPLACE_ME>
	// Acknowledge the end-of-task interrupt for task 1
	PieCtrlRegs.PIEACK.all = M_INT11;
//	asm(" ESTOP0");
}
#endif
