#!/usr/local/bin/perl

# Description:
#
#   This script generates a pwl statements for tx_even and tx_odd to drive the TLK10002
#   low-speed transmitter model.  By default it generates 1000 bits worth of pattern but
#   the length can be defined by the user.
#
#   The user can either request one of three PRBS patterns (prbs7, prbs23 or prbs31) or 
#   can specify an arbitrary pattern in a text file containing 1's, 0's and white-space.
#   The format of this file is arbitrary and all white-space will be ignored.
#
#   If the requested pattern is less than the requested number of bits to generate, the
#   pattern will be repeated
#
# Required arguments:
#
#   -p pattern where pattern is prbs7, prbs23, prbs31 or the name of a file containing the pattern
#
#   -p outputfile : The script writes the pwl statements to outputfile
#
# Optional arguments
#
#   -n pattern_length : The script will generate pattern_length bits worth of pwl statement
#                       If not specified, pattern_length defaults to 1000.
#
#   -s seed : This parameter is only relevant with prbs pattern generation.  By default, the
#             prbs generator starts with a seed of all 1's.  If some other portion of a long 
#             prbs sequence is desired a different seed can be specified via this parameter.
#             The seed must unly contain 1's and 0's and be exactly equal in length to the
#             prbs sequence requested with the -p parameter.  For example, if -p prbs23 is 
#             used the seed must be 23 characters long.
#

# setup defaults
$tprop = "0";
$Nbits = 1000;
@seed = ();
@bits = ();

# process arguments
while (@ARGV) {
  if ($ARGV[0] eq "-p") {
    $pattern_file = shift(@ARGV);
    $pattern_file = shift(@ARGV);
  } elsif ($ARGV[0] eq "-n") {
    $Nbits = shift(@ARGV);
    $Nbits = shift(@ARGV);
  } elsif ($ARGV[0] eq "-o") {
    $outfile = shift(@ARGV);
    $outfile = shift(@ARGV);
  } elsif ($ARGV[0] eq "-s") {
    $tmpseed = shift(@ARGV);
    $tmpseed = shift(@ARGV);
    @seed = split(//, $tmpseed);
    ($#seed == 6) || ($#seed == 22) || ($#seed == 30) || die "Only seeds of length 7, 23, or 31 are supported\n";
    foreach $tmp (@seed) {
      ($tmp eq "0") || ($tmp eq "1") || die "Seed must be string of 0's and 1's without any other characters\n";
    }          
  } else {
    die "Option $ARGV[0] not defined\n";
  }
}

if ($pattern_file eq "prbs7") {
  &gen_prbs7;
} elsif ($pattern_file eq "prbs23") { 
  &gen_prbs23;
} elsif ($pattern_file eq "prbs31") { 
  &gen_prbs31;
} else {  
  (-e $pattern_file) || die "Pattern file $pattern_file not found\n";
  # read pattern file 
  open(IN_P, "$pattern_file");
  while (<IN_P>) {
    s/\s+//g;
    @tmp = split(//);
    foreach $bit (@tmp) {
      if (($bit eq "0") || ($bit eq "1")) {
        push (@bits, $bit);
      }
    }
  }
  close(IN_P);
}

$patlen = $#bits + 1;
print "Found $patlen bits in pattern file\n";

open(OUT_P, ">$outfile");

print OUT_P "* TLK10002 stimulus file for LSTX HSPICE model\n\n";
print OUT_P "*   Generated using bit pattern in $pattern_file\n\n";

# Main cursor output
@pins = ("tx_even", "tx_odd");
for ($i=0; $i<2; $i++) {
  $bp = $#bits - 1 + $i;
  $bpn = $i;
  print OUT_P "v$pins[$i] $pins[$i] 0 pwl 0 '$bits[$bp]*vdda_val'\n";
  for ($n=0; $n<$Nbits; $n+=2) {
    print OUT_P "+ '(2+$n)*tbit+$tprop-30p' '$bits[$bp]*vdda_val' '(2+$n)*tbit+$tprop' '$bits[$bpn]*vdda_val'\n";
    $bp = $bpn;
    $bpn = ($bpn + 2) % $patlen;
  }
}


sub gen_prbs7 {
  if (!@seed) {
    for ($i=0; $i<7; $i++) {
      $seed[$i] = 1;
    }
  }
  ($#seed == 6) || die "Invalid seed length of $#seed for prbs7\n";
  
  for ($i=0; $i<$Nbits; $i++) {
    $nb = ($seed[6] == $seed[5]) ? 0 : 1;
    push (@bits, $nb);
    for ($j=6; $j>0; $j--) {
      $seed[$j] = $seed[$j-1];
    }
    $seed[0] = $nb;
  }
}

sub gen_prbs23 {
  if (!@seed) {
    for ($i=0; $i<23; $i++) {
      $seed[$i] = 1;
    }
  }
  ($#seed == 22) || die "Invalid seed length of $#seed for prbs23\n";
  
  for ($i=0; $i<$Nbits; $i++) {
    $nb = ($seed[22] == $seed[17]) ? 0 : 1;
    push (@bits, $nb);
    for ($j=22; $j>0; $j--) {
      $seed[$j] = $seed[$j-1];
    }
    $seed[0] = $nb;
  }
}

sub gen_prbs31 {
  if (!@seed) {
    for ($i=0; $i<31; $i++) {
      $seed[$i] = 1;
    }
  }
  ($#seed == 30) || die "Invalid seed length of $#seed for prbs31\n";
  for ($i=0; $i<$Nbits; $i++) {
    $nb = ($seed[30] == $seed[27]) ? 0 : 1;
    push (@bits, $nb);
    for ($j=30; $j>0; $j--) {
      $seed[$j] = $seed[$j-1];
    }
    $seed[0] = $nb;
  }
}

          
  
