//***************************************************************************
//
// File:        buffman.c
//
// Description: This file contains the code for buffer management
//
// Author:      Anne Kelley
//
// Copyright 1996, 1997, Counterpoint Systems Foundry, Inc.
// Portions also copyright 1996, 1997, Alpha Omega Computer Systems, Inc.
// All rights reserved worldwide
//
//***************************************************************************
#define WANTVXDWRAPS

#include <basedef.h>                             
#include <vmm.h>
#include <vcomm.h>
#include "port.h"
#include "common.h"
#include "buffman.h"
#include "dma.h"

#pragma VxD_LOCKED_CODE_SEG
#pragma VxD_LOCKED_DATA_SEG

#ifndef WIN31COMPAT
  #ifdef  DEBUG
    void    DebugPrintf(char *Format, ...);
    #define VPRINTF(args) DebugPrintf args
  #else   // Release
    #define VPRINTF(args)
  #endif  // Release
#else   // WIN31COMPAT
  #ifdef DEBUG
    void DbgPrint(char *afmt, ...);
    #define DebugPrint(arg) DbgPrint arg
    #define VPRINTF(arg) DbgPrint arg
  #else //release
    #define DebugPrint(arg)
    #define VPRINTF(arg)
  #endif //release
#endif  // WIN31COMPAT


/******************************************************/
/*  RemoveBuf                                         */
/*  Removes a buffer from the specified queue.        */
/*                                                    */
/*  Inputs:   *pQueue     The buffer queue            */
/*  Outputs:  buffer      The buffer that was removed */
/******************************************************/

Buffer *RemoveBuf (BufQueue *pQueue)
{
  Buffer    *buf;
  U32       flags;


  EnterCritical (flags);
  if (pQueue->head != 0)
  {
    buf = pQueue->head;
    pQueue->head = buf->next;
    if (pQueue->head == 0)
      pQueue->tail = 0;

    if (buf->ownedBy != pQueue)
      VPRINTF(("SERIAL [RemoveBuf] %08lx owned by %08lx, not %08lx\n", buf, buf->ownedBy, pQueue));

    buf->ownedBy = NULL;  // Not owned
  }
  else
    buf = 0;

  ExitCritical (flags);
  
  return (buf);
}

/******************************************************/
/*  AppendBuf                                         */
/*  Removes a buffer from the specified queue.        */
/*                                                    */
/*  Inputs:   *pQueue     The buffer queue            */
/*            buffer      The buffer to append        */
/******************************************************/

void AppendBuf (BufQueue *pQueue, Buffer *buf)
{
  U32   flags;

  if (buf != 0)
  {
    buf->next = 0;
    EnterCritical (flags);
    if (pQueue->tail != 0) 
		{
      // Not the first buffer in list
      pQueue->tail->next = buf;
		}
    else
		{
      // first buffer in list
      pQueue->head = buf;
		}


    pQueue->tail = buf;

    ExitCritical (flags);

    if (buf->ownedBy != NULL)
    {
      VPRINTF(("SERIAL [AppendBuf] %08lx already owned by %08lx, now %08lx\n",
              buf, buf->ownedBy, pQueue));
    }

    buf->ownedBy = pQueue;  // Remember who now "owned" it
  }
}

//************************************************************************
//  InitBuf                                                            
//  Initializes buffers and queues
//                                                                      
//  Inputs:   queue
//            buffer
//            Number of buffers.
//  Outputs:  None                                                      
//

void InitBuf (BufQueue *pQueue, Buffer buf[], U8 *data, int nBuffers, void *dev)
{
  U16   count;

  pQueue->head = 0;
  pQueue->tail = 0;
  for (count = 0; count < nBuffers; count++)
  {
    if (data != NULL)
    {
      buf[count].buf = data+BufSize*count;
      buf[count].dev = dev;
      buf[count].length = BufSize;
      buf[count].numbBytes = 0;
      buf[count].ownedBy = NULL;
    }
    else
    {
      // Building holding queue - no data attached
      buf[count].buf = NULL;
      buf[count].dev = dev;
      buf[count].length = 0;
      buf[count].numbBytes = 0;
      buf[count].ownedBy = NULL;
    }
    AppendBuf (pQueue, &buf[count]);
  }
}
