//***************************************************************************
//
// File:        dma.c
//
// Description: This file contains the code for DMA operations
//
// Author:      Anne Kelley
//
// Copyright 1996, 1997, Counterpoint Systems Foundry, Inc.
// Portions also copyright 1996, 1997, Alpha Omega Computer Systems, Inc.
// All rights reserved worldwide
//
//***************************************************************************

#define WANTVXDWRAPS

#include <basedef.h>
#include <vmm.h>
#include <vcomm.h>
#include <debug.h>
#include <..\common.h>
// #include "vardef.h"
#include "port.h"
#include "hsdriver.h"
#include "dma.h"
#include "irq.h"
#include <vxdwraps.h>                                           
#ifdef WIN31COMPAT
#include "intern31.h"
#else
#include "internal.h"
#endif

#pragma VxD_LOCKED_CODE_SEG
#pragma VxD_LOCKED_DATA_SEG

#ifndef WIN31COMPAT
  #ifdef  DEBUG
    void    DebugPrintf(char *Format, ...);
    #define VPRINTF(args) DebugPrintf args
  #else   // Release
    #define VPRINTF(args)
  #endif  // Release
#else   // WIN31COMPAT
  #ifdef DEBUG
    void DbgPrint(char *afmt, ...);
    #define DebugPrint(arg) DbgPrint arg
    #define VPRINTF(arg) DbgPrint arg
  #else //release
    #define DebugPrint(arg)
    #define VPRINTF(arg)
  #endif //release
#endif  // WIN31COMPAT


// ## Debug for Winice
U32 WIdebugFlag = 0;
// ## End of debug

U8        *MemAddress;         // Base address of DMA memory space     
U16 BaseAddr[]  = { 0x00, 0x02, 0x04, 0x06, 0xC0, 0xC4, 0xC8, 0xCC };
U16 CountReg[]  = { 0x01, 0x03, 0x05, 0x07, 0xC2, 0xC6, 0xCA, 0xCE };
U16 MaskReg[]   = { 0x0A, 0x0A, 0x0A, 0x0A, 0xD4, 0xD4, 0xD4, 0xD4 };
U16 ModeReg[]   = { 0x0B, 0x0B, 0x0B, 0x0B, 0xD6, 0xD6, 0xD6, 0xD6 };
U16 ClearFF[]   = { 0x0C, 0x0C, 0x0C, 0x0C, 0xD8, 0xD8, 0xD8, 0xD8 };
U16 PageAddr[]  = { 0x87, 0x83, 0x81, 0x82, 0x00, 0x8B, 0x89, 0x8A };

//**********************************************************
//* IoDelay
//* makes a small delay for i/o accesses
//**********************************************************

int IoDelay(void) 
{
  int n;
  n = 0;
  n += 1;
  return n; 
}

//**********************************************************************
//  Virtualize                                                        
//  Virtualizes the interrupts                                        
//                                                                    
//  Input:  IR device data structure.                                                      
//  Output: Returns 0 if the interrupt controller could not be virtualized.
//                                                                    

U16 Virtualize (HSControl *HSControlData)
{

  static VPICD_IRQ_DESCRIPTOR irlapdIRQDescriptor;
  
  irlapdIRQDescriptor.vidEOIProc    = (U32) VIRLAPD_EoiProc;
  irlapdIRQDescriptor.vidIRETProc   = (U32) VIRLAPD_IretProc;
  irlapdIRQDescriptor.vidOptions    = VPICD_Opt_Can_Share;
  if (HSControlData->dmaMode)
  {
    /* In DMA mode, set up the DMA interrupt service routine */
    irlapdIRQDescriptor.vidHwIntProc = (U32) VIRLAPD_HwIntProcDMA;
  }
  else
  {
    /* In PIO mode, set up the PIO interrupt service routine */
    irlapdIRQDescriptor.vidHwIntProc  = (U32) VIRLAPD_HwIntProcPIO;
  }
  irlapdIRQDescriptor.vidIRQNumber  = (U8) HSControlData->irq;
  return vpicdVirtualizeIRQ (&irlapdIRQDescriptor);
}


//*************************************************************************
// InitDMAXfer                                                          
// Initialize DMA to prepare for a data transfer.                       
//                                                                      
// Inputs:      DMA channel for transfer,                               
//              Mode: Mode Register bits 7 to 2                         
//              Address and Size of buffer to transfer                  
// Outputs:     Address of the first byte of the buffer being sent                                                   
//                                                                      

U16 InitDMAXfer (U32  ChannelNum,
				  U32 Mode,
				  char *BufferAddr,
				  U32  BufferSize)

{
  U16            SimpleAddr;
  U16            DMA_Page;
  U32            physAddr;
  U16            startAddress;       // debug kludge

#if 1
  physAddr = LinToPhysAddress ( (U32) BufferAddr );
#else
  physAddr = BufferAddr;
#endif

//  VPRINTF (("InitDMAXfer:  buffer size = %d\n", BufferSize));


  DMA_Page = HiU16 (physAddr);
  SimpleAddr = LoU16 (physAddr);
  startAddress = SimpleAddr;

  // mask the DMA channel
  outportb (MaskReg[ChannelNum], (U8)(SET_MASK_BIT | (ChannelNum & 3)));
  IoDelay ();
  outportb (ClearFF[ChannelNum], 0);
  IoDelay ();

  // set up buffer address for DMA controller
  outportb (BaseAddr[ChannelNum], (U8)(SimpleAddr & 0xFF));            /* low byte */
  IoDelay ();
  outportb (BaseAddr[ChannelNum], (U8)((SimpleAddr >> 8) & 0xFF));     /* high byte */
  IoDelay ();
  outportb (PageAddr[ChannelNum], (U8) DMA_Page);
  IoDelay ();
  // Set up #bytes to transmit for DMA controller
  outportb (CountReg[ChannelNum], (U8)(BufferSize & 0xFF));            /* low byte */
  IoDelay ();
  outportb (CountReg[ChannelNum], (U8)((BufferSize >> 8) & 0xFF));     /* high byte */
  IoDelay ();

  outportb (ClearFF[ChannelNum], 0);   // clear byte pointer flip/flop
  IoDelay ();


  // Set up mode register on DMA controller
	outportb (ModeReg[ChannelNum], (U8)((Mode & 0xFC) | (ChannelNum & 3)));  // here it works
  IoDelay ();

  return (startAddress);

} /* InitDMAXfer */



// ************************************************************************************
// StartDMAXfer                                                         
// Start transfer initialized for channel passed as parameter by unmasking that channel.
//                                                                      
// Inputs:      DMA channel                                             
// Outputs:     none                                                    
//                                                                      

void StartDMAXfer (U32 ChannelNum)
{ 
	outportb (MaskReg[ChannelNum], (U8)(RESET_MASK_BIT | (ChannelNum & 3)));
}


/***********************************************************************************
 * Disable the dma (don't change current settings)
 * Inputs:    The DMA channel to disable
 * Outputs:   None
 ***********************************************************************************/

void DisableDMA(U32 ChannelNum)
{
  DebugPrint (("SERIAL [DisableDMA]\n"));
  outportb(MaskReg[ChannelNum], (U8)(SET_MASK_BIT | (ChannelNum& 3)));
}



//************************************************************************
// NumbBytesRemaining                                                   
// Returns "Current Word Count" for channel.                            
// !!! the returned value = number or bytes still to transfer - 1       
//
// Inputs:      DMA channel                                             
// Outputs:     "Current Word Count"                                    
//                                                                      

U16 NumbBytesRemaining (U32 ChannelNum)
{
  U32 flags;
  U16 Temp;
     
  EnterCritical (flags);
  outportb (ClearFF[ChannelNum], 0);
  IoDelay ();
  Temp = inportb (CountReg[ChannelNum]);
  IoDelay ();
  Temp = Temp + ((U16) (inportb (CountReg[ChannelNum])) << 8);
  IoDelay ();
  ExitCritical (flags);

  return (Temp);


} /* NumbBytesRemaining */
                                                                        

