//***************************************************************************
//
// File:        fastser.c
//
// Description: This file contains the serial.vxd harness
//
// Author:      Anne Kelley
//
// Copyright 1996, 1997, Counterpoint Systems Foundry, Inc.
// Portions also copyright 1996, 1997 Alpha Omega Computer Systems, Inc.
// All rights reserved worldwide
//
//***************************************************************************


/*lint -e714 */
/*lint -e715 */
/*lint -e766 */
#define WANTVXDWRAPS

#include <basedef.h>                             
#include <vmm.h>
#include <vtd.h>
#include <vcomm.h>
#include <vmmreg.h>
#include <debug.h>
#include <vxdwraps.h>
#include <vwin32.h>
#include "port.h"
#include "common.h"
#include "hsdriver.h"
#include "dma.h"
#include "irdaprnt.h"
/*lint +libh(internal.h) */
#ifdef WIN31COMPAT
#include "intern31.h"
#else
#include "internal.h"
#endif

#pragma VxD_LOCKED_CODE_SEG
#pragma VxD_LOCKED_DATA_SEG

#ifndef WIN31COMPAT
  #ifdef  DEBUG
    void    DebugPrintf(char *Format, ...);
    #define VPRINTF(args) DebugPrintf args
  #else   // Release
    #define VPRINTF(args)
  #endif  // Release
#else   // WIN31COMPAT
  #ifdef DEBUG
    void DbgPrint(char *afmt, ...);
    #define DebugPrint(arg) DbgPrint arg
    #define VPRINTF(arg) DbgPrint arg
  #else //release
    #define DebugPrint(arg)
    #define VPRINTF(arg)
  #endif //release
#endif  // WIN31COMPAT


#define	DL_READ			0x0001
#define	DL_WRITE		0x0002
#define	DL_INFO			0x0004
#define	DL_NOTES		0x0008
#define	DL_TIMING		0x0010
#define	DL_VERBOSE	0x0020

//#define DEBUGLEVEL DL_INFO
//#define DEBUGLEVEL DL_TIMING

#ifndef DEBUGLEVEL
#define	DEBUGLEVEL	(DL_VERBOSE|DL_INFO|DL_NOTES)
#endif


//***************************************************************************
//
// Local data

#ifdef WIN31COMPAT
/* Windows 3.11 variables and procedure prototypes */
extern U32 IrPortNum;  // IrPortNum is the port number of the IR port. COM1 is 1 COM2 is 2, etc.

PU8 _stdcall Get_Profile_String(PU8 section, PU8 keyName, PU8 def);
int _stdcall Get_Profile_Decimal_Int(PU8 section, PU8 keyName, int def);

#define MAX_STRING 32
static PU8 section = "irlapfrm";

#else
/* Windows 95 variables */
#define NAME_KEY						HKEY_LOCAL_MACHINE
#define FRAMER_SUBKEY				"Enum\\Infrared\\Framer"
#endif

#define MAX_PORT_NAME_LEN		64
#define PORT_NAME						"HsPortName"
#define DMA1_NAME           "DMA1Channel"
#define DMA2_NAME           "DMA2Channel"
#define DEVICE_TYPE         "DevType"


static char HsPortName[MAX_PORT_NAME_LEN] = "COM3"; 
static char dma1str[MAX_PORT_NAME_LEN] = "1";
static char dma2str[MAX_PORT_NAME_LEN] = "ff";
static char deviceTypeStr[MAX_PORT_NAME_LEN] = "TIR2000";
static DWORD errorStatus = 0;

static BufQueue   RdQueue;  // Queue that holds the read data to pass
                            // back to the framer.

void StoreTxSize (U32 TxThreshold, U32 RxThreshold, U32 FIFOSize);

BOOL StrCmpPort(PPortInformation hPort, char *portName);

hsResults ProcessRdBuffer (HSControl *dev, U8 *buffer, U32 length,
                         hsResults status);
hsResults ProcessWrBuffer (HSControl *dev, hsResults status);


#ifndef WIN31COMPAT

//***************************************************************************
// IR_Init
// This function is called when the VxD is first loaded. This would be a good
// time to find out where the Fast IR resources are located. Memory should
// not be allocated here since there is no corresponding deinit call.
//
// Information found here should be stored in static variables.

VOID IR_Init(void)
{
  DWORD     dwLen;
  DWORD     dwType;
  VMMHKEY   hKey;
  U32       temicDongle;
  U8        dmaMode;
  U32       FIFOsize;
  U32       RxThreshold;
  U32       TxThreshold;
  U8        lowPowerEnable;
  U8        sleepEnable;
  U8        FCRdata;
  U8        PLRdata;

  // Get the name of the com port where IR is attached.

#if DEBUGLEVEL && DL_VERBOSE
  DebugPrint (("SERIAL [IR_Init]\n"));
#endif
  if (_RegOpenKey(NAME_KEY,FRAMER_SUBKEY,&hKey) == 0) 
  {
		// Query for the name that is available for Hi Speed action
    dwLen = MAX_PORT_NAME_LEN;
    _RegQueryValueEx(hKey,PORT_NAME,NULL,&dwType,HsPortName,&dwLen);

    // Query for device type
    dwLen = MAX_PORT_NAME_LEN;
    _RegQueryValueEx(hKey,DEVICE_TYPE,NULL,&dwType,deviceTypeStr,&dwLen);

    // Query for dma channel 1
    dwLen = MAX_PORT_NAME_LEN;
    errorStatus = _RegQueryValueEx(hKey,DMA1_NAME,NULL,&dwType,dma1str,&dwLen);
#if DEBUGLEVEL && DL_INFO
  	VPRINTF (("IR_Init: errorStatus = %ld\n", errorStatus));
#endif

    // Query for dma channel 2
    dwLen = MAX_PORT_NAME_LEN;
    _RegQueryValueEx(hKey,DMA2_NAME,NULL,&dwType,dma2str,&dwLen);

    // must get the fifo size, receive threshold and transmit threshold to modify the 
    // low speed IR assembly code.  This data gets used and set before IR_PortInit gets
    // called.
    parseDevType (deviceTypeStr, &temicDongle, &dmaMode, &FIFOsize, &RxThreshold,
                &TxThreshold, &lowPowerEnable, &sleepEnable, &FCRdata, &PLRdata);
    StoreTxSize (TxThreshold, RxThreshold, FIFOsize);
  }


#if DEBUGLEVEL & DL_NOTES
  VPRINTF (("SERIAL [IR_Init] IR high speed port name is %s\n", HsPortName));
  VPRINTF (("SERIAL [IR_Init] DMA1 = %s\n", dma1str));
  VPRINTF (("SERIAL [IR_Init] dma2 = %s\n", dma2str));
  VPRINTF (("SERIAL [IR_Init] DEVTYPE = %s\n", deviceTypeStr));
  DebugPrint (("SERIAL [IR_Init] IR port name is %s\n", HsPortName));
#endif // DEBUGLEVEL
}

#endif //not win31 compatible



//***************************************************************************
//  IR_PortInit
//    This function is called to initialize each port handled by this driver.
//    This is the place to allocate memory and associate it with a port 
//    information structure.
//
//  Inputs:   Communication port information.
//  Outputs:  None
//

VOID IR_PortInit(PPortInformation portInfo)
{
  int       comPort, irq, dma1, dma2;
  HSControl *controlData;
  char      *p;
  
#ifdef WIN31COMPAT
  PU8  dma1str;
  PU8  dma2str;
  PU8  deviceTypeStr;

  deviceTypeStr = Get_Profile_String (section, DEVICE_TYPE, "TIR2000");
  DebugPrint (("SERIAL: Device type = %s\n", deviceTypeStr));
  dma1str = Get_Profile_String (section, DMA1_NAME, "1");
  DebugPrint (("SERIAL: dma 1 = %s\n", dma1str));
  dma2str = Get_Profile_String (section, DMA2_NAME, "ff");
  DebugPrint (("SERIAL: dma 2 = %s\n", dma2str));
#endif

  DebugPrint (("SERIAL: IR_PortInit\n"));


#if DEBUGLEVEL & DL_VERBOSE
  VPRINTF (("SERIAL [IR_PortInit]\n"));
  DebugPrint (("SERIAL [IR_PortInit]\n"));
#endif // DEBUGLEVEL

#ifndef WIN31COMPAT
  VPRINTF (("SERIAL [IR_PortInit] portname = %s\n", portInfo->MyName));
  if (StrCmpPort(portInfo, HsPortName))
  {
#endif
    // Indicate that this port is the IR port
#if DEBUGLEVEL & DL_NOTES
    VPRINTF (("SERIAL [IR_PortInit] Initializing the IR port with name %s\n", HsPortName));
    DebugPrint (("SERIAL [IR_PortInit] Initializing the IR port with name %s\n", HsPortName));
#endif // DEBUGLEVEL

  	if (errorStatus == 0)
	  	dma1 = xstrtol (dma1str, &p, 10);
	  else
	  {
  		VPRINTF (("No dma channel value\n"));
	  	dma1 = 0xff;
	  }

    dma2 = xstrtol (dma2str, &p, 10);

    irq = portInfo->IRQn;
    comPort = portInfo->Port;
    VPRINTF (("comPort = %x\n", comPort));

    RdQueue.head = 0;
    RdQueue.tail = 0;

    portInfo->IR_Flags = IR_FLAGS_IR_PORT;
    controlData = HSOpen (comPort, irq, dma1, dma2, deviceTypeStr, portInfo);
    VPRINTF (("SERIAL [IR_PortInit] controlData = %x\n", controlData));
    
    (HSControl *)portInfo->HSControlBlock = controlData;
    VPRINTF (("SERIAL [IR_PortInit] dev = %lx\n", portInfo->HSControlBlock));

#if DEBUGLEVEL & DL_NOTES
    VPRINTF (("SERIAL [IR_PortInit] dma1 = %d\n", dma1));
    VPRINTF (("SERIAL [IR_PortInit] dma2 = %d\n", dma2));
    VPRINTF (("SERIAL [IR_PortInit] dmaChannel = %d\n", controlData->dmaChannel));
    VPRINTF (("SERIAL [IR_PortInit] irq = %d\n", irq));
    VPRINTF (("SERIAL [IR_PortInit] portInfo->IRQn = %d\n", portInfo->IRQn));

    DebugPrint (("SERIAL [IR_PortInit]  DMA data address = %lx\n", MemAddress));
    DebugPrint (("SERIAL [IR_PortInit] dma1 = %d\n", dma1));
    DebugPrint (("SERIAL [IR_PortInit] dma2 = %d\n", dma2));
    DebugPrint (("SERIAL [IR_PortInit] dmaChannel = %d\n", controlData->dmaChannel));
    DebugPrint (("SERIAL [IR_PortInit] irq = %d\n", irq));
#endif // DEBUGLEVEL
#ifndef WIN31COMPAT
  }
  else
    (HSControl *)portInfo->HSControlBlock = 0;

  VPRINTF (("SERIAL [IR_PortInit] hscontrolblock = %x\n", (HSControl *)portInfo->HSControlBlock));

  if ((HSControl *)portInfo->HSControlBlock == 0)
  {
    // Indicate that this port is a normal (non-IR) port.

#if DEBUGLEVEL & DL_NOTES
    VPRINTF (("SERIAL [IR_PortInit] Initializing non IR port\n"));
    DebugPrint (("SERIAL [IR_PortInit] Initializing non IR port\n"));
#endif // DEBUGLEVEL
    portInfo->IR_Flags = 0;
  }
#endif
  DebugPrint (("SERIAL [IR_PortInit] dev = %lx\n", portInfo->HSControlBlock));
  DebugPrint (("SERIAL [IR_PortInit] portInfo = %lx\n", portInfo));
  DebugPrint (("SERIAL [IR_PortInit] portInfo->pData = %lx\n", portInfo->pData));
  DebugPrint (("SERIAL [IR_PortInit] portInfo->DCB = %lx\n", portInfo->DCB));
  DebugPrint (("SERIAL [IR_PortInit] portInfo->XOffPoint = %lx\n", portInfo->XOffPoint));
  DebugPrint (("SERIAL [IR_PortInit] portInfo->IntVecNum = %lx\n", portInfo->IntVecNum));
  DebugPrint (("SERIAL [IR_PortInit] portInfo->IRQn = %x\n", portInfo->IRQn));
  DebugPrint (("SERIAL [IR_PortInit] portInfo->MyIRQStruc = %lx\n", portInfo->MyIRQStruc));
  DebugPrint (("SERIAL [IR_PortInit] portInfo->OwnerVM = %lx\n", portInfo->OwnerVM));
  return;
}

#ifndef WIN31COMPAT

//***************************************************************************
//  IR_PortDeinit
//    This function is called to deinitialize each port. If memory was
//    allocated for this port then it should be deallocated here.
//
//  Inputs:   Communication port information.
//  Outputs:  None
//

VOID IR_PortDeinit(PPortInformation portInfo)
{
  hsResults   success;

#if DEBUGLEVEL & DL_VERBOSE
  DebugPrint (("SERIAL [IR_PortDeinit]\n"));
  VPRINTF (("SERIAL [IR_PortDeinit]\n"));
#endif // DEBUGLEVEL

  if (portInfo->IR_Flags & IR_FLAGS_IR_PORT)
  {
#if DEBUGLEVEL & DL_NOTES
    VPRINTF (("SERIAL [IR_PortDeinit]  Deinitializing IR port\n"));
    DebugPrint (("SERIAL [IR_PortDeinit]  Deinitializing IR port\n"));
#endif // DEBUGLEVEL
    DebugPrint (("SERIAL [IR_PortDeinit] dev = %lx\n", portInfo->HSControlBlock));
    DebugPrint (("SERIAL [IR_PortDeinit] portInfo = %lx\n", portInfo));
    DebugPrint (("SERIAL [IR_PortDeinit] portInfo->pData = %lx\n", portInfo->pData));
    DebugPrint (("SERIAL [IR_PortDeinit] portInfo->DCB = %lx\n", portInfo->DCB));
    DebugPrint (("SERIAL [IR_PortDeinit] portInfo->XOffPoint = %lx\n", portInfo->XOffPoint));
    DebugPrint (("SERIAL [IR_PortDeinit] portInfo->IntVecNum = %lx\n", portInfo->IntVecNum));
    DebugPrint (("SERIAL [IR_PortDeinit] portInfo->IRQn = %hx\n", portInfo->IRQn));
    DebugPrint (("SERIAL [IR_PortDeinit] portInfo->MyIRQStruc = %lx\n", portInfo->MyIRQStruc));
    DebugPrint (("SERIAL [IR_PortDeinit] portInfo->OwnerVM = %lx\n", portInfo->OwnerVM));
    success = HSClose ((HSControl *)portInfo->HSControlBlock);
  }
}

#endif  // not win31 compatible

#ifdef WIN31COMPAT
//***************************************************************************
//
// IR_PortOpen
//   This function is called when the application calls VCOMM_OpenComm. If
//   for some reason Fast IR will not work now may be the time to fail
//   the open. If FALSE is returned the port open fails. Resources can
//   be allocated if needed. The check to see if the port is valid and Not
//   being used has already been made. The interrupt has not been acquired yet.

BOOL IR_PortOpen(PPortInformation hPort)
{
  BOOL    handleValid;

  DebugPrint(("IR_PortOpen. port num = %d\n", hPort->DEBId));

  HSGetStatus ((HSControl *)hPort->HSControlBlock, STS_HsGetHandle, 
                       &handleValid);
  DebugPrint (("ir_portOpen, handlevalid = %ld\n", handleValid));

  if (!handleValid)
    return FALSE;

  return TRUE;
}

//***************************************************************************
//
// IR_PortClose
//   This function is called when the application calls VCOMM_CloseComm.
//   The port has not been closed yet.

VOID IR_PortClose(PPortInformation hPort)
{
  hsResults   success;

  DebugPrint (("IR_PORTDEINIT: IR_PortDeinit\n"));
  
  if (hPort->IR_Flags & IR_FLAGS_IR_PORT)
  {
    DebugPrint (("IR_PORTDEINIT:  Deinitializing IR port\n"));
    success = HSClose ((HSControl *)hPort->HSControlBlock);
  }
}
#endif


//***************************************************************************
//  IR_CheckSpeed
//    This function is called whenever an application calls VCOMM_SetCommState.
//    This is the hook for determining if the hPort->IR_Flags should be set.
//
//    Return TRUE if you do not want the normal setCommState code to execute
//    otherwise return FALSE. This function is called before the setCommState
//    code has a chance to run.
//
//  Inputs:   Communication port information
//            Data control block.
//            ActionMask ?
//  Outputs:  Returns TRUE for high speed communications.
//            

BOOL IR_CheckSpeed(PPortInformation hPort, _DCB *pDcb, UINT ActionMask)
{
  hsResults success;
  BOOL      hiSpeed;
  U32       TxThreshold;


#if (DEBUGLEVEL & DL_VERBOSE) || (DEBUGLEVEL & DL_VERBOSE)
  VPRINTF (("SERIAL [IR_CheckSpeed]\n"));
  DebugPrint (("SERIAL [IR_CheckSpeed]\n"));
  VPRINTF (("IR_CheckSpeed:  mode = %x\n", inportb (hPort->Port + 8)));
#endif // DEBUGLEVEL

  // If this is the IR port then deal with it

  if (hPort->IR_Flags & IR_FLAGS_IR_PORT)
  {
#if DEBUGLEVEL & DL_NOTES
    VPRINTF (("SERIAL [IR_CheckSpeed] on IR port\n"));
    DebugPrint (("SERIAL [IR_CheckSpeed] on IR port\n"));
#endif // DEBUGLEVEL

    // Get the speed mode
    HSGetStatus ((HSControl *)hPort->HSControlBlock, STS_HsGetSpeedMode, &hiSpeed);

    VPRINTF (("IR_CheckSpeed:  TxFifoTrigger = %x\n", TxThreshold));
		if (pDcb->BaudRate > 115200)
    {
      HSSetControl ((HSControl *)hPort->HSControlBlock, CTL_resetRegisters, &TxThreshold);
      outportb (hPort->Port + 1, 0);    // disable interrupts from low speed.
#if DEBUGLEVEL & DL_NOTES
      VPRINTF (("SERIAL [IR_CheckSpeed] Baud - %ld\n", pDcb->BaudRate));
#endif // DEBUGLEVEL
      if (!hiSpeed)
      {
        // changing from low speed to high speed.
#if DEBUGLEVEL & DL_NOTES
        VPRINTF (("SERIAL [IR_CheckSpeed] High speed\n"));
        DebugPrint (("SERIAL [IR_CheckSpeed] High speed\n"));
#endif // DEBUGLEVEL
        DeInitSpeedLo (hPort);
        Virtualize ((HSControl *)hPort->HSControlBlock);  // Set up interrupt handler
                                                          // for high speed
      }

      success = HSSetControl ((HSControl *)hPort->HSControlBlock, CTL_HSBaud, pDcb->BaudRate);
      success = HSRead ((HSControl *)hPort->HSControlBlock, &ProcessRdBuffer);

      if (success != STS_OK)
        return (TRUE); /* Highspeed part is off- make it appear ok */

      if (!hiSpeed)
      {
        hPort -> IR_Flags |= IR_FLAGS_ENABLED;
      }

#if DEBUGLEVEL & DL_VERBOSE
      VPRINTF (("SERIAL [IR_CheckSpeed] returning TRUE\n"));
#endif // DEBUGLEVEL

      return (TRUE);
    }
    else
    {
      if (hiSpeed)
      {
        // changing to low speed.
#if DEBUGLEVEL & DL_NOTES
        VPRINTF (("SERIAL [IR_CheckSpeed] switching to low speed\n"));
        DebugPrint (("SERIAL [IR_CheckSpeed] switching to low speed\n"));
#endif // DEBUGLEVEL

        hPort -> IR_Flags &= ~IR_FLAGS_ENABLED;   // disable Ir flags.

        outportb (hPort->Port+1, 0);
        ioDelay ();
        // Turn on low speed port
        vpicdForceDefaultBehavior ();
        InitSpeedLo (hPort);

        HSSetControl ((HSControl *)hPort->HSControlBlock, CTL_resetRegisters, &TxThreshold);
        success = HSSetControl ((HSControl *)hPort->HSControlBlock, CTL_HSBaud, pDcb->BaudRate);

      }
      HSSetControl ((HSControl *)hPort->HSControlBlock, CTL_resetRegisters, &TxThreshold);
#if (DEBUGLEVEL & DL_NOTES) || (DEBUGLEVEL & DL_INFO)
      VPRINTF (("SERIAL [IR_CheckSpeed] baud = %ld, Port=%x IRQn = %d\n",
								pDcb->BaudRate, hPort->Port, hPort->IRQn));
      VPRINTF (("IR_CheckSpeed:  mode = %x\n", inportb (hPort->Port + 8)));
//      outportb (hPort->Port + 8, 1);
      outportb (hPort->Port + 9, 0x80);
      outportb (hPort->Port + 0xa, 26);
      VPRINTF (("IR_CheckSpeed:  mode = %x, presc = %x\n", inportb (hPort->Port + 8), 
                  inportb (hPort->Port+0xa)));
      
#endif  //DEBUGLEVEL
      return (FALSE);
    }
  }
  VPRINTF (("IR_CheckSpeed:  mode = %x\n", inportb (hPort->Port + 8)));
  outportb (hPort->Port + 8, 1);
  VPRINTF (("IR_CheckSpeed:  mode = %x\n", inportb (hPort->Port + 8)));
  return (FALSE);	// Not a high-speed port.
}


//***************************************************************************
//  ProcessRdBuffer
//    Callback function to process the receive queue.
//
//  Inputs:   High speed control block structure.
//            Buffer containing the received data
//            The number of bytes received.
//            Status of the received data
//  Outputs:  Returns status of the processed data.
//            

hsResults ProcessRdBuffer (HSControl *dev, U8 *buffer, 
                           U32 length, hsResults status)
{
  U32         eventMask;
  Buffer      *rdBuffer;


  // Put the received data in a buffer that is accessible to IR_PortRead
  rdBuffer = RemoveBuf (&(dev->freeQueue));

  eventMask = *(dev->portInfo->AddrEvtDWord);
 
  if (rdBuffer != 0)
  {
		rdBuffer->status = status;

		// Limit to actual length of buffer
		if (length > rdBuffer->length)
			length = rdBuffer->length;

		rdBuffer->numbBytes = length;

		// !! Remember when we queued it.
		rdBuffer->timerStart = GetRealTime();

    if ((status == STS_OK) && buffer)
    {
			memCopy(rdBuffer->buf, buffer, length);

      *(dev->portInfo->AddrEvtDWord) |= EV_RXCHAR;
#if 0
//#if DEBUGLEVEL & DL_NOTES
	    VPRINTF (("EV_RXCHAR\n"));
#endif
    }
    else
    {
      *(dev->portInfo->AddrEvtDWord) |= EV_ERR;
	    VPRINTF (("EV_ERR\n"));
    }


    AppendBuf (&RdQueue, rdBuffer);

    NotifyFramer (dev->portInfo, eventMask);


    return STS_OK;
  }
  else
    return STS_FAIL;
} // ProcessRdBuffer




//***************************************************************************
//  ProcessWrBuffer
//    Callback function to process the transmit queue.
//
//  Inputs:   High speed control block structure.
//            Status of the transmitted data
//  Outputs:  Returns status of the processed data.
//            

hsResults ProcessWrBuffer (HSControl *dev, hsResults status)
{

	if (dev->wrFinish->lastFrame)
	{
		U32 eventMask = *(dev->portInfo->AddrEvtDWord);
		*(dev->portInfo->AddrEvtDWord) |= EV_TXEMPTY;

		
    NotifyFramer (dev->portInfo, eventMask);
	}


  return STS_OK;
} //ProcessWrBuffer


//***************************************************************************
//  IR_PortRead
//  This function is called by IrLapFrm to retrieve a completed frame. It should
//  return nothing by setting numBRead to 0 if called when there are no valid
//  frames available. Frames with bad FCS should not be returned.
//
//  If lpBuf is 0 then the code is trying to flush any existing frames. No data
//  should be copied but the next available frame should be discarded and the
//  the size returned in NumBRead.
//
//  When a frame is received IrLapFrm assumes that an event notification will be
//  called inside the interrupt service routine (see irasm.asm for an example). This
//  event callback will trigger a call to PortRead.
//
//  IrLapFrm also uses the event callback to single media change and set some
//  activity flags. Therefore even frames with Bad FCSs should trigger the event
//  callback even though a call to PortRead will return 0.
//
//  Inputs:   Communication port information
//            buffer to store received data
//            Number of characters to read
//            Number of characters read
//  Outputs:  Returns TRUE all the time.
//

BOOL IR_PortRead(PPortInformation hPort, char *lpBuf, U32 NumBToRead, U32 *lpNumBRead)
{
  Buffer *buf;
#if defined DEBUG
  U32	n;
#endif

	*lpNumBRead = 0;	// Assume the worst

	// Loop until we find a non-zero length buffer (i.e. no errors) or we run out of buffers
	do
	{
		buf = RemoveBuf (&RdQueue);

		if (buf != 0)
		{
			U32		nBytes = NumBToRead;
#if defined(DEBUG) && (DEBUGLEVEL & DL_TIMING)
			unsigned int		n;
#endif // DEBUG && DEBUGLEVEL

			// If we want more than is available, limit to what is in buffer.
			if (nBytes > buf->numbBytes)
				nBytes = buf->numbBytes;

			if (buf->status == 0)
				*lpNumBRead = nBytes;

			if ((lpBuf != 0) && (nBytes != 0) && (buf->status == 0))
			  memCopy (lpBuf, buf->buf, nBytes);

#if defined(DEBUG) && (DEBUGLEVEL & DL_INFO)
			// Display debug info of what actually was delivered.
      VPRINTF (("SERIAL [IR_PortRead] # bytes read %d [", nBytes));
      DebugPrint (("SERIAL [IR_PortRead] # bytes read %d [", nBytes));

			for (n = 0; ((n < buf->numbBytes) && (n < 8)); ++n)
      {
				VPRINTF ((" %02x", lpBuf[n] & 0xFF));
				DebugPrint ((" %02x", lpBuf[n] & 0xFF));
      }
      VPRINTF (("/"));
      DebugPrint (("/"));
      for (n = buf->numbBytes-19; ((n >= 0) && (n < buf->numbBytes)); ++n)
      {
        VPRINTF ((" %02x", lpBuf[n] & 0xFF));
        DebugPrint ((" %02x", lpBuf[n] & 0xFF));
      }

//			if (buf->numbBytes > 8)
//				VPRINTF ((" ..."));

      VPRINTF ((" ]\n"));
      DebugPrint ((" ]\n"));
#endif // DEBUG && DEBUGLEVEL

			// Give this buffer back to the device that owns it.
		  AppendBuf (&(((HSControl *)buf->dev)->freeQueue), buf);
		}
	} while ((*lpNumBRead == 0) && buf);

  return (TRUE);
}



//***************************************************************************
//  IR_PortWrite
//  This function is called by the IrLapFrm to send a frame. Each call to
//  PortWrite is to send one frame. The number of BOFs to send is found in
//  hPort->IR_BOFs. It was set via a call to VCOMM_TransmitCommChar. If the
//  the number of BOFs are 0 or 1 then just send a normal frame.
//
//  IrLapFrm assumes that when the packet is sent it will be called back
//  via the notification callback (see irasm.asm for example).
//  
//  Inputs:   Communications port information
//            Buffer holding the data to transmit.
//            Number of characters to send
//            Number of characters sent
//  Outputs:  Returns FALSE if the frame to transmit is invalid.
//

BOOL IR_PortWrite(PPortInformation hPort, char *lpBuf, U32 NumBWrite, U32 *lpNumBWritten)
{

  U32   n;

#if 0
//#if DEBUGLEVEL & DL_VERBOSE
  VPRINTF (("SERIAL [IR_PortWrite]\n"));
#endif // DEBUGLEVEL

#if 01
//#if defined(DEBUG) && (DEBUGLEVEL & DL_INFO)
			// Display debug info of what actually was delivered.
  VPRINTF (("SERIAL [IR_PortWrite] # to write %ld ", NumBWrite));
  DebugPrint (("SERIAL [IR_PortWrite] # to write %ld \n", NumBWrite));
  for (n = 0; ((n < NumBWrite) && (n < 8)); ++n)
  {
    VPRINTF ((" %02x", lpBuf[n] & 0xFF));
    DebugPrint ((" %02x", lpBuf[n] & 0xFF));
  }
  VPRINTF (("/"));
  DebugPrint (("/"));

#if 01
  for (n = NumBWrite-1; ((n >= 0) && (n > (NumBWrite - 9))); --n)
  {
    VPRINTF ((" %02x", lpBuf[n] & 0xFF));
    DebugPrint ((" %02x", lpBuf[n] & 0xFF));
  }
  VPRINTF (("\n"));
  DebugPrint (("\n"));
#endif

#endif // DEBUG && DEBUGLEVEL

#if DEBUGLEVEL & DL_TIMING
	// !! See how long it's been since last write
	VPRINTF(("SERIAL [IR_PortWrite] %lu ticks, %d bytes [%02x %02x...] %s\n",
						 GetElapsedTime(), NumBWrite,
						 lpBuf[0] & 0xFF, lpBuf[1] & 0xFF, decodeIrda(lpBuf)));
#endif // DEBUGLEVEL

#ifdef TREAT_EVERY_PACKET_AS_A_STANDALONE_FRAME
  if (HSWrite (hPort->HSControlBlock, lpBuf, NumBWrite, /* lastFrame */ TRUE, &ProcessWrBuffer)
#else
  // lpBuf[1] & 0x10 is the poll/final bit which indicates if there are more packets
  // to send in this batch.
	if (HSWrite ((HSControl *)hPort->HSControlBlock, lpBuf, NumBWrite, lpBuf[1] & 0x10 /* P/F */, &ProcessWrBuffer)
#endif
					== STS_OK)
	{
    *lpNumBWritten = NumBWrite;
	}
  else
    *lpNumBWritten = 0;

  return (TRUE);
}
