
//***************************************************************************
//
// File:        hsdriver.c
//
// Description: This file contains the code for the National Semiconductor 87xxx 
// series IR device.
//
// Author:      Anne Kelley
//
// Copyright 1996, 1997 Alpha Omega Computer Systems, Inc.
// All rights reserved worldwide
//
//***************************************************************************

#define WANTVXDWRAPS

//#define DONGLE_MANIPULATION	1	/* Non-zero if dongles are set up */

#include <basedef.h>
#include <vmm.h>
#include <vcomm.h>
#include <debug.h>
#include "stdarg.h"
#include "port.h"
#include "common.h"
#include "dma.h"
#include "buffman.h"
#include "hsdriver.h"
#include "irq.h"
#include <vxdwraps.h> 
#include <vtd.h>                                          
#include <vpicd.h>
#ifdef WIN31COMPAT
#include "intern31.h"
#else
#include "internal.h"
#endif

#pragma VxD_LOCKED_CODE_SEG
#pragma VxD_LOCKED_DATA_SEG

#ifndef WIN31COMPAT
  #ifdef  DEBUG
    void    DebugPrintf(char *Format, ...);
    #define VPRINTF(args) DebugPrintf args
  #else   // Release
    #define VPRINTF(args)
  #endif //release
#else //WIN31COMPAT
  #ifdef DEBUG
    void DbgPrint(char *afmt, ...);
    #define DebugPrint(arg) DbgPrint arg
    #define VPRINTF(arg) DbgPrint arg
  #else //release
    #define DebugPrint(arg)
    #define VPRINTF(arg)
  #endif //release
#endif //WIN31COMPAT

#define	DL_READ			0x0001
#define	DL_WRITE		0x0002
#define	DL_INFO			0x0004
#define	DL_NOTES		0x0008
#define	DL_TIMING		0x0010
#define	DL_VERBOSE	0x0020
#define DL_ERROR    0x0040

//#define	DEBUGLEVEL DL_TIMING

#ifndef DEBUGLEVEL
#define	DEBUGLEVEL	(DL_VERBOSE|DL_INFO|DL_NOTES)
#endif

HSControl  HSControlData;      // Interrupt needs access to this
                                      // structure. 

#define LocalPort HSControlData.comPort
#define SB(b)		(b)
#define SV(p,v)	(v)

U16  Iirtest = 0;
U16  Iirtest1 = 0;
U16  Iirtest2 = 0;
U16  Iirtest3 = 0;

U8  TxDMAChannel[] =
{
  DCSR_TX_0,
  DCSR_TX_1,
  0x00,
  DCSR_TX_3
};

U8  RxDMAChannel[] =
{
  DCSR_RX_0,
  DCSR_RX_1,
  0x00,
  DCSR_RX_3
};

int irqArray [16] = 
{
  0,
  0,
  0,
  ICR_IRQ_3, 
  ICR_IRQ_4, 
  ICR_IRQ_5, 
  ICR_IRQ_6, 
  ICR_IRQ_7, 
  0,
  ICR_IRQ_9, 
  ICR_IRQ_10, 
  ICR_IRQ_11, 
  ICR_IRQ_12, 
  0, 
  ICR_IRQ_14, 
  ICR_IRQ_15
};

U32   testFlag = 0;
U32   threshold;
char *bufferPtr;
U32   bufLength;

U32   bufferAddress;
U16   bufferCount;
U16   txThresh;

#ifdef DEBUG
#define int3 _asm int 3;
#else
#define int3
#endif

#define	BankSelect(bank)				\
	(outportb(LocalPort + CCR, (bank)<<6))

#define PortWrite(reg, v)  \
  outportb(LocalPort + (U32)(reg), (U8)(v))

#define PortRead(reg) \
    inportb(LocalPort + (reg))

#define SetBits(reg, v)   PortWrite((reg), (U8)(PortRead(reg) | (v)))

#define ClearBits(reg, v) PortWrite((reg), (U8)(PortRead(reg) & ~(v)))

int HSBaud(HSControl *dev, U32 baud);
void HSRestartRead(HSControl *dev);
void HSSetupDma(HSControl *dev, char *buf, int length, int iomode);
void HSSetupPIO (HSControl *dev, char *buf, int length, int iomode, int threshold);
void HSSetTxLength(int length);
void HSSetRxLength(int length);
U32 HSWriteBuffer(HSControl *dev);
U32 ReadRxLength(void);


/******************************************************************************************
/*  parseDevType
/*
/*  Parses the DevType string and returns the data that has been parsed.  Cannot pass in 
/*  the IR data structure because this is called by IR_INIT which doesn't have that 
/*  information yet.  Even though the data that is returned is stored in the 
/*  HSControlData structure, the individual data needs to be passed in because IR_Init
/*  doesn't have access to the HSControlData structure.
/*
/*  Input:    char *    The device type string
/*  Outputs:  U32 *     Dongle type: 0 = HP, Sharp; non-zero = Temic, default = 0
/*            U8 *      Transfer mode: 0 = PIO mode; non-zero = DMA mode, returned as 8 for
/*                      the FCR data.
/*                      default = DMA mode
/*            U32 *     size of the FIFO, default = 64.
/*            U32 *     Receive threshold size, default = 32
/*            U32 *     Transmit threshold size, default = 48
/*            U8 *      low power mode enable: 0 = disable, 1 = enable, default will be 1
/*            U8 *      sleep mode enable: 0 = disable, 1 = enable, default is 0
/*            U8 *      FIFO control register data
/*            U8 *      Preamble length register data
/******************************************************************************************/

void parseDevType (char * devInfo, U32 *temicDongle, U8 *dmaMode, U32 *FIFOsize, 
                   U32 *RxThreshold, U32 *TxThreshold, U8 *lowPowerEnable, 
                   U8 *sleepModeEnable, U8 *FCRdata, U8 *PLRdata)
{
  char          devtype[80];
  U32           len;
  char          *p;

  len = xstrlen(devInfo);
  if (len > sizeof(devtype) - 1)
    len = sizeof(devtype) - 1;

  memCopy(devtype, devInfo, len);

  // devtype: device type/dongle type/pio or dma mode/FIFO size/Rx threshold/Tx threshold/
  //        low power mode enable/sleep mode enable
  p = xstrchr(devtype, '/');
  if (p != NULL)
    *p++ = '\0';

	// Parse devInfo and extract the dongle type

  *temicDongle = xstrtol(p, &p, 16);    // dongle type
  DebugPrint (("parseDevType: temicDongle %d\n", *temicDongle));

  // get transfer mode  (default = dma)
  if (*p == '/')
 	{
		++p;
	  *dmaMode = (U8)xstrtol(p, &p, 10);   // xfer mode (DMA or PIO)
    DebugPrint (("parseDevType: dmaMode %d\n", *temicDongle));
    if (*dmaMode != 0)
      *dmaMode = 8;
    else
      *dmaMode = 0;
  }
  else
    *dmaMode = 8;                   // default xfer mode is DMA

  // get FIFO size  (default = 64)
  if (*p == '/')
 	{
    ++p;
    *FIFOsize = xstrtol(p, &p, 10);
    DebugPrint (("parseDevType: FIFOsize %d\n", *FIFOsize));
  }
  else
    *FIFOsize = 64;
  

  // get Rx threshold (default = 32)
  if (*p == '/')
 	{
    ++p;
    *RxThreshold = xstrtol(p, &p, 10);
    DebugPrint (("parseDevType: RxThreshold %d\n", *RxThreshold));
  }
  else
    *RxThreshold = 32;

  // get Tx threshold (default = 48)
  if (*p == '/')
 	{
    ++p;
    *TxThreshold = xstrtol(p, &p, 10);
    DebugPrint (("parseDevType: TxThreshold %d\n", *TxThreshold));
  }
	else
    *TxThreshold = 48;
  
  // get low power mode enable (default for now = 0)
  if (*p == '/')
  {
    ++p;
    *lowPowerEnable = (U8)xstrtol (p, &p, 10);
    VPRINTF (("lowpower %x\n", *lowPowerEnable));
    DebugPrint (("parseDevType: lowpower %d\n", *lowPowerEnable));
    if (*lowPowerEnable != 0)
    {
      *lowPowerEnable = 0x10;
    }
  }
  else
  {
    *lowPowerEnable = 0;
  }

  // get sleep mode enable (default for now = 0)
  if (*p == '/')
  {
    ++p;
    *sleepModeEnable = (U8)xstrtol (p, NULL, 10);
    DebugPrint (("parseDevType: sleepModeEnable = %d\n", *sleepModeEnable));
    if (*sleepModeEnable != 0)
    {
      *sleepModeEnable = 0x8;
    }
  }
  else
  {
    *sleepModeEnable = 0;
  }
   
  // set up data variables for writing to the FCR and PLR registers.  Set values of
  // thresholds to valid number if the value in the registry is not valid.
  *FCRdata = FCR_FIFOEN | FCR_RXFR | FCR_TXFR | *dmaMode;
  switch (*FIFOsize)
  {
  case 16:
    *FIFOsize = 16;
    *FCRdata |= FCR_FS_16;
    switch (*RxThreshold)
    {
    case 1:
      *FCRdata |= FCR_RX16_1;
      break;
    case 4:
      *FCRdata |= FCR_RX16_4;
      break;
    case 8:
      *FCRdata |= FCR_RX16_8;
      break;
    case 14:
    default:
      *FCRdata |= FCR_RX16_14;
      *RxThreshold = 14;
      break;
    }
    switch (*TxThreshold)
    {
    case 2:
      if (!(*dmaMode))
      {
        *PLRdata = 0x30;
      }
      else
      {
        *PLRdata = 0x10;
        *TxThreshold = 12;
      }
      break;
    case 8:
      *PLRdata = 0x20;
      break;
    case 12:
    default:
      *PLRdata = 0x10;
      *TxThreshold = 12;
      break;
    }
    break;
  case 64:
  default:
    *FCRdata |= FCR_FS_64;
    switch (*RxThreshold)
    {
    case 1:
      *FCRdata |= FCR_RX64_1;
      break;
    case 16:
      *FCRdata |= FCR_RX64_16;
      break;
    case 56:
      *FCRdata |= FCR_RX64_56;
      break;
    case 32:
    default:
      *FCRdata |= FCR_RX64_32;
      *RxThreshold = 32;
      break;
    }
    switch (*TxThreshold)
    {
    case 8:
      if (!(*dmaMode))
      {
        *PLRdata = 0x30;
      }
      else
      {
        *PLRdata = 0x10;
        *TxThreshold = 48;
      }
      break;
    case 32:
      *PLRdata = 0x20;
      break;
    case 48:
    default:
      *PLRdata = 0x10;
      *TxThreshold = 48;
      break;
    }
    break;
  }
}  //parseDevType


  

/*
 *	TISetup
 *
 *	Sets up the TIR2000 chip for serial communications.
 *
 *	Entry:	device control structure
 *
 *	Returns TRUE if success
 */
int TISetup (HSControl *dev)
{

  static int	lastMode = -1;
  U32         flags;
  
  if (lastMode == dev->speedMode)
  {
    return (0);		/* Don't do a thing if not different modes */
  }

  lastMode = dev->speedMode;
  dev->sir576en = 0;    // 0 => not running at 576kbps

  EnterCritical (flags);

  if (dev->dmaMode)
    DisableDMA (dev->dmaChannel);

  /* Set up high speed communications */
  
  if (dev->speedMode > SIR_115200)      // MIR or FIR mode
  {
    PortWrite (IER, 0);     // Disable all interrupts
    PortWrite (MCR, 8);
    VPRINTF (("TISetup: dev->FCRdata: %x\n", dev->FCRdata));
    VPRINTF (("TISetup: dev->PLRdata: %x\n", dev->PLRdata));

    // enable appropriate interrupts
    if (!dev->dmaMode)
      PortWrite (IER, IER_RXIEN | IER_MSIEN);   // PIO mode read interrupts
    else
      PortWrite (IER, IER_TOIEN | IER_SFIEN | IER_DMAIEN | IER_TXURIEN);  // DMA mode read interrupts
    PortWrite (MCR, 0xb);           // Enable interrupts to interrupt

    if ((dev->speedMode == MIR_1152000) || (dev->speedMode == SIR_576000))
    {
      // set up for MIR mode
      PortWrite (MDR, MDR_MD_MIR);
      
      // set number of start flags to 2
      BankSelect (0); 
      dev->PLRdata |= 0x2;
//      dev->PLRdata |= 0x6;      // #### debugging purposes for physical error
      PortWrite (PLR, dev->PLRdata);
      VPRINTF (("TISetup: dev->PLRdata: %x\n", dev->PLRdata));

      if (dev->speedMode == SIR_576000)
      { 
        // set up for 576kbps
        BankSelect (0);
        dev->sir576en = ACREG_580K;
        PortWrite (ACREG, ACREG_580K | ACREG_RXEN);
        BankSelect (2);
        PortWrite (PRESC, 3);
      }
    }
    else
    if (dev->speedMode == FIR_4000000)
    {
      // set up for 4mbps
      BankSelect (2);
      PortWrite (PRESC, 6);  // set the prescaler register for 8 MHz internal clock
      
      PortWrite (MDR, MDR_MD_FIR | dev->sleepModeEnable | dev->lowPowerEnable);    // FIR mode, Frame end mode

      BankSelect (0);
      dev->PLRdata = dev->PLRdata | 0xC0;
      PortWrite (PLR, dev->PLRdata);
    }

    // If the dongle type is temic, set up the TIR2000 for a temic dongle.
    if (dev->temicDongle)
    {
      BankSelect (3);
      PortWrite (GPIODIR, 0);
      PortWrite (GPIODAT, 1);
      PortWrite (IRCFG, IRCFG_TEMIC);
      PortWrite (GPIODAT, 0);
      PortWrite (IRCFG, 0);
    }

  }
  else
  {
    // set up for SIR mode

    PortWrite (IER, 0x00);
    BankSelect (2);
    PortWrite (PRESC, 26);
    PortWrite (MDR, MDR_MD_SIR | dev->sleepModeEnable | dev->lowPowerEnable);
    PortWrite(LCR, LCR_WLS_8 | LCR_STB_1);
    PortWrite (IRCFG, 0);
    PortWrite (IER, IER_RXIEN | IER_LSIEN/*IER_MSIEN*/);
    PortWrite (MCR, 0xb);
    BankSelect (0);
    PortWrite (PLR, dev->PLRdata);
    PortWrite (ACREG, ACREG_RXEN | ACREG_TXEN);

    // If the dongle type being used is a temic, set up the TIR2000 accordingly.
    if (dev->temicDongle)
    {
      BankSelect (3);
      PortWrite (GPIODIR, 0);
      PortWrite (GPIODAT, 1);
      PortWrite (GPIODAT, 0);
    }
    VPRINTF (("TISetup: mode = %x, ACREG = %x\n", PortRead (MDR), PortRead (ACREG)));
  }

  PortWrite(LCR, LCR_WLS_8 | LCR_STB_1);

  /* Flush the input registers */
  (void) PortRead(IIR);
  (void) PortRead(IIR);


  ExitCritical (flags);
  return (1);
}



//*********************************************************
//*	GetElapsedTime
//*	Obtains the time elapsed since the last time the timer was
//*	checked.  This routine is only for debugging purposes.
//*
//*	Inputs:	 None
//*	Returns: The elapsed time
//**********************************************************

unsigned long GetElapsedTime(void)
{
  static unsigned int lastTime = 0;
  unsigned int        elapsed = GetTimerValue() - lastTime;

  lastTime += elapsed;

  return (elapsed);
}

//**********************************************************
//* ioDelay
//* makes a small delay for i/o accesses
//**********************************************************

int ioDelay(void) 
{ 
  return 0; 
}


//***********************************************************
//* HSOpen
//* Initializes the HS driver port.
//* 
//* Inputs:
//*   Com port for the IR driver
//*   Interrupt request number for the IR driver.
//*   DMA channel # 1
//*   DMA channel # 2
//*   device information (contains the device type and the control port)
//*   Communication port information structure.
//* Outputs:
//*   IR specific data.
//**********************************************************

HSControl *HSOpen (int comPort, int irq, int dma1, int dma2,
                   char *devInfo, PPortInformation hPort)
{
  U32           version;
  U32           numbRxPages, numbPoolPages;
  HSControl *   dev = &HSControlData;     // For now, there is only one
  PVOID         CopyPageBuf;
  U32           PageAllocateFlags;

  VPRINTF (("HSOpen\n"));

  parseDevType (devInfo, &dev->temicDongle, &dev->dmaMode, &dev->FIFOsize, &dev->RxThreshold,
                &dev->TxThreshold, &dev->lowPowerEnable, &dev->sleepModeEnable, &dev->FCRdata,
                &dev->PLRdata);

  threshold = dev->RxThreshold;   // set here for the PIO interrupt service routine to save time
  hPort->FCRdata = dev->FCRdata;  // For some reason Microsoft feels like resetting the FCR so
                                  // we need to store the data to write to the FCR when MS does
                                  // reset it.
  
  dev->dmaChannel = dma1;

  dev->irq = irq;
  dev->comPort = comPort;
  dev->speedMode = BAD_MODE;

  dev->portInfo = hPort;
  dev->readCallBack = 0;
  dev->writeCallBack = 0;
  dev->characterCount = 0;

//#if DEBUGLEVEL & DL_INFO
  VPRINTF (("HSOPEN: IRQ: %x, com: %x, dma: %x, dongle type: %d, xfer mode: %d\n",
            dev->irq, dev->comPort, dev->dmaChannel, dev->temicDongle, dev->dmaMode));
  VPRINTF (("HSOPEN:  FIFO size: %d, RxThreshold: %d, TxThreshold: %d\n",
            dev->FIFOsize, dev->RxThreshold, dev->TxThreshold));
  VPRINTF (("HSOPEN:  low power mode enable: %x, sleep mode enable: %x\n", 
            dev->lowPowerEnable, dev->sleepModeEnable));
//#endif

  // set up the interrupt configuration register: totem pole mode, IRQ active high, 
  // status FIFO threshold = 4.
  BankSelect (2);
  irq = irqArray [dev->irq];
  PortWrite (ICR, irq | ICR_TOTEM_POLE | ICR_ALS_HIGH | ICR_SFT_7);

  HSBaud (dev, 9600l);    // 9600 baud

  if ((version = vdmadGetVersion ()) < 0x030A)
  {
    return (dev);
  }

//#if DEBUGLEVEL & DL_INFO
  VPRINTF (("HSOPEN: dma channel = %x\n", dev->dmaChannel));
//#endif

#ifndef WIN31COMPAT
  // DMA virtualized only for windows 95 and when the transfer mode is DMA.
  if (dev->dmaMode)
    dev->vdmadRxHandle = vdmadVirtualizeChannel (dev->dmaChannel);

  if ((dev->vdmadRxHandle == 0xFFFFFFFF) && (dev->dmaMode))
  {
    VPRINTF (("HSOpen:  Invalid DMA handle, exiting high speed initialization\n"));
    return (0);       /* DMA handle invalid - return 0 to force slow speed communications */
  }
#endif

//#if DEBUGLEVEL & DL_INFO
    VPRINTF (("HSOPEN: dma handle: %x\n", dev->vdmadRxHandle));
//#endif


  // Allocate memory for DMA transfers.

  numbRxPages = (((NumRxBlocks + 1) * BufSize) + PageSize - 1) / PageSize;
  numbPoolPages = ((NumPool * BufSize) + PageSize - 1) / PageSize;
  PageAllocateFlags = PAGECONTIG | PAGELOCKED | PAGEUSEALIGN | 
						PAGEFIXED | PAGEZEROINIT;
#ifdef WIN31COMPAT
  // _Page_Allocate is different for Windows for Workgroups and Windows 95.
  _Page_Allocate (numbRxPages, &CopyPageBuf, &dev->inputBufHandle, &dev->inputBuffer);
  DebugPrint (("hsopen: input - handle = %lx, buffer = %lx\n", dev->inputBufHandle, dev->inputBuffer));
  dev->currentBufferPtr = dev->inputBuffer;
  _Page_Allocate (numbPoolPages, &CopyPageBuf, &dev->bufPoolHandle, &dev->bufPool);
#else //win 95
  dev->inputBuffer = _PageAllocate (numbRxPages, 
                                    PG_SYS, 
                                    0, 
                                    0x0f, 
                                    0, 
                                    0x0fff,
                                    &CopyPageBuf, 
                                    PageAllocateFlags);
  dev->currentBufferPtr = dev->inputBuffer;

  dev->bufPool = _PageAllocate (numbPoolPages, 
                                PG_SYS, 
                                0, 
                                0x0f, 
                                0, 
                                0x0fff,
                                &CopyPageBuf, 
                                PageAllocateFlags);
#endif

//#if DEBUGLEVEL & DL_INFO
  VPRINTF (("HSOPEN: input: %lx, pool: %lx\n", 
            dev->currentBufferPtr, dev->bufPool));
  VPRINTF (("HSOPEN:  buffer pointer: %p\n", dev->currentBufferPtr));
  DebugPrint (("HSOPEN:  buffer pointer: %p\n", dev->currentBufferPtr));
  VPRINTF (("HSOPEN:  PLR: %x\n", dev->PLRdata));
  DebugPrint (("HSOPEN:  PLR: %x\n", dev->PLRdata));
//#endif

  InitBuf (&dev->freeQueue, dev->buffers, dev->bufPool, 
           NumPool, (void *)dev);

  return (dev);
} // HSOpen



//***********************************************************
//* HSClose
//* Shuts down the high speed port.
//*
//* Inputs:
//*   IR specific data
//* Outputs:
//*   Status of the operation
//***********************************************************
hsResults HSClose (HSControl *dev)
{
  U32   freeFlag;
  U32   operationFlag;
  U32   flags;
#ifndef WIN31COMPAT
  BOOL  flag;
#endif

  EnterCritical (flags);

  if (dev->dmaMode)
    DisableDMA (dev->dmaChannel);	  /* Disable any DMA in process */
  HSBaud(dev, 9600L);               // set speed back to 9600 baud

  ExitCritical (flags);

  operationFlag = 0;

#if DEBUGLEVEL & DL_INFO
  DebugPrint (("hsclose: handle = %x, bufPool = %x, input = %x\n", 
          dev->vdmadRxHandle, dev->bufPool, dev->inputBuffer));
#endif

#ifndef WIN31COMPAT
  // unvirtualize the DMA channel in Win95 - this wasn't virtualized in WFW.
  if ((dev->vdmadRxHandle != -1) && (dev->dmaMode))
    flag = UnVirtualizeDMAChannel (dev->vdmadRxHandle);
#endif

  // Deallocate memory used for our buffers.
  freeFlag = _PageFree (dev->bufPool, operationFlag);
  freeFlag = _PageFree (dev->inputBuffer, operationFlag);

  return STS_OK;
} //HSClose

//***********************************************************
//* HSWrite
//* Requests a write of the user supplied buffer.
//*
//* Inputs:
//*   IR specific data
//*   Data buffer
//*   Data buffer length
//*   last frame?  (0x10 if poll/final bit is set, 0 if not)
//*   Callback function
//* Outputs:
//*   Status
//***********************************************************
hsResults HSWrite (HSControl *dev, U8 *buffer, U32 length,
                   BOOL lastFrame, writeCallBack callBack)
{
  hsResults   rc = STS_OK;
  U8          mdr;

  if ((dev != 0) && (buffer != 0) && (length != 0))
  {
    Buffer *buf;

    // valid data, so set up for transmit.

    buf = RemoveBuf (&dev->freeQueue);
    dev->writeCallBack = callBack;

    if (buf != 0)
    {
      // There was an available buffer in the buffer free queue so can transmit
      memCopy(buf->buf, buffer, length);      // buf will contain the data in the DMA memory 
                                              // space to transmit
      buf->numbBytes = length;
      buf->lastFrame = lastFrame;
      AppendBuf (&dev->wrQueue, buf);
      
      // if in idle mode (read mode) and we have copied the last frame to transmit into memory
      // space, start writing data.
      if ((dev->irState == IrStateRead) && buf->lastFrame)
      {
        BankSelect (0);
        PortWrite (ACREG, ACREG_TXEN | dev->sir576en);  // disable receiver
        if ((dev->speedMode == MIR_1152000) || (dev->speedMode == SIR_576000))
        {
          // MIR and 576kb requires the sleep mode and low power mode to be disabled during 
          // receive.  Reenable sleep mode and low power mode as applicable
          mdr = PortRead (MDR);
          PortWrite (MDR, mdr | (dev->sleepModeEnable | dev->lowPowerEnable));
        }
        // Driver is idle: start the write process
        HSWriteBuffer(dev);
      }
    }
    else
    { // no available buffers in the buffer free queue.
      rc = STS_FAIL;
    }
  }
  else
  { // no data to transmit
    rc = STS_FAIL;
  }

  return rc;
} // HSWrite

//***********************************************************
//* HSRead
//* Requests a read into the user-supplied buffer.
//*
//* Inputs:
//*   IR specific data
//*   Callback function
//* Outputs:
//*   Status
//***********************************************************
hsResults HSRead (HSControl *dev, readCallBack callBack)
{
  VPRINTF (("HSRead: mode = %x\n", PortRead (MDR)));
  DebugPrint (("HSRead: mode = %x\n", PortRead (MDR)));
  if (dev != 0)
    dev->readCallBack = callBack;

  return (dev ? STS_OK : STS_FAIL);
} //HSRead


//***********************************************************
//* HSSetControl
//* Issues a specific control function.
//*
//* Inputs:
//*   IR specific data
//*   Control function
//* Outputs:
//*   Status
//***********************************************************
hsResults HSSetControl (HSControl *dev, CONTROL_SEL cntlSelector, ...)
{
  U32 baud;
  va_list marker;
  U32 *   TxTrigger;
  int     irq;


  switch (cntlSelector)
  {
  case CTL_HSBaud:    // set baud rate
    va_start (marker, cntlSelector);
    baud = va_arg (marker, long);
    HSBaud (dev, baud);
    va_end (marker);
    VPRINTF (("HSSetControl: mode = %x\n", PortRead (MDR)));
    DebugPrint (("HSSetControl: mode = %x\n", PortRead (MDR)));
    break;

  case CTL_resetRegisters:    
    // re-setup registers.  This is necessary because for some reason Windows clears some of 
    // the registers that were previously set up during the initialization phase.

    // set up the mode definition register.
    if ((dev->speedMode == MIR_1152000) || (dev->speedMode == SIR_576000))
    {
      // low power mode and sleep mode disabled during read for MIR
      PortWrite (MDR, MDR_MD_MIR);
    }
    else if (dev->speedMode == FIR_4000000)
    {
      PortWrite (MDR, MDR_MD_FIR | dev->sleepModeEnable | dev->lowPowerEnable);
    }
    else
    {
      PortWrite (MDR, MDR_MD_SIR | dev->sleepModeEnable | dev->lowPowerEnable);
    }

    // set up FIFO control register
    PortWrite (LCR, 0x80);
    PortWrite (FCR, dev->FCRdata);
    PortWrite (LCR, 0);

    // set up the interrupt configuration register
    BankSelect (2);
    irq = irqArray [dev->irq];
    PortWrite (ICR, irq | ICR_TOTEM_POLE | ICR_ALS_HIGH | ICR_SFT_4);

    va_start (marker, cntlSelector);
    TxTrigger = (U32 *)va_arg (marker, long);
    va_end (marker);
    *TxTrigger = (U32) dev->TxThreshold;
    VPRINTF (("HSSetControl:  TxFifoTrigger: %x\n", *TxTrigger));
    VPRINTF (("HSSetControl:  TxFifoTrigger: %x\n", TxTrigger));
    VPRINTF (("HSSetControl:  TxThreshold: %x\n", dev->TxThreshold));
    VPRINTF (("HSSetControl:  FCR = %x\n", dev->FCRdata));
    VPRINTF (("HSSetControl:  MDR = %x\n", PortRead (MDR)));
    DebugPrint (("HSSetControl:  MDR = %x\n", PortRead (MDR)));
    break;
  }

  return STS_OK;
} // HSSetControl

//***********************************************************
//* HSGetStatus
//* Returns a packet driver status
//*
//* Inputs:
//*   IR specific data
//*   Status function
//*   status data
//* Outputs:
//*   status
//***********************************************************
hsResults HSGetStatus (HSControl *dev, STATUS_SEL statSelector, 
                       void *status)
{
  switch (statSelector)
  {
  case STS_HsGetSpeedMode:    // TRUE = currently in MIR/FIR mode
    if (dev->speedMode > SIR_115200)
      *((BOOL *)status) = TRUE;
    else
      *((BOOL *)status) = FALSE;
    break;

  case STS_HsGetTimerVal:     // read the elapsed time since previous timer read
    *((U32 *)status) = GetElapsedTime();
    break;
#ifdef WIN31COMPAT
  case STS_HsGetHandle:       // TRUE = DMA buffer handles valid.
    if (dev->inputBufHandle && dev->bufPoolHandle)
      *((BOOL *)status) = TRUE;
    else
      *((BOOL *)status) = FALSE;
    
    break;
#endif
  }

  return STS_OK;
} // HSGetStatus


  
/*
 *	HSBaud
 *
 *	Set the baud rate on the TIR2000 device.
 *
 *	Entry:
 *    dev     ir device control structure.
 *    baud		Baud rate (e.g. 1152000) for connection
 *
 *	Returns TRUE if success
 */

int HSBaud(HSControl *dev, U32 baud)
{
  U32 flags;

  switch (baud)
  {
  case 4000000L:  dev->speedMode = FIR_4000000; break;

  case 1152000L:  dev->speedMode = MIR_1152000; break;
  case 576000L:	  dev->speedMode = SIR_576000;  break;
  case 115200L:   dev->speedMode = SIR_115200;  break;
  case 57600L:    dev->speedMode = SIR_57600;   break;
  case 38400L:    dev->speedMode = SIR_38400;   break;
  case 19200L:    dev->speedMode = SIR_19200;   break;
  case 9600L:     dev->speedMode = SIR_9600;    break;
  case 2400L:     dev->speedMode = SIR_2400;    break;

  default:        dev->speedMode = BAD_MODE;    break;
  }

  TISetup (dev);

  BankSelect (0);
  VPRINTF (("HSBaud:  ACREG = %x, dev->sir576en = %x\n", PortRead (ACREG), dev->sir576en));
  DebugPrint (("HSBaud:  ACREG = %x, dev->sir576en = %x\n", PortRead (ACREG), dev->sir576en));

  EnterCritical (flags);

  if (dev->speedMode > SIR_115200)
    HSRestartRead(dev);     // put in high speed read mode.

  VPRINTF (("HSBaud: mode = %x\n", PortRead (MDR)));
  DebugPrint (("HSBaud: mode = %x\n", PortRead (MDR)));

  ExitCritical(flags);

  return (dev->speedMode != BAD_MODE);
}


/*
 *	HSRestartRead
 *
 *	Restart the read into the currently established user buffer.
 *
 *	Entry:	*dev      The pointer to the HSControlData structure.
 *
 *	Returns nothing
 */
void
HSRestartRead(HSControl *dev)
{

  U8    mdr;

  dev->irState = IrStateRead;

  // Set up input buffer to put received data.
  dev->currentBufferPtr = dev->inputBuffer;
  dev->currentBufferLen = 0;
  bufferPtr = dev->currentBufferPtr;
//  DebugPrint (("input buffer = %x, buffer pointer = %x, dev->currentBufferPtr = %x\n", 
//                dev->inputBuffer, bufferPtr, dev->currentBufferPtr));
  bufLength = 0;
  
  // Enable receiver, disable transmitter
  BankSelect (0);
  PortWrite (ACREG, ACREG_RXEN | dev->sir576en);

  if ((dev->speedMode == MIR_1152000) || (dev->speedMode == SIR_576000))
  {
    // disable sleep and low power modes for MIR read.
    mdr = PortRead (MDR);
    PortWrite (MDR, mdr & ~(dev->sleepModeEnable | dev->lowPowerEnable));
  }

  if (!dev->dmaMode)     // programmed I/O mode
    HSSetupPIO (dev, dev->currentBufferPtr, MAXDEVICEBUFLEN, IrStateRead, dev->RxThreshold);
  else                  // DMA mode
    HSSetupDma (dev, dev->currentBufferPtr, MAXDEVICEBUFLEN, IrStateRead);
}

/*
 *	HSSetupDma
 *
 *	Sets up a dma transfer
 *
 *	Entry:	
 *    dev       IR data structure
 *    buf		    User buffer for the transfer
 *		length		Length of user buffer
 *		iomode		Mode of the I/O:
 *				IrStateRead	- read state.
 *				IrStateWrite - write state.
 *
 *	Returns nothing.
 */
void HSSetupDma(HSControl *dev, char *buf, int length, int iomode)
{
  U32 flags;

  BankSelect (2);
  EnterCritical (flags);

  // Reset the status FIFO
  PortWrite (DCSR, DCSR_SFRESET);

  if (iomode == IrStateRead)
  {
    //reset FIFOs
    PortWrite(FCR, dev->FCRdata);

    (void) PortRead(RxBufferReg);
    (void) PortRead(RxBufferReg);
  }
  else // iomode = IrStateWrite
  {
    // reset FIFOs, clear any extra transmitter underruns.
    PortWrite (FCR, dev->FCRdata);
    BankSelect (0);
    PortRead (RESUME);
  }

  /* Set up for the DMA transfer */
  InitDMAXfer(dev->dmaChannel, 
        (iomode == IrStateWrite) ? DEMAND_READ : DEMAND_WRITE,
        buf, length - 1);

  /* Load device frame length registers */
  if (iomode == IrStateWrite)
    HSSetTxLength(length);
  else
    HSSetRxLength (MaxPacketLength + 2);

  StartDMAXfer(dev->dmaChannel);

  // enable DMA channel for read or write
  BankSelect(2);
  if (iomode == IrStateWrite)
    PortWrite (DCSR, TxDMAChannel[dev->dmaChannel] | DCSR_DMAEN);
  else
    PortWrite (DCSR, RxDMAChannel[dev->dmaChannel] | DCSR_DMAEN);

  ExitCritical(flags);
}



/*
 *	HSSetupPIO
 *
 *	Sets up a programmed I/O mode transfer
 *
 *	Entry:	
 *    dev       IR data structure
 *    buf		    User buffer for the transfer
 *		length		Length of user buffer
 *		iomode		Mode of the I/O:
 *				IrStateRead	- read state.
 *				IrStateWrite - write state.
 *    threshold The FIFO threshold level.
 *
 *	Returns nothing.
 */

void HSSetupPIO (HSControl *dev, char *buf, int length, int iomode, int threshold)
{
  U16   transferLength;

  if (iomode == IrStateRead)
  {
    PortWrite (IER, 0);

    // reset FIFO, clear errors
    PortWrite (LCR, 0x80);
    PortWrite (FCR, dev->FCRdata);
    PortWrite (LCR, 0);
    BankSelect (0);
    PortRead (RESUME);

    // Set maximum packet size for error checking.
    HSSetRxLength (MaxPacketLength);

    // Enable receive threshold, receive overrun, and end of packet interrupts
    PortWrite (IER, IER_RXIEN /*| IER_RXEIEN */| IER_RXORIEN | IER_EOPIEN);
  }

  
  else // iomode = IrStateWrite
  {
    PortWrite (IER, 0);

    // Set transferLength to the minimum of 64 or the packet size for the first set of
    // data to be transmitted.
    transferLength = (U16)(((U32)length < dev->FIFOsize) ? length : dev->FIFOsize);
    HSSetTxLength(length);

    // Transmit transferLength bytes of the packet to send.
    _asm
    {
      push  esi
      mov   esi, buf
      mov   dx, word ptr [LocalPort]
      xor   ecx, ecx
      mov   cx, word ptr [transferLength]
      rep   outsb
      mov   buf, esi
      pop   esi
    }

    // adjust the buffer to point to the next set of data in the packet to send.
    dev->currentBufferLen = length - transferLength;
    dev->currentBufferPtr = buf;

    // bufferAddress is the location in the packet to send the next set of data,
    // bufferCount is the number of bytes that has already been transmitted.
    // These variables are set here so that there is less work to be done in the
    // interrupt service routine to hopefully reduce the number of transmitter 
    // underruns
    bufferAddress = dev->currentBufferPtr;
    bufferCount = dev->currentBufferLen;
    txThresh = (U16)dev->TxThreshold;

    // enable transmit threshold and transmit underrun interrupts
    PortWrite (IER, IER_TXIEN | IER_TXURIEN);
  }
}


/*
 *	HSSetTxLength
 *
 *	Set the registers defining the transmit packet length
 *
 *	Entry:	length		Length of packet
 *
 *	Returns nothing.
 */
void
HSSetTxLength(int length)
{
  BankSelect (0);
  PortWrite (TXFLH, length / 256);
  PortWrite (TXFLL, length & 0xFF);
}

/*
 *	HSSetRxLength
 *
 *	Set the registers defining the receive packet length.
 *
 *	Entry:	length
 *
 *	Returns nothing
 */

void HSSetRxLength(int length)
{
  if (length > MAXDEVICEBUFLEN)
    length = MAXDEVICEBUFLEN;

  BankSelect (0);
  PortWrite (RXFLL, length & 0xFF);
  PortWrite (RXFLH, length / 256);
}

/*
 *	HSWriteBuffer
 *
 *	Transmit a user specified buffer.
 *
 *	Entry:	None
 *
 *	Returns TRUE if success.
 */

U32 HSWriteBuffer(HSControl *dev)
{
  U32           done = 0;
  U32           flags;
  Buffer        *buf;

  // get next buffer in the write queue to send out.
  buf = dev->wrQueue.head;
  if (buf != 0)
  {
    // There is data to transmit
    EnterCritical (flags);

    if (!dev->dmaMode)
      HSSetupPIO (dev, buf->buf, buf->numbBytes, IrStateWrite, dev->TxThreshold);
    else
      HSSetupDma(dev, buf->buf, buf->numbBytes, IrStateWrite);

    dev->irState = IrStateWrite;
    ExitCritical (flags);

    done = 1;
  }
  return (done); /* done = 0 => no data in the transmit queue to send. */
}


/*
 *	ReadRxLength
 *
 *	Read the received packet length from the status FIFO.  This is only called in DMA mode
 *
 *	Entry:	none
 *
 *	Returns the last read packet length.
 */

U32 ReadRxLength(void)
{
  U32 length;

  length = PortRead(SFREGL);

  length += PortRead(SFREGH) * 256;

  return (length);
}




static int recurseFlag = 0;   // indicator for interrupt service routine getting reentered.

/*
 *	TIIntProcDMA
 *
 *	Interrupt service routine for processing TIR2000 PIO mode interrupts
 *
 *	Entry:  IR device control structure
 *
 *	Returns nothing.
 */

void TIIntProcDMA (HSControl *dev)
{
  static int  i;
  static int  fst;
  static int  iir;
  static int  ier;
  static int  lsr;
  static int  mcr;
  static int  oldBsr;
  static int  maxrlen;
  static int  userlen;
  
  static struct statusfifo 
  { 
    int fst; 
    int len; 
  } fifo[10];
  
  static int  status = 0;
  static int  nextStatus = 0;
  static int  oldBank0;
  static int  pollCount;
  U32         flags;

  if (recurseFlag)
    VPRINTF (("ISR reentered!!!!!!!!!!!!!!!!!!!!!!\n"));
  recurseFlag++;

  iir = PortRead(IIR);
  lsr = PortRead(LSR);
  ier = PortRead(IER);
  PortWrite(IER, 0x00);

  if (dev->irState == IrStateWrite)
  {
    EnterCritical (flags);
    if (iir & IIR_DMAEV)
    {
      // Wait for DMA to complete.
      while (!(lsr & LSR_TXFEMP))
        lsr = PortRead (LSR);
      
      dev->wrFinish = RemoveBuf(&dev->wrQueue);

      /* DMA has completed.  Check for errors. */
      if (dev->wrFinish && dev->writeCallBack)
        dev->writeCallBack (dev, STS_OK);

      ExitCritical (flags);

      // Free the buffer just received
      AppendBuf(&dev->freeQueue, dev->wrFinish);

      // And forget we saw it.
      dev->wrFinish = NULL;
      
      // Start next write.  If none, restart read mechanism.
      if (! HSWriteBuffer(dev))
      {
        (void) HSRestartRead(dev);
        PortRead (IIR);
      }
    }
    else if (iir & IIR_TXUREV)
    {
      // Transmitter underrun. Reset FIFO.
      VPRINTF (("ISR:  IR underrun\n"));
      int3;
      PortWrite (LCR, 0x80);
      PortWrite (FCR, dev->FCRdata);
      PortWrite (LCR, 0);
      BankSelect (0);
      PortRead (RESUME);
//      int3;
    }
  }
  else    // Device in read mode.
  {
    if ((iir & IIR_SFEV) || ((iir & IIR_TOEV) && !(lsr & LSR_SFEMP)))
    {
      int i;
      /* Status Fifo interrupt.  process the current frame */
      status = 0;

      EnterCritical (flags);

      /* Disable DMA until we read the status FIFO */
      BankSelect (2);
      PortWrite (DCSR, 0);

      /* Read the status FIFO information */
      BankSelect (0);
      fifo[status].len = ReadRxLength ();
      fst = PortRead (SFLSR);
      fifo[status++].fst = fst;

      /* While we have a frame, read out the status fifo */
      while (!(fst & SFLSR_SFEMP) && (status < 9))
      {
        fifo[status].len = ReadRxLength();
        fst = PortRead(SFLSR);
        fifo[status++].fst = fst;
      }

      /* Enable DMA */
      BankSelect (2);
      PortWrite (DCSR, RxDMAChannel[dev->dmaChannel] | DCSR_SFRESET | DCSR_DMAEN);

      ExitCritical (flags);

      /* Process each frame we know about */
      if (dev->readCallBack != 0)
      {
        for (i = 0; i < status; ++i)
        {
          if (fifo[i].fst & SFLSR_ORERR)
          { // error
#if 0

            // Debug print of data
            U32 count;
            U8  *tempBuffer;
            U32 buflength;

            tempBuffer = dev->currentBufferPtr;
            buflength = fifo[i].len;
            DebugPrint (("ISR: dma mode: #bytes read = %d\n", buflength));
            for (count=0; count < buflength; count ++)
            {
              DebugPrint (("%x ", *tempBuffer));
              tempBuffer++;
            }
            DebugPrint (("\n"));
#endif
            // CRC, overrun, aborted frame or bad chip error.
            DebugPrint (("ISR: data overrun error, sflsr = %x\n", fifo[i].fst));
            dev->currentBufferPtr += fifo[i].len;
            PortWrite (LCR, 0x80);
            PortWrite (FCR, HSControlData.FCRdata);
            PortWrite (LCR, 0);
            BankSelect (0);
            PortRead (RESUME);
            (void) HSRestartRead(dev);
       	    int3;
          } // status error
          else if (fifo[i].fst & (SFLSR_CRCERR | SFLSR_PHYERR | SFLSR_FLERR))
          { // error
#if 0

            // Debug print of data
            U32 count;
            U8  *tempBuffer;
            U32 buflength;

            tempBuffer = dev->currentBufferPtr;
            buflength = fifo[i].len;
            DebugPrint (("ISR: dma mode: #bytes read = %d\n", buflength));
            for (count=0; count < buflength; count ++)
            {
              DebugPrint (("%x ", *tempBuffer));
              tempBuffer++;
            }
            DebugPrint (("\n"));
#endif
            // CRC, physical or frame too long error.
            DebugPrint (("ISR: CRC. PHY OR FL error, sflsr = %x\n", fifo[i].fst));
            PortWrite (LCR, 0x80);
            PortWrite (FCR, HSControlData.FCRdata);
            PortWrite (LCR, 0);
            BankSelect (0);
            PortRead (RESUME);
            int3;
            dev->currentBufferPtr += fifo[i].len;
          } // status error
          else
          {
            /* Deliver the bytes */
            userlen = fifo[i].len - ((dev->speedMode == FIR_4000000) ? 4 : 2);
            dev->readCallBack(dev, dev->currentBufferPtr, userlen, STS_OK);
            dev->currentBufferPtr += fifo[i].len;
          } // valid buffer
         
        }   // for loop scanning the status fifo
      }
      else
      { 
        // No call back address - leave status in local fifo.
        VPRINTF (("ISR: no call back address\n"));
      }

    } // end of status fifo poll.
  }

  PortWrite(IER, ier);	/* Restore interrupt selects */

  recurseFlag = 0;
  return;
}

/*
 *	TIIntProcPIO
 *
 *	Interrupt service routine for processing TIR2000 interrupts in PIO mode
 *
 *	Entry:  IR device control structure
 *
 *	Returns nothing.
 */

void TIIntProcPIO (void)
{
  static int  iir;
  static int  lsr;
  static int  userlen;
  U32         flags;
  U32         ier;

  if (recurseFlag)
    DebugPrint (("ISR reentered\n"));
  recurseFlag++;

  iir = PortRead(IIR);
  ier = PortRead (IER);
  EnterCritical (flags);
  
  if ((iir & IIR_TXEV) || (iir & IIR_TXUREV))
  {   // write state
    if (iir & IIR_TXUREV)
    {
      // transmitter underrun error, reset FIFO
      DebugPrint (("ISR:  TX underrun\n"));
      if (iir & 0x2)
        int3;
      HSControlData.wrFinish = RemoveBuf(&HSControlData.wrQueue);

      PortWrite (LCR, 0x80);
      PortWrite (FCR, HSControlData.FCRdata);
      PortWrite (LCR, 0);
      BankSelect (0);
      PortRead (RESUME);

      // Free the buffer just received
      AppendBuf(&HSControlData.freeQueue, HSControlData.wrFinish);

      // And forget we saw it.
      HSControlData.wrFinish = NULL;
      
      // Start next write.  If none, restart read mechanism.
      if (! HSWriteBuffer(&HSControlData))
        (void) HSRestartRead(&HSControlData);
      
      ExitCritical(flags);
      vpicdPhysEOI();
      recurseFlag = 0;
      return;
    }
    if ((iir & IIR_TXEV) && (HSControlData.currentBufferLen))
    {
      // transmit threshold interrupt and there's still more data to send.

      _asm
      {
        push  esi
        mov   esi, bufferAddress
        xor   ecx, ecx
        xor   ebx, ebx
        mov   cx, txThresh

        cmp   cx, bufferCount     ; are there more than threshold bytes to send?
        jb    storeCount          ; no, the number of bytes to write to the FIFO is threshold bytes
        mov   cx, bufferCount     ; set the number of bytes to write to the FIFO the number of 
                                  ; bytes left in the packet to transmit
storeCount:
        mov   bx, cx
startLoop:                        ; transmit ecx bytes of data
        mov   al, byte ptr [esi]
        mov   dx, word ptr [LocalPort]
        out   dx, al
        inc   esi
        loop  startLoop

        mov   bufferAddress, esi
        sub   bufferCount, bx
        mov   dx, word ptr [LocalPort]
        add   dx, IER
        xor   al, al
        out   dx, al
        mov   al, IER_TXIEN | IER_TXURIEN
        out   dx, al
        pop   esi
      }

      HSControlData.currentBufferPtr = bufferAddress;
      HSControlData.currentBufferLen = bufferCount;
    }
    else if (iir & IIR_TXEV) 
    {
      PortWrite (IER, 0);

      while (!(PortRead (LSR) & LSR_TXFEMP))
        ;   // do nothing but wait for the FIFO to empty.

      // Transmitter threshold interrupt and no more data to transmit, inform framer.
      HSControlData.wrFinish = RemoveBuf(&HSControlData.wrQueue);
      if (HSControlData.wrFinish && HSControlData.writeCallBack)
        HSControlData.writeCallBack (&HSControlData, STS_OK);

      // Free the buffer just received
      AppendBuf(&HSControlData.freeQueue, HSControlData.wrFinish);
      
      // And forget we saw it.
      HSControlData.wrFinish = NULL;
     
      // Send out next packet if there is a packet to be written, otherwise set read mode.
      if (!HSWriteBuffer(&HSControlData))
      {
        U8 iir1;
        (void) HSRestartRead(&HSControlData);
        iir1 = PortRead (IIR);    // read the IIR port to remove any false interrupts
      }
    }
  }
  else if (((iir & IIR_RXEV) || (iir &  IIR_EOPEV)) || (iir & IIR_RXOREV)) // read state
  {
    lsr = PortRead (LSR);

    // Turn off receive threshold and end of packet interrupts. This is to prevent the 
    // end-of-packet interrupt being missed when 2 short packets are sent back-to-back.
    PortWrite (IER, ier & ~(IER_RXIEN | IER_EOPIEN | IER_RXORIEN));

    // service interrupts as long as there are interrupts to service.
    if (((iir & IIR_RXEV) && !(iir & (IIR_EOPEV | IIR_RXOREV))) && !(lsr & LSR_RXRDY))
    {
      // receive threshold interrupt - read dev->RxThreshold bytes of data.  threshold was
      // set up in HSOpen.
      _asm
      {
        push  edi
        mov   dx, word ptr [LocalPort]
        mov   edi, bufferPtr
        mov   ebx, bufLength
        mov   ecx, threshold
        add   ebx, ecx
        rep   insb
        mov   bufLength, ebx
        mov   bufferPtr, edi
        pop   edi
      }
      ExitCritical(flags);
      vpicdPhysEOI();
      recurseFlag = 0;
      PortWrite (IER, ier);
      return;
    }  // receive threshold interrupt

    else if (((iir & IIR_EOPEV) && !(iir & IIR_RXOREV)) && !(lsr & LSR_RXRDY))
    {
      // end-of-packet interrupt - read data until the last byte is read from the FIFO as
      // indicated by IIR[2]

      PortWrite (IER, IER_RXEIEN | IER_RXORIEN);
      _asm
      {
        push  edi
        mov   dx, word ptr [LocalPort]
        mov   edi, bufferPtr
        mov   ecx, 65             ; if the processor is too slow, the interrupts
                                  ; don't necessarily get read properly.  Need a 
                                  ; check to stop reading data if the interrupt 
                                  ; doesn't get caught.
    
readLoop1:
        add   dx, IIR
        in    al, dx
        mov   ah, al
        and   al, 0Ch             ; overrun error or FIFO empty?
        jnz   endLoop1            ; yes, quit
        dec   ecx
        jcxz  endLoop1

        mov   dx, word ptr [LocalPort]    ; read the next character
        in    al, dx
        inc   bufLength
        mov   byte ptr [edi], al
        inc   edi
        jmp   readLoop1
endLoop1:
        mov   byte ptr [iir], ah          ; store the contents of IIR for error checking
        pop   edi
      }

      if (iir & IIR_RXOREV)
      {
        // receive overrun error, reset FIFO, restart read and reenable interrupts
        VPRINTF (("ISR:  RX overrun error\n"));
        DebugPrint (("ISR:  RX overrun error: iir: %x, buflength: %d\n", 
                      iir, bufLength));
        PortWrite (LCR, 0x80);
        PortWrite (FCR, HSControlData.FCRdata);
        PortWrite (LCR, 0);
        BankSelect (0);
        PortRead (RESUME);
        (void) HSRestartRead(&HSControlData);
        PortWrite (IER, IER_RXIEN |/* IER_RXEIEN |*/ IER_RXORIEN | IER_EOPIEN);
 
        ExitCritical (flags);
        vpicdPhysEOI();
        recurseFlag = 0;
        return;
      }  // overrun error

      lsr = PortRead (LSR);
      if ((HSControlData.readCallBack != 0) && (iir & IIR_FFEV))
      {
        if (lsr & LSR_BAD_CRC)
        {
          // bad CRC error

#if 0     // Debug information
          U32 count;
          U8  *tempBuffer;

          // Debug data display
          tempBuffer = HSControlData.currentBufferPtr;
          DebugPrint (("ISR: pio mode: #bytes read = %d\n", bufLength));
          for (count=0; count < bufLength; count ++)
          {
            DebugPrint (("%x ", *tempBuffer));
            tempBuffer++;
          }

          VPRINTF (("ISR: read - bad CRC\n"));
          DebugPrint (("ISR: read - bad CRC, LSR = %x\n", lsr));
//          int3;
#endif    // Debug information

          // Lost all hope - restart the read
          (void) HSRestartRead(&HSControlData);
        }
        else if (lsr & LSR_PHY_ERR)
        {
          // physical error

#if 0     // Debug information
          U32 count;
          U8  *tempBuffer;

          // debug data display
          tempBuffer = HSControlData.currentBufferPtr;
          DebugPrint (("ISR: pio mode: LSR = %x, #bytes read = %d\n", lsr, bufLength));
          for (count=0; count < bufLength; count ++)
          {
            DebugPrint (("%x ", *tempBuffer));
            tempBuffer++;
          }

          VPRINTF (("ISR: read - physical error\n"));
          DebugPrint (("ISR: read - physical error: LSR = %x\n", lsr));
//          int3;
#endif    // Debug information

          // Lost all hope - restart the read 
          (void) HSRestartRead(&HSControlData);
        }
        else if (lsr & LSR_MAX_LEN)
        {
          // frame longer than the maximum size to be received error

          VPRINTF (("ISR: read - frame too long\n"));
          DebugPrint (("ISR: read - frame too long, lsr = %x\n", lsr));
          int3;

          // Lost all hope - restart the read dma
          (void) HSRestartRead(&HSControlData);
        }
        else
        {
          // good data, send up to framer.
          userlen = bufLength - ((HSControlData.speedMode == FIR_4000000) ? 4 : 2);
          HSControlData.readCallBack(&HSControlData, HSControlData.currentBufferPtr, userlen, STS_OK);
          HSControlData.currentBufferPtr += bufLength;
          bufLength = 0;
          bufferPtr = HSControlData.currentBufferPtr;
          PortWrite (IER, IER_RXIEN |/* IER_RXEIEN |*/ IER_RXORIEN | IER_EOPIEN);
        }
      } // if (readCallBack)
    }  // end of packet interrupt

    else if (iir & IIR_RXOREV)
    {
      // receive overrun error, reset FIFO, restart the read
      VPRINTF (("ISR:  1.152M RX overrun error\n"));
      DebugPrint (("ISR:  RX overrun error: iir: %x, buflength: %d\n", 
                    iir, bufLength));
      PortWrite (LCR, 0x80);
      PortWrite (FCR, HSControlData.FCRdata);
      PortWrite (LCR, 0);
      BankSelect (0);
      PortRead (RESUME);
      (void) HSRestartRead(&HSControlData);
      PortWrite (IER, IER_RXIEN |/* IER_RXEIEN |*/ IER_RXORIEN | IER_EOPIEN);
      ExitCritical (flags);
      vpicdPhysEOI();
      recurseFlag = 0;
      return;
    }
  }   // read state
  
  ExitCritical(flags);
  vpicdPhysEOI();
  recurseFlag = 0;
} // TIIntProcPIO



// *
// *	HSWriteDivisor
// *
// *	Write to the TIR2000 <COM port> divisor latches using the auxiliary baud-rate
// *  divisor registers in bank 3.  This avoids the "fall back" to UART
// *	mode that will occur when writing to the normal baud-rate divisor
// *  registers.
// *
// *	Entry:	val		16 bit value to write to the DLH and DLL
// *
// *	Returns nothing.
// *
// *	This function is called in a few places in throughout the driver
// *	where changing the COM port (low speed) baud rate is needed.
// *	These locations were changed from direct writes to the DLH and
// *	DLL register offsets.
// */
void
HSWriteDivisor(U16 val)
  {
  U32       flags;

  EnterCritical (flags);
  BankSelect(3);
  PortWrite (ABDL, val & 0xFF);
  PortWrite (ABDH, (val >> 8) & 0xFF);
  BankSelect (2);
  ExitCritical (flags);
  }


//*****************************************************************
//*  HSInterrupt
//*  Interrupt service routine
//* 
//*  No inputs or outputs
//******************************************************************

void HSInterrupt (void)
{

  TIIntProcDMA (&HSControlData);

  vpicdPhysEOI();

}

