//***************************************************************************
//
// File:        hsdriver.h
//
// Description: This file contains the definitions for the National Semiconductor
// PC87xxx IR driver.
//
// Author:      Anne Kelley
//
// Copyright 1996, 1997 Alpha Omega Computer Systems, Inc.
// All rights reserved worldwide
//
//***************************************************************************

#ifndef HSDRIVER_H
#define HSDRIVER_H

#include "port.h"
#include "common.h"
#ifdef WIN31COMPAT
#include "intern31.h"
#else
#include "internal.h"
#endif
#include "buffman.h"


#define	arraySize(array)	(sizeof(array) / sizeof(array[0]))
#define	MAXDEVICEBUFLEN		16383


/**/
/*  TIR2000 configuration and port definitions  */

#define RxBufferReg   0         /* Receiver Buffer Register - Read only */
#define TxHoldingReg  0         /* Transmitter holding register - Write only */

#define IER           1         /* Interrupt enable register - read and write */
#define  IER_RXIEN    0x01      /* Receive data available interrupt enable */
#define  IER_TXIEN    0x02      /* Transmitter holding register empty interrupt enable */
#define  IER_LSIEN    0x04      /* Receiver line status interrupt enable - SIR/UART */
#define  IER_RXEIEN   0x04      /* Last byte from RX FIFO enable - MIR/FIR */
#define  IER_MSIEN    0x08      /* Modem status interrupt enable - SIR/UART */
#define  IER_RXORIEN  0x08      /* RX FIFO overrun interrupt enable - MIR/FIR */
#define  IER_TXURIEN  0x20      /* transmitter underrun interrupt enable */
#define  IER_EOPIEN   0x80      /* received end-of-packet interrupt enable */

#define  IER_TOIEN    0x04      /* status FIFO time-out interrupt enable */
#define  IER_SFIEN    0x10      /* status FIFO threshold interrupt enable */
#define  IER_DMAIEN   0x40      /* DMA terminal count interrupt enable */

#define IIR           2         /* Interrupt identification register - read only */
#define	 IIR_RXEV		  0x01	    /* RX event			*/
#define	 IIR_TXEV		  0x02	    /* TX event			*/
#define  IIR_FFEV     0x04      /* Last byte read from FIFO */
#define  IIR_RXOREV   0x08      /* receive FIFO overrun event */
#define  IIR_TXUREV   0x20      /* transmit underrun event */
#define  IIR_EOPEV    0x80      /* received end-of-packet event */

#define  IIR_TOEV     0x04      /* status FIFO time-out event */
#define  IIR_SFEV     0x10      /* status FIFO threshold event */
#define  IIR_DMAEV    0x40      /* DMA event */

#define FCR           2         /* FIFO control register - write only */
#define  FCR_FIFOEN   0x01      /* FIFO enable */
#define	 FCR_RXFR	    0x02      /* Clear Receive Fifo		*/
#define	 FCR_TXFR	    0x04      /* Clear transmit fifo		*/
#define  FCR_DMAMD    0x08      /* DMA mode select */
#define  FCR_FS_16    0x00      /* FIFO size 16 */
#define	 FCR_FS_64    0x20      /* FIFO size 64 */
#define	 FCR_RXFT     0xC0      /* Receive fifo threshold	*/
#define	  FCR_RX16_1  0x00      /* 16 byte mode: 1 byte		*/
#define	  FCR_RX16_4  0x40      /* 16 byte mode: 4 bytes	*/
#define	  FCR_RX16_8  0x80      /* 16 byte mode: 8 bytes	*/
#define	  FCR_RX16_14 0xC0      /* 16 byte mode: 14 bytes	*/
#define	  FCR_RX64_1  0x00      /* 64 byte mode: 1 byte		*/
#define	  FCR_RX64_16 0x40      /* 64 byte mode: 16 bytes	*/
#define	  FCR_RX64_32 0x80      /* 64 byte mode: 32 bytes	*/
#define	  FCR_RX64_56 0xC0      /* 64 byte mode: 56 bytes	*/

#define LCR           3         /* Line control register - read and write */
#define  LCR_WLS      0x03      /* word length select */
#define   LCR_WLS_5   0x00      /* word length = 5 bits */
#define   LCR_WLS_6   0x01      /* word length = 6 bits */
#define   LCR_WLS_7   0x02      /* word length = 7 bits */
#define   LCR_WLS_8   0x03      /* word length = 8 bits */
#define	 LCR_STB      0x04      /* stop bits:			*/
#define	  LCR_STB_1   0x00      /* 1 stop bits			*/
#define	  LCR_STB_2   LCR_STB   /* 2 stop bits			*/
#define	 LCR_PEN      0x08      /* Parity enable		*/
#define	 LCR_EPS      0x10      /* Even parity select		*/
#define	 LCR_STKPS    0x20      /* Stick parity select		*/
#define	 LCR_SBRK     0x40      /* Set break			*/
#define	 LCR_DLAB     0x80      /* divisor latch access bit		*/

#define	MCR           4         /* Modem control register	- read and write */
#define	 MCR_DTR      0x01      /* Data Terminal Ready		*/
#define	 MCR_RTS      0x02      /* Request to send		*/
#define  MCR_LOOP     0x10      /* Serial loopback		*/
#define  MCR_FCE      0x20      /* Flow-control enable */

#define	LSR           5         /* Line status register - read only */
#define	 LSR_RXRDY    0x01	    /* Receiver ready */
#define  LSR_SFEMP    0x02      /* status FIFO empty (DMA mode) */
#define	 LSR_OE       0x02	    /* Overrun */
#define	 LSR_PE       0x04	    /* Parity error */
#define	 LSR_BAD_CRC  LSR_PE    /* Bad CRC */
#define	 LSR_FE       0x08      /* Framing error */
#define	 LSR_PHY_ERR  LSR_FE    /* Physical medium error */
#define	 LSR_BRK      0x10      /* Break detected */
#define	 LSR_MAX_LEN  LSR_BRK   /* Max packet len reached on RX */
#define	 LSR_TXRDY    0x20      /* Transmitter ready */
#define	 LSR_TXEMP    0x40      /* Transmitter empty */
#define	 LSR_ER_INF   0x80      /* Error info in status FIFO */
#define	 LSR_TXFEMP   0x80      /* transmitter FIFO empty */

#define	MSR           6         /* Modem status register - read only */
#define  MSR_DCTS     0x01      /* Delta CTS */
#define  MSR_DDSR     0x02      /* Delta DSR */
#define  MSR_TERI     0x04      /* Trailing Edge Ring Indicator */
#define  MSR_DDCD     0x08      /* Delta DCD */
#define	 MSR_CTS      0x10      /* Clear To Send */
#define  MSR_DSR      0x20      /* Data Set Ready */
#define  MSR_RI       0x40      /* Ring Indicator */
#define  MSR_DCD      0x80      /* Data Carrier Detect */

#define SCR           7         /* Scratch register - read and write */

#define DLL           0         /* baud-rate divisor latch LSB, LCR(7)=1 - write and read */
#define DLM           1         /* baud-rate divisor latch MSB, LCR(7)=1 - write only */

#define MDR           8         /* mode definition register - read and write */
#define  MDR_MDSEL    0x07      /* mode select */
#define   MDR_MD_UART 0x00      /* UART mode */
#define   MDR_MD_SIR  0x01      /* SIR mode */
#define   MDR_MD_MIR  0x03      /* MIR mode */
#define   MDR_MD_FIR  0x04      /* FIR mode */
#define  MDR_SLEEP    0x08      /* Sleep mode enabled */
#define  MDR_LPM      0x10      /* low power mode enabled */
#define  MDR_TXDEF    0x20      /* deferred transmission enabled */
#define  MDR_IRPLS    0x40      /* 2us IR pulse select method */
#define  MDR_EOTSEL   0x80      /* frame end mode select

/* Bank registers */
#define CCR           9         /* Configuration control register - write only */
#define  CCR_BANKSEL  0xC0      /* bank select */
#define  CCR_BANK_0   0x00      /* bank 0 */
#define  CCR_BANK_1   0x40      /* bank 1 */
#define  CCR_BANK_2   0x80      /* bank 2 */
#define  CCR_BANK_3   0xC0      /* bank 3 */

/* Bank 0 registers */
#define TXFLL         0x0A      /* transmit frame length register low - write only */

#define SFLSR         0x0A      /* status FIFO line status register read only */
#define  SFLSR_CRCERR 0x01      /* CRC error */
#define  SFLSR_PHYERR 0x02      /* physical error */
#define  SFLSR_FLERR  0x04      /* frame length error */
#define  SFLSR_ORERR  0x08      /* overrun errors */
#define  SFLSR_SFEMP  0x10      /* status FIFO empty */

#define TXFLH         0x0B      /* transmit frame length register high - write only */

#define RESUME        0x0B      /* Resume register - read to clear Tx underrun */

#define RXFLL         0x0C      /* receive frame length register low - write only */

#define SFREGL        0x0C      /* status FIFO register low - read only */

#define RXFLH         0x0D      /* receive frame length register high - write only */

#define SFREGH        0x0D      /* status FIFO register high - read only */
 
#define PLR           0x0E      /* Preamble length register - write only */

#define ACREG         0x0F      /* Auxilliary control Register - write and read */
#define  ACREG_EOT    0x01      /* End 0f transmission bit */
#define  ACREG_FA     0x02      /* frame abort */
#define  ACREG_TXDEF  0x04      /* deferred transmit start */
#define  ACREG_IRPLS  0x08      /* send IR pulse */
#define  ACREG_580K   0x10      /* .58 Mbps in MIR mode */
#define  ACREG_STO    0x20      /* when 1, status FIFO timeout = 128us */
#define  ACREG_RXEN   0x40      /* when 0, receiver disabled */
#define  ACREG_TXEN   0x80      /* when 0, transmitter disabled */

/* Bank 1 registers */

#define TVCFG             0x0A      /* TV configuration register */
#define  TVCFG_MOD        0x03      /* modulation mode */
#define   TVCFG_MOD_C_PLS 0x00      /* C_PLS modulation mode */
#define   TVCFG_MOD_8_PLS 0x01      /* 8_PLS modulation mode */
#define   TVCFG_MOD_6_PLS 0x02      /* 6_PLS modulation mode */
#define  TVCFG_FREQ_30K   0x00      /* TV carrier freq. range 30k-56k */
#define  TVCFG_FREQ_400K  0x04      /* TV carrier freq. range 400k-500k ...*/

/* Bank 2 registers */
#define PRESC             0x0A      /* prescaler register - read and write */

#define ICR               0x0B      /* Interrupt configuration register - write only */
#define  ICR_IRQSEL       0x0F      /* irq channel select */
#define   ICR_IRQ_3       0x01      /* irq channel 3 */
#define   ICR_IRQ_4       0x02      /* irq channel 4 */
#define   ICR_IRQ_5       0x03      /* irq channel 5 */
#define   ICR_IRQ_6       0x04      /* irq channel 6 */
#define   ICR_IRQ_7       0x05      /* irq channel 7 */
#define   ICR_IRQ_9       0x06      /* irq channel 9 */
#define   ICR_IRQ_10      0x07      /* irq channel 10 */
#define   ICR_IRQ_11      0x08      /* irq channel 11 */
#define   ICR_IRQ_12      0x09      /* irq channel 12 */
#define   ICR_IRQ_14      0x0A      /* irq channel 14 */
#define   ICR_IRQ_15      0x0B      /* irq channel 15 */
#define  ICR_SFT          0x30      /* status FIFO threshold select */
#define   ICR_SFT_1       0x00      /* status FIFO threshold = 1 */
#define   ICR_SFT_4       0x10      /* status FIFO threshold = 4 */
#define   ICR_SFT_7       0x20      /* status FIFO threshold = 7 */
#define   ICR_SFT_8       0x30      /* status FIFO threshold = 8 */
#define  ICR_MODESEL      0x40      /* irq mode select */
#define   ICR_OPEN_DRAIN  0x00      /* irq output is open drain */
#define   ICR_TOTEM_POLE  0x40      /* irq output is totem pole */
#define  ICR_ALS          0x80      /* irq active level select */
#define   ICR_ALS_HIGH    0x00      /* irq active high */
#define   ICR_ALS_LOW     0x80      /* irq active low */

#define DCSR              0x0C      /* DMA channel select register - write only */
#define  DCSR_TXSEL       0x07      /* DMA transmit channel select */
#define   DCSR_TX_DIS     0x00      /* DMA transmit channel disabled */
#define   DCSR_TX_0       0x01      /* DMA transmit channel 0 */
#define   DCSR_TX_1       0x02      /* DMA transmit channel 0 */
#define   DCSR_TX_3       0x04      /* DMA transmit channel 0 */
#define  DCSR_RXSEL       0x38      /* DMA receive channel select */
#define   DCSR_RX_DIS     0x00      /* DMA receive channel disabled */
#define   DCSR_RX_0       0x08      /* DMA receive channel 0 */
#define   DCSR_RX_1       0x10      /* DMA receive channel 0 */
#define   DCSR_RX_3       0x20      /* DMA receive channel 0 */
#define  DCSR_DMAEN       0x40      /* DMA enable */
#define  DCSR_SFRESET     0x80      /* Status FIFO reset */

/* Bank 3 registers */
#define IRCFG             0x0A      /* IR mode configuration register - read and write */
#define  IRCFG_TEMIC      0x01      /* temic bit */
#define  IRCFG_IRRVSEL    0x02      /* IRRVL/IRRVH select in TV and SHARP mode */
#define  IRCFG_SIRPLS     0x08      /* SIR Pulse width - 0: 3/16 of bit rate, 1: 1.6us */

#define GPIODIR           0x0B      /* General purpose I/O direction register - write only */

#define GPIODAT           0x0C      /* general purpose I/O data register - write and read */

#define GPFSR             0x0D      /* general purpose function select register write only */

#define ABDL              0x0E      /* Auxilliary baud rate divisor lower register - write*/

#define ABDH              0x0F      /* Auxilliary baud rate divisor higher register - write */


#define IntEoi()	vpicdPhysEOI();

#define TIMER_RATE	5		/* Rate for polling */

#define	MAX_TX_RETRIES	5

#define	MAX_TX_TIMER	100		/* Max time to TX a packet */
#define	SIP_PULSE_TIMER	500/TIMER_RATE	/* Rate to generate SIPs */

#define	BAD_MODE	-1
#define	SIR_2400	0
#define	SIR_9600	1
#define	SIR_19200	2
#define	SIR_38400	3
#define	SIR_57600	4
#define	SIR_115200	5
#define SIR_576000	6
#define	MIR_1152000	7
#define	FIR_4000000	8

#define MaxPacketLength       (2048 + 4 + 2 + 1)
                // Allow 4 for CRC and 2 for Irlap header and 1 for EOF byte (?)

typedef enum 
{
  IrStateRead,
  IrStateWrite
} IRState;

typedef enum
{
  STS_OK = 0,
  STS_FAIL,
  STS_FRAME_ERROR,
  STS_LOST_FRAME
} hsResults;

typedef enum 
{
  STS_HSBaud,
  STS_HsdriverEnable,
  STS_HsHighSpeedRx,
  STS_HsLowSpeedRx,
  STS_HsGetSpeedMode,
  STS_HsGetTimerVal,
  STS_HsGetHandle
} STATUS_SEL;

typedef enum 
{
  CTL_HSBaud,
  CTL_HsdriverEnable,
  CTL_HsHighSpeedRx,
  CTL_HsLowSpeedRx,
  CTL_resetRegisters
} CONTROL_SEL;

typedef hsResults (*readCallBack)(struct HSControlStruct *dev, U8 *buffer,
                                  U32 length, hsResults status);
typedef hsResults (*writeCallBack)(struct HSControlStruct *dev, hsResults status);

typedef struct HSControlStruct
{
  U32               devType;          // Device type.
  U32               temicDongle;      // 0 for HP dongle, non-zero for Temic dongle
  U8                dmaMode;          // 0 = DMA mode (default), 1 = programmed I/O mode 
  U8                lowPowerEnable;   // 0 = low power mode disabled, 1 = low power mode enabled
  U8                sleepModeEnable;  // 0 = sleep mode disabled, 1 = sleep mode enabled
  U32               dmaChannel;       // Device DMA receive channel
  U32               vdmadRxHandle;    // DMA receive channel handle
  U32               comPort;          // Device serial port
	U32						    irq;              // IRQ number
  U32               scePort;          // The control port address.
  U32               deviceId;         // Device ID
  U8                FCRdata;          // holds the last setting written to the FIFO control reg.
  U8                sir576en;         // 0 if not 576kbps mode, 0x10 if 576kbps mode
  U8                PLRdata;          // holds the setting to write to the PLR register.
  U32               FIFOsize;         // holds the size of the FIFO
  U32               TxThreshold;      // holds the transmitter threshold level
  U32               RxThreshold;      // holds the receiver threshold level
  U32               characterCount;   // holds the number of characters that have been transmitted.


  Buffer        rdPreBufs[NumPool];   // Used to save data in read queue
  BufQueue      rdPreFree;            // Queue of free preread buffers
  BufQueue      rdPreQueue;           // Queue to be processed
  BufQueue          wrQueue;          // Write buffer queue
  Buffer *          wrFinish;
  BufQueue          freeQueue;        // free buffer queue
  Buffer            buffers[NumPool]; // write buffers
  U8 *              inputBuffer;      // read buffer in dma space.
  U8 *              bufPool;          // buffer pool for read/write
  U8 *              currentBufferPtr; // Current position within buffer
  U16               currentBufferLen; // Length of the current buffer to transmit.

#ifdef WIN31COMPAT
  U32               inputBufHandle;   // memory handle for the input buffer
  U32               bufPoolHandle;    // memory handle for the buffer pool
#endif

  IRState           irState;          // Flag indicating read/write mode
  int               speedMode;        // holds the speed mode.

  readCallBack      readCallBack;     // Read callback procedure address
  writeCallBack     writeCallBack;    // Write callback procedure address
  PPortInformation  portInfo;         // holds the port information structure

  
  int           writeTimeout;         // Limits time in WRITE state
#define WRITE_TIMEOUT  (100/TIMER_RATE) // 100 milliseconds for timeout
} HSControl;

	// timing kludge.
U32		RdTimer1;
U32		RdTimer2;
U32		WrTimer1;
U32		WrTimer2;
U32   NumRetran;


// Procedure prototypes

#ifdef WIN31COMPAT
void          _Page_Allocate (U32 numbPages, PVOID CopyPageBuf, 
                             U32 * handle, U8 ** linAddr);
#endif

void parseDevType (char * devInfo, U32 *temicDongle, U8 *dmaMode, U32 *FIFOsize, 
                   U32 *RxThreshold, U32 *TxThreshold, U8 *lowPowerEnable, U8 *sleepEnable, 
                   U8 *FCRdata, U8 *PLRdata);
int	ioDelay(void);
unsigned long GetRealTime();
HSControl     *HSOpen (int comPort, int irq, int dma1, int dma2, 
                       char * devInfo, PPortInformation hPort);
hsResults     HSClose (HSControl *dev);
hsResults     HSWrite (HSControl *dev, U8 *buffer, U32 length,
                       BOOL lastFrame, writeCallBack callBack);
hsResults     HSRead (HSControl *dev, readCallBack callBack);
hsResults     HSSetControl (HSControl *dev, enum CONTROL_SEL cntlSelector, ...);
hsResults     HSGetStatus (HSControl *dev, enum STATUS_SEL statSelector, 
                       void *status);

void          HSInterrupt(void);
void          TIIntProcPIO(void);
void          HSWriteDivisor(U16 val);
void          PrintMessage (void);
void          PrintInt (U32 number);
U32           GetTimerValue (void);
unsigned long GetElapsedTime(void);

#endif